# Add Purchase Function - Code Review

## Overview
This review covers both the frontend (`add-purchase.html`) and backend (`purchase.routes.js`) implementation of the add purchase functionality.

---

## 🔴 CRITICAL ISSUES

### 1. Missing HTTP Method in Fetch Call
**Location:** `frontend/pages/add-purchase.html:852`
**Issue:** The fetch call is missing the `method: 'POST'` parameter.
```javascript
const response = await fetch('/api/purchases', {
    // method: 'POST',  // ❌ MISSING!
    headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${getAuthToken()}`
    },
    body: JSON.stringify(formData)
});
```
**Impact:** May default to GET request, causing the purchase to fail.
**Fix:** Add `method: 'POST'` to the fetch options.

---

### 2. Supplier ID Not Sent to Backend
**Location:** `frontend/pages/add-purchase.html:828-843`
**Issue:** The form collects `supplier_id` (line 170, 431) but it's not included in the `formData` object sent to the backend.
```javascript
const formData = {
    purchase_voucher_id: null,
    cat_id: document.getElementById('category_id').value,
    // supplier_id: document.getElementById('supplier_id').value,  // ❌ MISSING!
    barcode: document.getElementById('barcode').value,
    // ...
};
```
**Impact:** Supplier information is lost, making it difficult to track purchases by supplier.
**Fix:** Add `supplier_id` to formData.

---

### 3. No Transaction Management in Backend
**Location:** `backend/routes/purchase.routes.js:76-284`
**Issue:** The purchase creation and stock update operations are not wrapped in a database transaction. If stock update fails, the purchase is still created, leading to data inconsistency.
```javascript
// Purchase is inserted first
const result = await query(sql, [...]);

// Stock update happens separately - if this fails, purchase still exists
if (name && (qty || 0) > 0) {
    try {
        // Stock update logic...
    } catch (stockError) {
        // Error is caught but purchase already committed
    }
}
```
**Impact:** Data inconsistency if stock update fails after purchase is created.
**Fix:** Wrap both operations in a database transaction.

---

### 4. Missing Validation for Negative Values
**Location:** Both frontend and backend
**Issue:** No validation to prevent negative quantities or prices.
- Frontend: `quantity` and `cost_price` inputs accept negative values
- Backend: No validation before database insert
**Impact:** Invalid data can be stored (negative quantities/prices).
**Fix:** Add validation in both frontend and backend.

---

## 🟡 MEDIUM PRIORITY ISSUES

### 5. Incomplete Error Handling
**Location:** `frontend/pages/add-purchase.html:861-874`
**Issue:** Error handling doesn't check HTTP response status before parsing JSON.
```javascript
const result = await response.json();  // ❌ May fail if response is not JSON

if (result.success) {
    // ...
} else {
    showError(result.message || 'Failed to save purchase');
}
```
**Impact:** If server returns non-JSON error (e.g., 500 HTML page), JSON parsing will throw an unhandled error.
**Fix:** Check `response.ok` and `response.status` before parsing JSON.

---

### 6. Stock Update Logic Complexity
**Location:** `backend/routes/purchase.routes.js:146-271`
**Issue:** The stock update logic is complex with multiple nested conditions and potential race conditions:
- Multiple queries to find stock by barcode/name
- No locking mechanism
- Silent failures (errors are caught but not propagated)
**Impact:** Race conditions in concurrent scenarios, difficult to debug failures.
**Fix:** Simplify logic, add proper error handling, consider using database locks.

---

### 7. Warehouse Stock Movement Error Handling
**Location:** `backend/routes/purchase.routes.js:238-264`
**Issue:** Warehouse stock movement errors are caught and logged but don't affect the response. The purchase succeeds even if warehouse assignment fails.
```javascript
try {
    await query(`INSERT INTO tbl_stock_movement...`);
} catch (warehouseError) {
    console.error('Warehouse stock movement error:', warehouseError.message);
    // ❌ Error is ignored - purchase still succeeds
}
```
**Impact:** Purchase appears successful but warehouse stock is not updated.
**Fix:** Decide whether warehouse errors should fail the purchase or be handled differently.

---

### 8. Missing Date Validation
**Location:** Both frontend and backend
**Issue:** No validation for:
- Expiry date should be after purchase date
- Alert months should be reasonable (1-24)
- Date formats
**Impact:** Invalid dates can be stored.
**Fix:** Add date validation in frontend and backend.

---

### 9. Missing Supplier Validation
**Location:** Backend
**Issue:** No validation to check if `supplier_id` exists in the database (though it's not currently being sent).
**Impact:** Invalid supplier references if supplier_id is added later.
**Fix:** Add validation when supplier_id is implemented.

---

## 🟢 MINOR ISSUES / IMPROVEMENTS

### 10. Duplicate Button in Header
**Location:** `frontend/pages/add-purchase.html:124-136`
**Issue:** "Inpatient Sales" button appears twice in the header.
**Fix:** Remove duplicate button.

---

### 11. Inconsistent Error Messages
**Location:** Frontend
**Issue:** Some errors show generic messages, others show specific ones.
**Fix:** Standardize error messages.

---

### 12. Missing Input Sanitization
**Location:** Backend
**Issue:** No sanitization of user inputs (though SQL injection is prevented by parameterized queries).
**Impact:** XSS if data is displayed elsewhere.
**Fix:** Add input sanitization/validation.

---

### 13. Hardcoded Default Values
**Location:** `backend/routes/purchase.routes.js:132`
**Issue:** `qty_alert` defaults to 10, which may not be appropriate for all products.
**Fix:** Make it configurable or require user input.

---

### 14. Missing Response Status Check
**Location:** Frontend
**Issue:** Doesn't check if response status is 201 (created) before showing success.
**Fix:** Check `response.status === 201` before showing success message.

---

## 📋 RECOMMENDATIONS

### High Priority Fixes:
1. ✅ Add `method: 'POST'` to fetch call
2. ✅ Include `supplier_id` in formData
3. ✅ Add transaction management in backend
4. ✅ Add validation for negative values

### Medium Priority Fixes:
5. ✅ Improve error handling (check response status)
6. ✅ Simplify stock update logic
7. ✅ Decide on warehouse error handling strategy
8. ✅ Add date validation

### Code Quality Improvements:
- Add JSDoc comments for functions
- Extract stock update logic to a separate service/function
- Add unit tests for critical paths
- Consider using a validation library (e.g., Joi, Yup)

---

## 🔍 TESTING RECOMMENDATIONS

1. **Test negative values:** Try entering negative quantity/price
2. **Test concurrent purchases:** Multiple users purchasing same product simultaneously
3. **Test warehouse assignment:** Verify warehouse stock is updated correctly
4. **Test error scenarios:** Network failures, database errors
5. **Test validation:** Missing required fields, invalid dates
6. **Test stock updates:** Verify stock is updated correctly for new and existing products

---

## 📝 SUMMARY

The add purchase function has several critical issues that need immediate attention:
- Missing HTTP method in fetch call
- Supplier ID not being sent
- No transaction management
- Missing validation

The code is functional but needs improvements in error handling, validation, and data consistency. The stock update logic is complex and could benefit from refactoring.
