# Clinic Pro V3 - Modern Web Application

<div align="center">

![Version](https://img.shields.io/badge/version-3.0.0-blue.svg)
![License](https://img.shields.io/badge/license-MIT-green.svg)
![Node](https://img.shields.io/badge/node-%3E%3D18.0.0-brightgreen.svg)
![MySQL](https://img.shields.io/badge/mysql-%3E%3D8.0-orange.svg)

**A comprehensive Clinic & Pharmacy Management System with POS, EMR, and multi-language support.**

[Features](#-features) • [Installation](#-installation) • [Quick Start](#-quick-start-guide) • [API Documentation](#-api-endpoints) • [Contributing](#-contributing)

</div>

---

## 📑 Table of Contents

- [Features](#-features)
- [Prerequisites](#-prerequisites)
- [Installation](#-installation)
- [Quick Start Guide](#-quick-start-guide)
- [Project Structure](#-project-structure)
- [Database Schema](#-database-schema)
- [API Endpoints](#-api-endpoints)
- [UI Features](#-ui-features)
- [Multi-language Support](#-multi-language-support)
- [Security Features](#-security-features)
- [Image Upload](#-image-upload)
- [Testing](#-testing)
- [Troubleshooting](#-troubleshooting)
- [Development Notes](#-development-notes)
- [Contributing](#-contributing)
- [License](#-license)
- [Support](#-support)

---

## 🌟 Features

### Core Features

| Feature | Description |
|---------|-------------|
| 🛒 **Point of Sale (POS)** | Modern POS with product images and real-time inventory |
| 📋 **Electronic Medical Records (EMR)** | Complete patient treatment history with multi-image upload |
| 👥 **Patient Management** | OPD and Inpatient management |
| 📦 **Inventory Management** | Stock control with image support, expiry tracking, and alerts |
| 📅 **Appointment System** | Schedule and manage patient appointments |
| 📊 **Reports & Analytics** | Comprehensive reporting system |
| 🌐 **Multi-language Support** | English and Myanmar (Burmese) |
| 🔐 **User Authentication** | Secure login with JWT tokens |
| 👤 **Role-based Access Control** | Different permissions for different user roles |

### Technical Features

- ✨ Modern Bootstrap 5 with Glassmorphism UI
- 🚀 RESTful API with Node.js/Express
- 🗄️ MySQL database with proper indexing
- 📸 Image upload with automatic resizing
- ⚡ Real-time data updates
- 📱 Responsive design for all devices

---

## 📋 Prerequisites

Before you begin, ensure you have the following installed:

- **Node.js** (v18.0.0 or higher)
- **MySQL** (v8.0 or higher)
- **npm** (v9.0.0 or higher)

### Verify Installation

```bash
node --version  # Should be >= 18.0.0
mysql --version # Should be >= 8.0.0
npm --version   # Should be >= 9.0.0
```

---

## 🚀 Installation

### Step 1: Clone the Repository

```bash
git clone https://github.com/yourusername/clinic-pro-web.git
cd clinic-pro-web
```

### Step 2: Database Setup

#### 2.1 Create MySQL Database

```sql
CREATE DATABASE clinic_pro_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
```

#### 2.2 Run Migrations

```bash
cd backend
npm install
node utils/migrate.js
```

#### 2.3 Run Seeding (Initial Data)

```bash
node utils/seed.js
```

### Step 3: Backend Setup

#### 3.1 Install Dependencies

```bash
cd backend
npm install
```

#### 3.2 Configure Environment Variables

Create a `.env` file from the template:

```bash
cp ENV_TEMPLATE.txt .env
```

Edit `.env` with your configuration:

```env
PORT=5000
NODE_ENV=development

# Database Configuration
DB_HOST=localhost
DB_PORT=3306
DB_USER=root
DB_PASSWORD=your_mysql_password
DB_NAME=clinic_pro_db

# JWT Configuration
JWT_SECRET=your_super_secret_jwt_key_change_this
JWT_EXPIRE=7d

# CORS Configuration
CORS_ORIGIN=http://localhost:3000
```

#### 3.3 Start Backend Server

```bash
# Production mode
npm start

# Development mode with auto-reload
npm run dev
```

The backend will run on `http://localhost:5000`

### Step 4: Frontend Setup

#### Option A: Simple HTTP Server (Recommended for Development)

```bash
# Using Python
cd frontend
python -m http.server 3000

# Using Node.js http-server
npx http-server frontend -p 3000
```

#### Option B: Direct File Access

Simply open `frontend/index.html` in a web browser.

#### Access Application

- **Frontend:** `http://localhost:3000`
- **Backend API:** `http://localhost:5000`

---

## 🔐 Default Login Credentials

```
Username: admin
Password: admin
```

> ⚠️ **Important:** Change the default password after first login!

---

## 🎯 Quick Start Guide

### 1. First Time Setup

```bash
# Clone or navigate to project
cd "Clinic-Pro-Web"

# Install backend dependencies
cd backend
npm install

# Create .env file
cp ENV_TEMPLATE.txt .env
# Edit .env with your MySQL credentials

# Run migrations
node utils/migrate.js

# Seed initial data
node utils/seed.js

# Start backend
npm run dev
```

### 2. Open Frontend

```bash
# In a new terminal, from project root
cd frontend

# Start simple HTTP server
python -m http.server 3000
# or
npx http-server -p 3000
```

### 3. Login

- Open browser: `http://localhost:3000`
- Login with: `admin` / `admin`
- Explore the dashboard!

---

## 📁 Project Structure

```
Clinic-Pro-Web/
│
├── backend/                          # Backend API Server
│   ├── config/                       # Configuration files
│   │   ├── database.js               # MySQL connection
│   │   ├── sequelize.js              # Sequelize ORM config
│   │   └── config.js                 # App configuration
│   │
│   ├── controllers/                  # Request handlers
│   │   ├── auth.controller.js        # Authentication
│   │   ├── patient.controller.js    # Patient management
│   │   ├── doctor.controller.js     # Doctor management
│   │   ├── pos.controller.js        # Point of Sale
│   │   ├── product.controller.js    # Product management
│   │   ├── emr.controller.js        # Medical Records
│   │   └── ...                       # More controllers
│   │
│   ├── routes/                       # API routes
│   │   ├── auth.routes.js           # Auth endpoints
│   │   ├── patient.routes.js       # Patient endpoints
│   │   ├── pos.routes.js            # POS endpoints
│   │   ├── emr.routes.js            # EMR endpoints
│   │   └── ...                      # More routes
│   │
│   ├── middleware/                   # Express middleware
│   │   ├── auth.js                  # JWT authentication
│   │   ├── upload.js                # File upload handling
│   │   ├── i18n.js                  # Multi-language
│   │   ├── permissions.js           # Role-based access
│   │   └── errorHandler.js          # Error handling
│   │
│   ├── models/                       # Database models
│   │   ├── BaseModel.js             # Generic CRUD operations
│   │   ├── Customer.model.js        # Customer model
│   │   └── index.js                 # Model exports
│   │
│   ├── migrations/                   # Database migrations
│   │   └── *.js                     # Migration scripts
│   │
│   ├── seeders/                      # Database seeders
│   │   └── *.js                     # Seed scripts
│   │
│   ├── locales/                      # Translation files
│   │   ├── en/
│   │   │   └── translation.json     # English translations
│   │   ├── my/
│   │   │   └── translation.json     # Myanmar translations
│   │   └── zh/
│   │       └── translation.json     # Chinese translations
│   │
│   ├── uploads/                      # Uploaded files
│   │   ├── products/                # Product images
│   │   └── medical/                 # Medical images
│   │
│   ├── utils/                        # Utility functions
│   │   ├── migrate.js               # Database migration
│   │   └── seed.js                  # Database seeding
│   │
│   ├── server.js                     # Main server file
│   └── package.json                  # Dependencies
│
├── frontend/                          # Frontend Application
│   ├── css/                          # Stylesheets
│   │   ├── glass-theme.css          # Glassmorphism theme
│   │   └── ux-utils.css             # Utility styles
│   │
│   ├── js/                           # JavaScript files
│   │   ├── api.js                   # API helper functions
│   │   └── ...                      # More JS files
│   │
│   ├── pages/                        # HTML pages
│   │   ├── dashboard.html           # Main dashboard
│   │   ├── pos.html                 # Point of Sale
│   │   ├── emr.html                 # Medical Records
│   │   ├── add-product.html         # Add Product with Image
│   │   ├── patients.html            # Patient management
│   │   ├── doctors.html             # Doctor management
│   │   └── ...                      # More pages
│   │
│   ├── images/                       # Static images
│   ├── includes/                     # HTML includes
│   ├── templates/                    # HTML templates
│   └── index.html                    # Login page
│
├── database/                          # Database scripts
│   ├── migrations/                   # SQL migrations
│   │   └── *.sql                    # Migration files
│   ├── seeds/                        # Seed data
│   │   └── *.sql                    # Seed files
│   ├── complete_schema.sql           # Complete schema
│   └── README.md                     # Database documentation
│
├── backups/                           # Database backups
│   └── *.sql                         # Backup files
│
├── ecosystem.config.js                # PM2 configuration
├── package.json                       # Root package.json
└── README.md                          # This file
```

---

## 📊 Database Schema

### Core Tables

| Table | Description |
|-------|-------------|
| `tbl_user` | System users |
| `tbl_role` | User roles |
| `tbl_shopinfo` | Shop information |

### Patient Management

| Table | Description |
|-------|-------------|
| `tbl_patient` | Patient records |
| `tbl_inpatient` | Inpatient records |
| `tbl_doctor` | Doctor information |
| `tbl_appointment` | Appointments |
| `tbl_treatment` | Treatment history (EMR) |
| `tbl_images` | Medical images |
| `tbl_case` | Medical cases |

### Inventory & Sales

| Table | Description |
|-------|-------------|
| `tbl_stock` | Product inventory |
| `tbl_catalog` | Product categories |
| `tbl_purchase` | Purchase records |
| `tbl_sale` | Sales transactions |
| `tbl_voucher` | Sale vouchers |
| `tbl_sale_return` | Return records |
| `tbl_warehouse_stock` | Warehouse inventory |
| `tbl_warehouse_transfer` | Stock transfers |

### Financial

| Table | Description |
|-------|-------------|
| `tbl_expenses` | Expense records |
| `tbl_customer_debt` | Customer credits |
| `tbl_supplier_debt` | Supplier credits |
| `tbl_payment_method` | Payment methods |

> **Total:** 46+ tables for complete management!

---

## 🔌 API Endpoints

### Authentication

| Method | Endpoint | Description |
|--------|----------|-------------|
| `POST` | `/api/auth/login` | User login |
| `POST` | `/api/auth/register` | Register new user |
| `GET` | `/api/auth/me` | Get current user |
| `GET` | `/api/auth/logout` | Logout |

### Point of Sale

| Method | Endpoint | Description |
|--------|----------|-------------|
| `GET` | `/api/pos/products` | Get products for POS |
| `GET` | `/api/pos/categories` | Get categories |
| `GET` | `/api/pos/next-voucher` | Get next voucher number |
| `POST` | `/api/pos/sale` | Process sale transaction |
| `POST` | `/api/pos/draft` | Save draft sale |

### Products

| Method | Endpoint | Description |
|--------|----------|-------------|
| `GET` | `/api/products` | Get all products |
| `POST` | `/api/products` | Create product (with image) |
| `PUT` | `/api/products/:id` | Update product |
| `DELETE` | `/api/products/:id` | Delete product |
| `GET` | `/api/products/generate-barcode` | Generate barcode |

### EMR (Electronic Medical Records)

| Method | Endpoint | Description |
|--------|----------|-------------|
| `GET` | `/api/emr/patient/:patientId/history` | Get patient history |
| `POST` | `/api/emr/treatment` | Add treatment record (with images) |
| `PUT` | `/api/emr/treatment/:id` | Update treatment |
| `DELETE` | `/api/emr/treatment/:id` | Delete treatment |
| `DELETE` | `/api/emr/image/:id` | Delete medical image |

### Patients

| Method | Endpoint | Description |
|--------|----------|-------------|
| `GET` | `/api/patients` | Get all patients |
| `POST` | `/api/patients` | Create patient |
| `PUT` | `/api/patients/:id` | Update patient |
| `DELETE` | `/api/patients/:id` | Delete patient |
| `GET` | `/api/patients/search?q=` | Search patients |

### Doctors

| Method | Endpoint | Description |
|--------|----------|-------------|
| `GET` | `/api/doctors` | Get all doctors |
| `POST` | `/api/doctors` | Create doctor |
| `PUT` | `/api/doctors/:id` | Update doctor |
| `DELETE` | `/api/doctors/:id` | Delete doctor |

### Dashboard

| Method | Endpoint | Description |
|--------|----------|-------------|
| `GET` | `/api/dashboard/stats` | Get statistics |
| `GET` | `/api/dashboard/recent` | Get recent activities |

### Stock & Inventory

| Method | Endpoint | Description |
|--------|----------|-------------|
| `GET` | `/api/stock` | Get stock items |
| `GET` | `/api/stock/alerts` | Get low stock alerts |

### Reports

| Method | Endpoint | Description |
|--------|----------|-------------|
| `GET` | `/api/reports/sales` | Sales report |
| `GET` | `/api/reports/profit` | Profit report |
| `GET` | `/api/reports/sales-by-voucher` | Sales by voucher |
| `GET` | `/api/reports/comprehensive-sales` | Comprehensive sales |

---

## 🎨 UI Features

### Glassmorphism Theme

- ✨ Beautiful glass-effect cards
- 🎭 Smooth animations
- 📱 Responsive design
- 🌙 Dark theme optimized
- 🎨 Modern gradient backgrounds

### Key Pages

1. **Dashboard** - Overview with statistics and quick actions
2. **POS** - Product selection with images, cart, and checkout
3. **EMR** - Complete medical records with multi-image upload
4. **Products** - Inventory management with image upload
5. **Patients** - Patient registration and management
6. **Reports** - Analytics and reporting

---

## 🌐 Multi-language Support

Switch between multiple languages:
- 🇺🇸 English
- 🇲🇲 Myanmar (Burmese)
- 🇨🇳 Chinese

### Features

- ✅ All UI elements translated
- ✅ Date/number formatting
- ✅ RTL support ready

### Adding New Language

1. Create translation file: `backend/locales/{lang}/translation.json`
2. Add language option in UI
3. Update language selector

---

## 🔒 Security Features

- 🔐 JWT authentication
- 🔑 Password hashing with bcrypt
- 🛡️ SQL injection protection
- 🚫 XSS prevention
- 🌐 CORS configuration
- ⏱️ Rate limiting
- 📎 File upload validation
- 🔒 Secure session management

---

## 📸 Image Upload

### Product Images

- ✅ Single image per product
- ✅ Auto-resize and optimize
- ✅ Supported formats: JPG, PNG, GIF, WebP
- ✅ Max size: 10MB

### Medical Images

- ✅ Multiple images per treatment (up to 10)
- ✅ Supports: JPG, PNG, GIF, TIFF, BMP
- ✅ Max size per image: 10MB
- ✅ Secure storage with unique filenames

---

## 🧪 Testing

```bash
# Run backend tests
cd backend
npm test

# Check database connection
node -e "require('./config/database').testConnection()"
```

---

## 🐛 Troubleshooting

### Database Connection Error

```
Error: connect ECONNREFUSED
```

**Solution:** Ensure MySQL is running and credentials in `.env` are correct

```bash
# Check MySQL status
sudo systemctl status mysql  # Linux
brew services list           # macOS
```

### Port Already in Use

```
Error: listen EADDRINUSE: address already in use :::5000
```

**Solution:** Change PORT in `.env` or kill process using the port

```bash
# Find process using port
lsof -i :5000  # macOS/Linux
netstat -ano | findstr :5000  # Windows

# Kill process
kill -9 <PID>  # macOS/Linux
taskkill /PID <PID> /F  # Windows
```

### Image Upload Error

```
Error: ENOENT: no such file or directory
```

**Solution:** Ensure `uploads/` directories exist:

```bash
mkdir -p backend/uploads/products
mkdir -p backend/uploads/medical
```

### CORS Error

```
Access to XMLHttpRequest blocked by CORS policy
```

**Solution:** Update `CORS_ORIGIN` in `.env` to match your frontend URL

```env
CORS_ORIGIN=http://localhost:3000
```

---

## 📝 Development Notes

### Adding New Features

1. Create controller in `backend/controllers/`
2. Create routes in `backend/routes/`
3. Add route to `server.js`
4. Create frontend page in `frontend/pages/`
5. Update API helper in `frontend/js/api.js`

### Database Changes

1. Create SQL file in `database/migrations/`
2. Run migration script
3. Update models if needed

### Code Style

- Use ESLint for JavaScript
- Follow RESTful API conventions
- Use async/await for async operations
- Add comments for complex logic

---

## 🤝 Contributing

Contributions are welcome! Please follow these steps:

1. **Fork the repository**
   ```bash
   git clone https://github.com/yourusername/clinic-pro-web.git
   ```

2. **Create a feature branch**
   ```bash
   git checkout -b feature/AmazingFeature
   ```

3. **Make your changes**
   - Write clean, documented code
   - Follow existing code style
   - Add tests if applicable

4. **Commit your changes**
   ```bash
   git commit -m 'Add some AmazingFeature'
   ```

5. **Push to the branch**
   ```bash
   git push origin feature/AmazingFeature
   ```

6. **Open a Pull Request**
   - Describe your changes
   - Reference any related issues
   - Add screenshots if UI changes

### Contribution Guidelines

- ✅ Write meaningful commit messages
- ✅ Update documentation as needed
- ✅ Test your changes thoroughly
- ✅ Follow the existing code style
- ✅ Be respectful and professional

---

## 📄 License

This project is licensed under the **MIT License**.

See the [LICENSE](LICENSE) file for details.

---

## 👥 Support

For support, email: **support@clinicpro.com**

Or open an issue on [GitHub Issues](https://github.com/yourusername/clinic-pro-web/issues)

---

## 🎉 Acknowledgments

- [Bootstrap 5](https://getbootstrap.com/) - CSS Framework
- [Express.js](https://expressjs.com/) - Web Framework
- [MySQL](https://www.mysql.com/) - Database
- [Font Awesome](https://fontawesome.com/) / [Bootstrap Icons](https://icons.getbootstrap.com/) - Icons
- All open-source contributors

---

<div align="center">

**Made with ❤️ for Healthcare Professionals**

**Version:** 3.0.0 | **Last Updated:** 2025

[⬆ Back to Top](#clinic-pro-v3---modern-web-application)

</div>
