const mysql = require('mysql2/promise');

const dbConfig = {
    host: process.env.DB_HOST || 'localhost',
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'clinic_pro_db',
    waitForConnections: true,
    connectionLimit: 10,
    queueLimit: 0
};

async function addActiveColumn() {
    let connection;
    
    try {
        console.log('🔗 Connecting to database...');
        connection = await mysql.createConnection(dbConfig);
        console.log('✅ Connected to database\n');
        
        // Check if column exists
        const [columns] = await connection.execute(`
            SELECT COLUMN_NAME 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = ? 
            AND TABLE_NAME = 'tbl_payment_method' 
            AND COLUMN_NAME = 'active'
        `, [dbConfig.database]);
        
        if (columns.length > 0) {
            console.log('✅ Column "active" already exists in tbl_payment_method');
        } else {
            console.log('➕ Adding "active" column to tbl_payment_method...');
            
            await connection.execute(`
                ALTER TABLE tbl_payment_method
                ADD COLUMN active TINYINT(1) NOT NULL DEFAULT 1 AFTER name
            `);
            
            console.log('✅ Column "active" added successfully!');
            
            // Add index
            try {
                await connection.execute(`
                    CREATE INDEX idx_active ON tbl_payment_method(active)
                `);
                console.log('✅ Index created successfully!');
            } catch (err) {
                if (err.code === 'ER_DUP_KEYNAME') {
                    console.log('ℹ️  Index already exists');
                } else {
                    throw err;
                }
            }
        }
        
        // Show final structure
        const [finalColumns] = await connection.execute(`
            SELECT COLUMN_NAME, DATA_TYPE, IS_NULLABLE, COLUMN_DEFAULT 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'tbl_payment_method'
            ORDER BY ORDINAL_POSITION
        `, [dbConfig.database]);
        
        console.log('\n📊 Final table structure:');
        console.table(finalColumns);
        
        console.log('\n✅ Migration completed successfully!');
        
    } catch (error) {
        console.error('❌ Error:', error.message);
        process.exit(1);
    } finally {
        if (connection) {
            await connection.end();
            console.log('\n🔌 Database connection closed');
        }
    }
}

addActiveColumn();

