/**
 * MySQL Database Configuration
 * All settings are read from .env file
 */

require('dotenv').config({ path: require('path').join(__dirname, '../.env') });
const mysql = require('mysql2/promise');

// Normalize host to ensure IPv4
let dbHost = process.env.DB_HOST || '127.0.0.1';
// Convert 'localhost' to '127.0.0.1' to force IPv4 and avoid IPv6 resolution issues
if (dbHost === 'localhost' || dbHost === '::1') {
    dbHost = '127.0.0.1';
    console.log('⚠️  Converted DB_HOST from localhost to 127.0.0.1 to force IPv4');
}

// Database connection pool configuration - all from .env
const poolConfig = {
    host: dbHost,  // Use 127.0.0.1 instead of localhost for IPv4
    port: parseInt(process.env.DB_PORT) || 3306,
    user: process.env.DB_USER || 'root',
    password: process.env.DB_PASSWORD || '',
    database: process.env.DB_NAME || 'clinic_pro_db',
    waitForConnections: process.env.DB_WAIT_FOR_CONNECTIONS !== 'false',
    connectionLimit: parseInt(process.env.DB_CONNECTION_LIMIT) || 50,
    queueLimit: parseInt(process.env.DB_QUEUE_LIMIT) || 0,
    maxIdle: parseInt(process.env.DB_MAX_IDLE) || 10,
    idleTimeout: parseInt(process.env.DB_IDLE_TIMEOUT) || 300000,
    enableKeepAlive: process.env.DB_ENABLE_KEEP_ALIVE !== 'false',
    keepAliveInitialDelay: parseInt(process.env.DB_KEEP_ALIVE_INITIAL_DELAY) || 10000,
    charset: process.env.DB_CHARSET || 'utf8mb4',
    timezone: process.env.DB_TIMEZONE || '+00:00',
    connectTimeout: parseInt(process.env.DB_CONNECT_TIMEOUT) || 30000,
    multipleStatements: process.env.DB_MULTIPLE_STATEMENTS === 'true',
    // Force IPv4 connection to prevent ::1 (IPv6) resolution issues
    family: 4,
    // Alternative: Use Unix socket for local connections
    ...(process.env.DB_SOCKET && { socketPath: process.env.DB_SOCKET })
};

// Create connection pool
const pool = mysql.createPool(poolConfig);

/**
 * Test database connection
 */
async function testConnection() {
    try {
        const connection = await pool.getConnection();
        console.log('✅ MySQL Database Connected Successfully');
        connection.release();
        return true;
    } catch (error) {
        console.error('❌ Database Connection Error:', error.message);
        return false;
    }
}

/**
 * Execute query with connection pool and automatic retry
 */
async function query(sql, params, retries = 2) {
    for (let attempt = 0; attempt <= retries; attempt++) {
        try {
            // Always use query() instead of execute() to avoid strict type checking
            // query() uses text protocol which handles type coercion automatically
            if (!params || params.length === 0) {
                const [rows] = await pool.query(sql);
                return rows;
            } else {
                const [rows] = await pool.query(sql, params);
                return rows;
            }
        } catch (error) {
            // Check if it's a connection error
            const isConnectionError = 
                error.code === 'PROTOCOL_CONNECTION_LOST' ||
                error.code === 'ECONNREFUSED' ||
                error.code === 'ETIMEDOUT' ||
                error.code === 'ER_CON_COUNT_ERROR' ||
                error.fatal === true;
            
            // If it's the last retry or not a connection error, throw immediately
            if (attempt === retries || !isConnectionError) {
                // Suppress expected table not found errors for optional tables
                const isOptionalTableError = error.code === 'ER_NO_SUCH_TABLE' && 
                    (sql.includes('tbl_lab_reports') || 
                     sql.includes('tbl_emr') || 
                     sql.includes('tbl_inpatient_deposits') ||
                     sql.includes('tbl_inpatient_usage_settled') ||
                     sql.includes('tbl_lab_tests'));
                
                if (!isOptionalTableError) {
                    console.error('Query Error:', {
                        attempt: attempt + 1,
                        code: error.code,
                        message: error.message,
                        sql: sql.substring(0, 100) + '...'
                    });
                }
                throw error;
            }
            
            // Wait before retrying (exponential backoff)
            const waitTime = Math.min(1000 * Math.pow(2, attempt), 5000);
            console.warn(`Connection error, retrying in ${waitTime}ms... (attempt ${attempt + 1}/${retries + 1})`);
            await new Promise(resolve => setTimeout(resolve, waitTime));
        }
    }
}

/**
 * Execute transaction
 */
async function transaction(callback) {
    const connection = await pool.getConnection();
    await connection.beginTransaction();
    
    try {
        const result = await callback(connection);
        await connection.commit();
        connection.release();
        return result;
    } catch (error) {
        await connection.rollback();
        connection.release();
        throw error;
    }
}

/**
 * Close all connections
 */
async function closePool() {
    await pool.end();
    console.log('Database pool closed');
}

/**
 * Get connection pool status
 */
function getPoolStatus() {
    return {
        totalConnections: pool.pool._allConnections.length,
        freeConnections: pool.pool._freeConnections.length,
        activeConnections: pool.pool._allConnections.length - pool.pool._freeConnections.length,
        queuedRequests: pool.pool._connectionQueue.length
    };
}

/**
 * Monitor pool health (log every 5 minutes)
 */
function monitorPoolHealth() {
    setInterval(async () => {
        try {
            const status = getPoolStatus();
            
            // Test connection health
            const connection = await pool.getConnection();
            await connection.ping();
            connection.release();
            
            // Warn about high usage
            if (status.activeConnections > 40) {
                console.warn('⚠️  WARNING: High database connection usage:', status);
            }
            if (status.queuedRequests > 10) {
                console.error('❌ CRITICAL: Many queued database requests:', status);
            }
            
            // Log status if there are active connections
            if (status.activeConnections > 0) {
                console.log('📊 DB Pool Status:', {
                    ...status,
                    timestamp: new Date().toISOString()
                });
            }
        } catch (error) {
            console.error('❌ Health check failed:', error.message);
            // Try to recover by testing connection
            testConnection();
        }
    }, 300000); // Every 5 minutes
}

/**
 * Periodic connection cleanup (every 10 minutes)
 * Closes idle connections that have been sitting too long
 */
function cleanupIdleConnections() {
    setInterval(async () => {
        try {
            // Force pool to clean up idle connections
            const status = getPoolStatus();
            if (status.freeConnections > status.maxIdle) {
                console.log('🧹 Cleaning up excess idle connections...');
                // The pool will automatically handle this with idleTimeout
            }
        } catch (error) {
            console.error('Cleanup error:', error.message);
        }
    }, 600000); // Every 10 minutes
}

// Test connection on startup
testConnection();

// Start pool monitoring
monitorPoolHealth();

// Start periodic cleanup
cleanupIdleConnections();

// Graceful shutdown handler
process.on('SIGINT', async () => {
    console.log('\n🛑 Shutting down gracefully...');
    try {
        await closePool();
        console.log('✅ Database connections closed');
        process.exit(0);
    } catch (error) {
        console.error('❌ Error during shutdown:', error);
        process.exit(1);
    }
});

module.exports = {
    pool,
    query,
    transaction,
    testConnection,
    closePool,
    getPoolStatus
};

