/**
 * Sequelize Configuration
 * Database connection setup for Sequelize ORM
 */

const { Sequelize } = require('sequelize');
const path = require('path');
require('dotenv').config({ path: path.join(__dirname, '../.env') });

// Create Sequelize instance
// Normalize host to ensure IPv4
let dbHost = process.env.DB_HOST || '127.0.0.1';
if (dbHost === 'localhost' || dbHost === '::1') {
    dbHost = '127.0.0.1';
}

const sequelize = new Sequelize(
    process.env.DB_NAME || 'clinic_pro_db',
    process.env.DB_USER || 'root',
    process.env.DB_PASSWORD || '',
    {
        host: dbHost,
        port: parseInt(process.env.DB_PORT) || 3306,
        dialect: 'mysql',
        logging: process.env.NODE_ENV === 'development' ? console.log : false,
        pool: {
            max: parseInt(process.env.DB_POOL_MAX) || 50,
            min: parseInt(process.env.DB_POOL_MIN) || 0,
            acquire: parseInt(process.env.DB_POOL_ACQUIRE) || 30000,
            idle: parseInt(process.env.DB_POOL_IDLE) || 10000
        },
        define: {
            timestamps: true,
            underscored: false,
            freezeTableName: true, // Keep table names as-is (tbl_*)
            charset: process.env.DB_CHARSET || 'utf8mb4',
            collate: process.env.DB_COLLATE || 'utf8mb4_unicode_ci'
        },
        timezone: process.env.DB_TIMEZONE || '+00:00',
        dialectOptions: {
            connectTimeout: parseInt(process.env.DB_CONNECT_TIMEOUT) || 30000
        }
    }
);

// Test connection
async function testConnection() {
    try {
        await sequelize.authenticate();
        console.log('✅ Sequelize connection established successfully');
        return true;
    } catch (error) {
        console.error('❌ Sequelize connection error:', error.message);
        return false;
    }
}

module.exports = {
    sequelize,
    testConnection
};

