/**
 * Comprehensive CRUD Controller for All Tables
 * Handles CRUD operations for all 46 database tables
 */

const { query } = require('../config/database');
const models = require('../models');

/**
 * Generic CRUD operations for any table
 */
class CRUDController {
    constructor(model) {
        this.model = model;
    }

    // Get all records
    getAll = async (req, res, next) => {
        try {
            const result = await this.model.findAll(req.query);
            res.json({
                success: true,
                data: result.data,
                total: result.pagination?.total || result.data?.length || 0,
                count: result.data?.length || 0,
                page: result.pagination?.page || 1,
                pages: result.pagination?.pages || 1
            });
        } catch (error) {
            next(error);
        }
    };

    // Get single record
    getOne = async (req, res, next) => {
        try {
            const data = await this.model.findById(req.params.id);
            if (!data) {
                return res.status(404).json({
                    success: false,
                    message: 'Record not found'
                });
            }
            res.json({ success: true, data });
        } catch (error) {
            next(error);
        }
    };

    // Create record
    create = async (req, res, next) => {
        try {
            const data = await this.model.create(req.body);
            res.status(201).json({
                success: true,
                message: 'Record created successfully',
                data
            });
        } catch (error) {
            next(error);
        }
    };

    // Update record
    update = async (req, res, next) => {
        try {
            const data = await this.model.update(req.params.id, req.body);
            res.json({
                success: true,
                message: 'Record updated successfully',
                data
            });
        } catch (error) {
            next(error);
        }
    };

    // Delete record
    delete = async (req, res, next) => {
        try {
            const deleted = await this.model.delete(req.params.id);
            if (!deleted) {
                return res.status(404).json({
                    success: false,
                    message: 'Record not found'
                });
            }
            res.json({
                success: true,
                message: 'Record deleted successfully'
            });
        } catch (error) {
            next(error);
        }
    };
}

// Create controllers for all models
const controllers = {
    alert: new CRUDController(models.Alert),
    appointment: new CRUDController(models.Appointment),
    case: new CRUDController(models.Case),
    catalog: new CRUDController(models.Catalog),
    catalog2: new CRUDController(models.Catalog2),
    catExpenses: new CRUDController(models.CatExpenses),
    customer: new CRUDController(models.Customer),
    customerDebt: new CRUDController(models.CustomerDebt),
    doctor: new CRUDController(models.Doctor),
    expenses: new CRUDController(models.Expenses),
    form: new CRUDController(models.Form),
    images: new CRUDController(models.Images),
    inpatient: new CRUDController(models.Inpatient),
    inpatientSale: new CRUDController(models.InpatientSale),
    inpatientSaleDebit: new CRUDController(models.InpatientSaleDebit),
    labour: new CRUDController(models.Labour),
    labourBalance: new CRUDController(models.LabourBalance),
    labourLevel: new CRUDController(models.LabourLevel),
    loss: new CRUDController(models.Loss),
    patient: new CRUDController(models.Patient),
    patientSale: new CRUDController(models.PatientSale),
    paymentMethod: new CRUDController(models.PaymentMethod),
    purchase: new CRUDController(models.Purchase),
    purchaseReturn: new CRUDController(models.PurchaseReturn),
    purchaseVoucher: new CRUDController(models.PurchaseVoucher),
    role: new CRUDController(models.Role),
    sale: new CRUDController(models.Sale),
    saleArchive: new CRUDController(models.SaleArchive),
    saleDraft: new CRUDController(models.SaleDraft),
    saleDraftNo: new CRUDController(models.SaleDraftNo),
    salePackage: new CRUDController(models.SalePackage),
    salePackageNo: new CRUDController(models.SalePackageNo),
    saleReturn: new CRUDController(models.SaleReturn),
    saleReturnItem: new CRUDController(models.SaleReturnItem),
    service: new CRUDController(models.Service),
    serviceCatalog: new CRUDController(models.ServiceCatalog),
    shopInfo: new CRUDController(models.ShopInfo),
    stock: new CRUDController(models.Stock),
    stockExpense: new CRUDController(models.StockExpense),
    supplier: new CRUDController(models.Supplier),
    supplierDebt: new CRUDController(models.SupplierDebt),
    treatment: new CRUDController(models.Treatment),
    unit: new CRUDController(models.Unit),
    user: new CRUDController(models.User),
    userLog: new CRUDController(models.UserLog),
    voucher: new CRUDController(models.Voucher),
    voucherArchive: new CRUDController(models.VoucherArchive),
    voucherReturn: new CRUDController(models.VoucherReturn)
};

module.exports = controllers;

