/**
 * Enhanced Analytics & Reporting Controller
 * Quality Metrics, Clinical Indicators, Hospital KPIs
 */

const { query } = require('../config/database');

// ============================================
// QUALITY METRICS
// ============================================

/**
 * @desc    Get quality metrics
 * @route   GET /api/analytics/quality-metrics
 * @access  Private
 */
exports.getQualityMetrics = async (req, res, next) => {
    try {
        const { category, department_id, from, to } = req.query;
        let sql = `SELECT q.*, d.dept_name as department_name
            FROM tbl_quality_metrics q
            LEFT JOIN tbl_departments d ON q.department_id = d.id
            WHERE 1=1`;
        const params = [];

        if (category) {
            sql += ' AND q.metric_category = ?';
            params.push(category);
        }
        if (department_id) {
            sql += ' AND q.department_id = ?';
            params.push(department_id);
        }
        if (from) {
            sql += ' AND q.period_start >= ?';
            params.push(from);
        }
        if (to) {
            sql += ' AND q.period_end <= ?';
            params.push(to);
        }

        sql += ' ORDER BY q.period_start DESC, q.metric_category';
        const metrics = await query(sql, params);

        res.json({
            success: true,
            count: metrics.length,
            data: metrics
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Add quality metric
 * @route   POST /api/analytics/quality-metrics
 * @access  Private
 */
exports.addQualityMetric = async (req, res, next) => {
    try {
        const {
            metric_code,
            metric_name,
            metric_category,
            department_id,
            measurement_period,
            period_start,
            period_end,
            numerator,
            denominator,
            actual_value,
            target_value,
            unit,
            notes
        } = req.body;

        const meets_target = target_value ? (actual_value >= target_value ? 1 : 0) : null;
        const variance = target_value ? (actual_value - target_value) : null;

        const sql = `INSERT INTO tbl_quality_metrics 
            (metric_code, metric_name, metric_category, department_id, measurement_period,
            period_start, period_end, numerator, denominator, actual_value, target_value,
            unit, meets_target, variance, notes, recorded_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;
        
        const result = await query(sql, [
            metric_code,
            metric_name,
            metric_category,
            department_id ?? null,
            measurement_period,
            period_start,
            period_end,
            numerator ?? null,
            denominator ?? null,
            actual_value,
            target_value ?? null,
            unit ?? null,
            meets_target,
            variance,
            notes ?? null,
            req.user.id
        ]);

        res.status(201).json({
            success: true,
            data: { id: result.insertId }
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// CLINICAL INDICATORS
// ============================================

/**
 * @desc    Get clinical indicators
 * @route   GET /api/analytics/clinical-indicators
 * @access  Private
 */
exports.getClinicalIndicators = async (req, res, next) => {
    try {
        const { indicator_type, department_id, year, month } = req.query;
        let sql = `SELECT i.*, d.dept_name as department_name
            FROM tbl_clinical_indicators i
            LEFT JOIN tbl_departments d ON i.department_id = d.id
            WHERE 1=1`;
        const params = [];

        if (indicator_type) {
            sql += ' AND i.indicator_type = ?';
            params.push(indicator_type);
        }
        if (department_id) {
            sql += ' AND i.department_id = ?';
            params.push(department_id);
        }
        if (year) {
            sql += ' AND i.year = ?';
            params.push(year);
        }
        if (month) {
            sql += ' AND i.month = ?';
            params.push(month);
        }

        sql += ' ORDER BY i.year DESC, i.month DESC';
        const indicators = await query(sql, params);

        res.json({
            success: true,
            count: indicators.length,
            data: indicators
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// INFECTION SURVEILLANCE
// ============================================

/**
 * @desc    Get infection surveillance data
 * @route   GET /api/analytics/infections
 * @access  Private
 */
exports.getInfectionSurveillance = async (req, res, next) => {
    try {
        const { infection_type, department_id, from, to } = req.query;
        let sql = `SELECT i.*, p.name as patient_name, d.dept_name as department_name
            FROM tbl_infection_surveillance i
            LEFT JOIN tbl_patient p ON i.patient_id = p.patient_id
            LEFT JOIN tbl_departments d ON i.department_id = d.id
            WHERE 1=1`;
        const params = [];

        if (infection_type) {
            sql += ' AND i.infection_type = ?';
            params.push(infection_type);
        }
        if (department_id) {
            sql += ' AND i.department_id = ?';
            params.push(department_id);
        }
        if (from) {
            sql += ' AND i.onset_date >= ?';
            params.push(from);
        }
        if (to) {
            sql += ' AND i.onset_date <= ?';
            params.push(to);
        }

        sql += ' ORDER BY i.onset_date DESC';
        const infections = await query(sql, params);

        res.json({
            success: true,
            count: infections.length,
            data: infections
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Report infection
 * @route   POST /api/analytics/infections
 * @access  Private
 */
exports.reportInfection = async (req, res, next) => {
    try {
        const {
            patient_id,
            infection_type,
            infection_site,
            onset_date,
            detected_date,
            days_since_admission,
            risk_factors,
            causative_organism,
            department_id,
            associated_procedure,
            device_related,
            device_type
        } = req.body;

        const sql = `INSERT INTO tbl_infection_surveillance 
            (patient_id, infection_type, infection_site, onset_date, detected_date,
            days_since_admission, risk_factors, causative_organism, department_id,
            associated_procedure, device_related, device_type, reported_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;
        
        const result = await query(sql, [
            patient_id ?? null,
            infection_type ?? null,
            infection_site ?? null,
            onset_date ?? null,
            detected_date ?? null,
            days_since_admission ?? null,
            risk_factors ?? null,
            causative_organism ?? null,
            department_id ?? null,
            associated_procedure ?? null,
            device_related ?? null,
            device_type ?? null,
            req.user.id
        ]);

        res.status(201).json({
            success: true,
            data: { id: result.insertId }
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// READMISSIONS TRACKING
// ============================================

/**
 * @desc    Get readmissions
 * @route   GET /api/analytics/readmissions
 * @access  Private
 */
exports.getReadmissions = async (req, res, next) => {
    try {
        const { department_id, from, to, preventable } = req.query;
        let sql = `SELECT r.*, p.name as patient_name, d.dept_name as department_name
            FROM tbl_readmissions r
            LEFT JOIN tbl_patient p ON r.patient_id = p.patient_id
            LEFT JOIN tbl_departments d ON r.department_id = d.id
            WHERE 1=1`;
        const params = [];

        if (department_id) {
            sql += ' AND r.department_id = ?';
            params.push(department_id);
        }
        if (from) {
            sql += ' AND r.readmission_date >= ?';
            params.push(from);
        }
        if (to) {
            sql += ' AND r.readmission_date <= ?';
            params.push(to);
        }
        if (preventable) {
            sql += ' AND r.preventability_assessment = ?';
            params.push(preventable);
        }

        sql += ' ORDER BY r.readmission_date DESC';
        const readmissions = await query(sql, params);

        res.json({
            success: true,
            count: readmissions.length,
            data: readmissions
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// PATIENT SATISFACTION
// ============================================

/**
 * @desc    Get patient satisfaction scores
 * @route   GET /api/analytics/patient-satisfaction
 * @access  Private
 */
exports.getPatientSatisfaction = async (req, res, next) => {
    try {
        const { department_id, from, to, encounter_type } = req.query;
        let sql = `SELECT ps.*, d.dept_name as department_name
            FROM tbl_patient_satisfaction ps
            LEFT JOIN tbl_departments d ON ps.department_id = d.id
            WHERE 1=1`;
        const params = [];

        if (department_id) {
            sql += ' AND ps.department_id = ?';
            params.push(department_id);
        }
        if (from) {
            sql += ' AND ps.survey_date >= ?';
            params.push(from);
        }
        if (to) {
            sql += ' AND ps.survey_date <= ?';
            params.push(to);
        }
        if (encounter_type) {
            sql += ' AND ps.encounter_type = ?';
            params.push(encounter_type);
        }

        sql += ' ORDER BY ps.survey_date DESC';
        const surveys = await query(sql, params);

        // Calculate averages
        if (surveys.length > 0) {
            const avgSql = `SELECT 
                AVG(overall_rating) as avg_overall,
                AVG(staff_courtesy) as avg_courtesy,
                AVG(communication) as avg_communication,
                AVG(cleanliness) as avg_cleanliness,
                AVG(wait_time) as avg_wait_time,
                AVG(pain_management) as avg_pain_mgmt,
                SUM(would_recommend) / COUNT(*) * 100 as recommend_percentage
                FROM tbl_patient_satisfaction
                WHERE survey_date BETWEEN ? AND ?`;
            
            const avgParams = [from || '1900-01-01', to || '2099-12-31'];
            if (department_id) {
                avgSql += ' AND department_id = ?';
                avgParams.push(department_id);
            }
            
            const averages = await query(avgSql, avgParams);
            
            return res.json({
                success: true,
                count: surveys.length,
                averages: averages[0],
                data: surveys
            });
        }

        res.json({
            success: true,
            count: 0,
            data: []
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// OPERATIONAL METRICS
// ============================================

/**
 * @desc    Get operational metrics
 * @route   GET /api/analytics/operational-metrics
 * @access  Private
 */
exports.getOperationalMetrics = async (req, res, next) => {
    try {
        const { department_id, from, to } = req.query;
        let sql = `SELECT o.*, d.dept_name as department_name
            FROM tbl_operational_metrics o
            LEFT JOIN tbl_departments d ON o.department_id = d.id
            WHERE 1=1`;
        const params = [];

        if (department_id) {
            sql += ' AND o.department_id = ?';
            params.push(department_id);
        }
        if (from) {
            sql += ' AND o.metric_date >= ?';
            params.push(from);
        }
        if (to) {
            sql += ' AND o.metric_date <= ?';
            params.push(to);
        }

        sql += ' ORDER BY o.metric_date DESC';
        const metrics = await query(sql, params);

        res.json({
            success: true,
            count: metrics.length,
            data: metrics
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// INCIDENT REPORTS
// ============================================

/**
 * @desc    Get incident reports
 * @route   GET /api/analytics/incidents
 * @access  Private
 */
exports.getIncidentReports = async (req, res, next) => {
    try {
        const { incident_type, severity, department_id, status } = req.query;
        let sql = `SELECT i.*, d.dept_name as department_name, u.name as reported_by_name
            FROM tbl_incident_reports i
            LEFT JOIN tbl_departments d ON i.department_id = d.id
            LEFT JOIN tbl_user u ON i.reported_by = u.id
            WHERE 1=1`;
        const params = [];

        if (incident_type) {
            sql += ' AND i.incident_type = ?';
            params.push(incident_type);
        }
        if (severity) {
            sql += ' AND i.severity = ?';
            params.push(severity);
        }
        if (department_id) {
            sql += ' AND i.department_id = ?';
            params.push(department_id);
        }
        if (status) {
            sql += ' AND i.investigation_status = ?';
            params.push(status);
        }

        sql += ' ORDER BY i.incident_date DESC LIMIT 100';
        const incidents = await query(sql, params);

        res.json({
            success: true,
            count: incidents.length,
            data: incidents
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Report incident
 * @route   POST /api/analytics/incidents
 * @access  Private
 */
exports.reportIncident = async (req, res, next) => {
    try {
        const {
            incident_date,
            incident_type,
            severity,
            patient_id,
            location,
            department_id,
            description,
            immediate_action
        } = req.body;

        if (!incident_date || !incident_type || !severity || !description) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing'
            });
        }

        // Generate incident number
        const year = new Date().getFullYear();
        await query('UPDATE tbl_incident_sequence SET last_incident_number = last_incident_number + 1 WHERE year = ?', [year]);
        const seqResult = await query('SELECT last_incident_number FROM tbl_incident_sequence WHERE year = ?', [year]);
        const incidentNum = seqResult[0].last_incident_number;
        const incident_number = `INC${year}${String(incidentNum).padStart(6, '0')}`;

        const sql = `INSERT INTO tbl_incident_reports 
            (incident_number, incident_date, incident_type, severity, patient_id, location,
            department_id, description, immediate_action, reported_by, investigation_status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Pending')`;
        
        const result = await query(sql, [
            incident_number,
            incident_date,
            incident_type,
            severity,
            patient_id ?? null,
            location ?? null,
            department_id ?? null,
            description,
            immediate_action ?? null,
            req.user.id
        ]);

        res.status(201).json({
            success: true,
            data: {
                id: result.insertId,
                incident_number: incident_number
            }
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// DASHBOARD & SUMMARY STATISTICS
// ============================================

/**
 * @desc    Get comprehensive dashboard statistics
 * @route   GET /api/analytics/dashboard
 * @access  Private
 */
exports.getDashboardStats = async (req, res, next) => {
    try {
        const today = new Date().toISOString().split('T')[0];
        const thisMonth = today.substring(0, 7);

        // Quality metrics summary
        const qualityMetrics = await query(`
            SELECT 
                metric_category,
                COUNT(*) as total_metrics,
                SUM(CASE WHEN meets_target = 1 THEN 1 ELSE 0 END) as met_targets,
                ROUND((SUM(CASE WHEN meets_target = 1 THEN 1 ELSE 0 END) / COUNT(*)) * 100, 2) as success_rate
            FROM tbl_quality_metrics
            WHERE DATE_FORMAT(period_start, '%Y-%m') = '${thisMonth}'
            GROUP BY metric_category
        `);

        // Infection rates
        const infections = await query(`
            SELECT 
                COUNT(*) as total_infections,
                SUM(CASE WHEN device_related = 1 THEN 1 ELSE 0 END) as device_related,
                SUM(CASE WHEN preventable = 1 THEN 1 ELSE 0 END) as preventable
            FROM tbl_infection_surveillance
            WHERE DATE_FORMAT(onset_date, '%Y-%m') = '${thisMonth}'
        `);

        // Readmissions
        const readmissions = await query(`
            SELECT 
                COUNT(*) as total_readmissions,
                AVG(days_since_discharge) as avg_days,
                SUM(CASE WHEN is_related = 1 THEN 1 ELSE 0 END) as related,
                SUM(CASE WHEN preventability_assessment IN ('Preventable', 'Potentially_Preventable') THEN 1 ELSE 0 END) as preventable
            FROM tbl_readmissions
            WHERE DATE_FORMAT(readmission_date, '%Y-%m') = '${thisMonth}'
        `);

        // Patient satisfaction
        const satisfaction = await query(`
            SELECT 
                AVG(overall_rating) as avg_rating,
                COUNT(*) as total_surveys,
                ROUND((SUM(would_recommend) / COUNT(*)) * 100, 2) as recommend_percentage
            FROM tbl_patient_satisfaction
            WHERE DATE_FORMAT(survey_date, '%Y-%m') = '${thisMonth}'
        `);

        // Open incidents
        const incidents = await query(`
            SELECT 
                COUNT(*) as total_incidents,
                SUM(CASE WHEN severity IN ('Severe', 'Death') THEN 1 ELSE 0 END) as critical,
                SUM(CASE WHEN investigation_status = 'Pending' THEN 1 ELSE 0 END) as pending_investigation
            FROM tbl_incident_reports
            WHERE investigation_status != 'Completed'
        `);

        res.json({
            success: true,
            data: {
                quality_metrics: qualityMetrics,
                infections: infections[0],
                readmissions: readmissions[0],
                satisfaction: satisfaction[0],
                incidents: incidents[0],
                period: thisMonth
            }
        });
    } catch (error) {
        next(error);
    }
};

