/**
 * Authentication Controller
 */

const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const { query } = require('../config/database');
const { getUserPermissions } = require('../utils/permissions');

/**
 * @desc    Register user
 * @route   POST /api/auth/register
 * @access  Public
 */
exports.register = async (req, res, next) => {
    try {
        const { name, passwd, role_id, ph_no, nrc_no, address } = req.body;

        // Hash password
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(passwd, salt);

        // Insert user
        const sql = `INSERT INTO tbl_user (name, passwd, role_id, ph_no, nrc_no, address, date) 
                     VALUES (?, ?, ?, ?, ?, ?, CURDATE())`;
        
        const result = await query(sql, [name, hashedPassword, role_id || '2', ph_no, nrc_no, address]);

        // Generate JWT token
        const token = jwt.sign(
            { id: result.insertId },
            process.env.JWT_SECRET,
            { expiresIn: process.env.JWT_EXPIRE }
        );

        res.status(201).json({
            success: true,
            token,
            user: {
                id: result.insertId,
                name,
                role_id: role_id || '2'
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Login user
 * @route   POST /api/auth/login
 * @access  Public
 */
exports.login = async (req, res, next) => {
    try {
        const { name, passwd } = req.body;

        console.log('Login attempt for user:', name);

        // Validate input
        if (!name || !passwd) {
            console.log('Missing username or password');
            return res.status(400).json({
                success: false,
                message: 'Please provide username and password'
            });
        }

        // Check if user exists
        const sql = 'SELECT * FROM tbl_user WHERE name = ?';
        const users = await query(sql, [name]);

        if (users.length === 0) {
            console.log('User not found:', name);
            return res.status(401).json({
                success: false,
                message: 'Invalid credentials'
            });
        }

        const user = users[0];
        console.log('User found, checking password...');

        // Check password
        const isMatch = await bcrypt.compare(passwd, user.passwd);
        console.log('Password match:', isMatch);

        if (!isMatch) {
            console.log('Password mismatch for user:', name);
            return res.status(401).json({
                success: false,
                message: 'Invalid credentials'
            });
        }

        // Generate JWT token
        console.log('Generating JWT token...');
        console.log('JWT_SECRET exists:', !!process.env.JWT_SECRET);
        
        const token = jwt.sign(
            { id: user.id },
            process.env.JWT_SECRET,
            { expiresIn: process.env.JWT_EXPIRE }
        );
        
        console.log('Token generated successfully');

        // Load user permissions into session for faster checks
        let userPermissions = {};
        try {
            userPermissions = await getUserPermissions(user.id);
            // Store permissions in session
            if (req.session) {
                req.session.userId = user.id;
                req.session.permissions = userPermissions;
                req.session.roleId = user.role_id;
            }
            console.log('User permissions loaded into session');
        } catch (permError) {
            console.log('Warning: Could not load user permissions:', permError.message);
            // Don't fail login if permission loading fails
        }

        // Log user login
        console.log('Logging user login...');
        try {
            const logSql = 'INSERT INTO tbl_user_log (user_name, login_time) VALUES (?, NOW())';
            await query(logSql, [user.name]);
            console.log('User login logged');
        } catch (logError) {
            console.log('Warning: Could not log user login:', logError.message);
            // Don't fail login if logging fails
        }

        console.log('Sending successful response...');
        res.status(200).json({
            success: true,
            token,
            user: {
                id: user.id,
                name: user.name,
                role_id: user.role_id,
                ph_no: user.ph_no
            },
            permissions: userPermissions // Include permissions in response for frontend
        });
        console.log('Login successful!');
    } catch (error) {
        console.error('Login error:', error);
        console.error('Error stack:', error.stack);
        next(error);
    }
};

/**
 * @desc    Get current logged in user
 * @route   GET /api/auth/me
 * @access  Private
 */
exports.getMe = async (req, res, next) => {
    try {
        const sql = 'SELECT id, name, role_id, ph_no, address, date FROM tbl_user WHERE id = ?';
        const users = await query(sql, [req.user.id]);

        // Load permissions if not in session
        let permissions = {};
        if (req.session && req.session.permissions) {
            permissions = req.session.permissions;
        } else {
            try {
                permissions = await getUserPermissions(req.user.id);
                if (req.session) {
                    req.session.permissions = permissions;
                }
            } catch (permError) {
                console.error('Error loading permissions:', permError);
            }
        }

        res.status(200).json({
            success: true,
            data: {
                ...users[0],
                permissions
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Logout user
 * @route   GET /api/auth/logout
 * @access  Private
 */
exports.logout = async (req, res, next) => {
    try {
        // Update logout time
        const sql = `UPDATE tbl_user_log 
                     SET logout_time = NOW() 
                     WHERE user_name = ? 
                     AND logout_time IS NULL 
                     ORDER BY login_time DESC 
                     LIMIT 1`;
        
        await query(sql, [req.user.name]);

        // Clear session
        if (req.session) {
            req.session.destroy((err) => {
                if (err) {
                    console.error('Error destroying session:', err);
                }
            });
        }

        res.status(200).json({
            success: true,
            message: 'Logged out successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update password
 * @route   PUT /api/auth/update-password
 * @access  Private
 */
exports.updatePassword = async (req, res, next) => {
    try {
        const { currentPassword, newPassword } = req.body;

        // Get user with password
        const sql = 'SELECT passwd FROM tbl_user WHERE id = ?';
        const users = await query(sql, [req.user.id]);
        const user = users[0];

        // Check current password
        const isMatch = await bcrypt.compare(currentPassword, user.passwd);

        if (!isMatch) {
            return res.status(401).json({
                success: false,
                message: 'Current password is incorrect'
            });
        }

        // Hash new password
        const salt = await bcrypt.genSalt(10);
        const hashedPassword = await bcrypt.hash(newPassword, salt);

        // Update password
        const updateSql = 'UPDATE tbl_user SET passwd = ? WHERE id = ?';
        await query(updateSql, [hashedPassword, req.user.id]);

        res.status(200).json({
            success: true,
            message: 'Password updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update user profile
 * @route   PUT /api/auth/update-profile
 * @access  Private
 */
exports.updateProfile = async (req, res, next) => {
    try {
        const { ph_no, nrc_no, address } = req.body;

        const sql = 'UPDATE tbl_user SET ph_no = ?, nrc_no = ?, address = ? WHERE id = ?';
        await query(sql, [ph_no, nrc_no, address, req.user.id]);

        res.status(200).json({
            success: true,
            message: 'Profile updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

