const { query } = require('../config/database');
const fs = require('fs').promises;
const path = require('path');
const { exec } = require('child_process');
const util = require('util');
const execPromise = util.promisify(exec);

// Create backups directory if it doesn't exist
const backupsDir = path.join(__dirname, '../../backups');

async function ensureBackupsDir() {
    try {
        await fs.access(backupsDir);
    } catch {
        await fs.mkdir(backupsDir, { recursive: true });
    }
}

/**
 * Create database backup
 */
exports.createBackup = async (req, res, next) => {
    try {
        await ensureBackupsDir();

        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
        const filename = `clinic_backup_${timestamp}.sql`;
        const filepath = path.join(backupsDir, filename);

        // Get all tables
        const tables = await query('SHOW TABLES');
        const tableName = Object.keys(tables[0])[0];
        const tableList = tables.map(t => t[tableName]);

        let sqlDump = `-- Clinic Pro V3 Database Backup
-- Created: ${new Date().toISOString()}
-- Tables: ${tableList.length}

SET FOREIGN_KEY_CHECKS=0;

`;

        // Dump each table (ensure deterministic order)
        for (const table of tableList.sort()) {
            // Get CREATE TABLE statement
            const createTableResult = await query(`SHOW CREATE TABLE \`${table}\``);
            const createTableSql = createTableResult[0]['Create Table'];
            
            sqlDump += `-- Table: ${table}\n`;
            sqlDump += `DROP TABLE IF EXISTS \`${table}\`;\n`;
            sqlDump += `${createTableSql};\n\n`;

            // Get table data
            const rows = await query(`SELECT * FROM \`${table}\``);
            
            if (rows.length > 0) {
                sqlDump += `-- Data for table: ${table}\n`;
                
                // Get column names
                const columns = Object.keys(rows[0]);
                const columnList = columns.map(col => `\`${col}\``).join(', ');
                
                // Insert data in batches
                for (let i = 0; i < rows.length; i += 100) {
                    const batch = rows.slice(i, i + 100);
                    
                    sqlDump += `INSERT INTO \`${table}\` (${columnList}) VALUES\n`;
                    
                    const values = batch.map(row => {
                        const vals = columns.map(col => {
                            const val = row[col];
                            if (val === null) return 'NULL';
                            if (typeof val === 'number') return val;
                            if (typeof val === 'bigint') return val.toString();
                            if (val instanceof Date) {
                                // Check if date is valid
                                if (isNaN(val.getTime())) {
                                    return 'NULL';
                                }
                                return `'${val.toISOString().slice(0, 19).replace('T', ' ')}'`;
                            }
                            // Escape single quotes and backslashes
                            const escaped = String(val)
                                .replace(/\\/g, '\\\\')
                                .replace(/\r?\n/g, '\\n')
                                .replace(/\t/g, '\\t')
                                .replace(/'/g, "\\'");
                            return `'${escaped}'`;
                        });
                        return `(${vals.join(', ')})`;
                    });
                    
                    sqlDump += values.join(',\n');
                    sqlDump += ';\n\n';
                }
            }
        }

        sqlDump += `SET FOREIGN_KEY_CHECKS=1;\n`;

        // Write to file
        await fs.writeFile(filepath, sqlDump, 'utf8');

        // Get file stats
        const stats = await fs.stat(filepath);

        res.status(200).json({
            success: true,
            message: 'Backup created successfully',
            data: {
                filename: filename,
                filepath: filepath,
                size: stats.size,
                tables: tableList.length,
                timestamp: new Date().toISOString()
            }
        });
    } catch (error) {
        console.error('Backup error:', error);
        next(error);
    }
};

/**
 * Get list of available backups
 */
exports.getBackups = async (req, res, next) => {
    try {
        await ensureBackupsDir();

        const files = await fs.readdir(backupsDir);
        const sqlFiles = files.filter(f => f.endsWith('.sql'));

        const backups = await Promise.all(
            sqlFiles.map(async (filename) => {
                const filepath = path.join(backupsDir, filename);
                const stats = await fs.stat(filepath);
                
                return {
                    filename: filename,
                    size: stats.size,
                    created: stats.birthtime,
                    modified: stats.mtime
                };
            })
        );

        // Sort by creation date (newest first)
        backups.sort((a, b) => b.created - a.created);

        res.status(200).json({
            success: true,
            count: backups.length,
            data: backups
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Download backup file
 */
exports.downloadBackup = async (req, res, next) => {
    try {
        const { filename } = req.params;
        
        // Validate filename (prevent directory traversal)
        if (filename.includes('..') || filename.includes('/') || filename.includes('\\')) {
            return res.status(400).json({
                success: false,
                message: 'Invalid filename'
            });
        }

        const filepath = path.join(backupsDir, filename);

        // Check if file exists
        try {
            await fs.access(filepath);
        } catch {
            return res.status(404).json({
                success: false,
                message: 'Backup file not found'
            });
        }

        res.download(filepath, filename);
    } catch (error) {
        next(error);
    }
};

/**
 * Restore database from backup
 */
exports.restoreBackup = async (req, res, next) => {
    try {
        const { filename } = req.body;

        if (!filename) {
            return res.status(400).json({
                success: false,
                message: 'Filename is required'
            });
        }

        // Validate filename
        if (filename.includes('..') || filename.includes('/') || filename.includes('\\')) {
            return res.status(400).json({
                success: false,
                message: 'Invalid filename'
            });
        }

        const filepath = path.join(backupsDir, filename);

        // Check if file exists
        try {
            await fs.access(filepath);
        } catch {
            return res.status(404).json({
                success: false,
                message: 'Backup file not found'
            });
        }

        // Read SQL file
        const sqlContent = await fs.readFile(filepath, 'utf8');

        // Split into statements safely (ignore semicolons inside quotes)
        function splitSqlStatements(input) {
            const parts = [];
            let current = '';
            let inSingle = false;
            let inDouble = false;
            for (let i = 0; i < input.length; i++) {
                const ch = input[i];
                const prev = i > 0 ? input[i - 1] : '';
                if (ch === "'" && !inDouble && prev !== '\\') inSingle = !inSingle;
                else if (ch === '"' && !inSingle && prev !== '\\') inDouble = !inDouble;
                if (ch === ';' && !inSingle && !inDouble) {
                    const stmt = current.trim();
                    if (stmt && !stmt.startsWith('--')) parts.push(stmt);
                    current = '';
                } else {
                    current += ch;
                }
            }
            const tail = current.trim();
            if (tail && !tail.startsWith('--')) parts.push(tail);
            return parts;
        }

        const statements = splitSqlStatements(sqlContent);

        // Execute each statement with FK checks off
        await query('SET FOREIGN_KEY_CHECKS=0');
        let successCount = 0;
        for (const statement of statements) {
            try {
                await query(statement);
                successCount++;
            } catch (error) {
                console.error('Statement error:', error.message);
            }
        }
        await query('SET FOREIGN_KEY_CHECKS=1');

        res.status(200).json({
            success: true,
            message: 'Database restored successfully',
            data: {
                filename: filename,
                statements: statements.length,
                successful: successCount
            }
        });
    } catch (error) {
        console.error('Restore error:', error);
        next(error);
    }
};

/**
 * Delete backup file
 */
exports.deleteBackup = async (req, res, next) => {
    try {
        const { filename } = req.params;

        // Validate filename
        if (filename.includes('..') || filename.includes('/') || filename.includes('\\')) {
            return res.status(400).json({
                success: false,
                message: 'Invalid filename'
            });
        }

        const filepath = path.join(backupsDir, filename);

        // Check if file exists
        try {
            await fs.access(filepath);
        } catch {
            return res.status(404).json({
                success: false,
                message: 'Backup file not found'
            });
        }

        // Delete file
        await fs.unlink(filepath);

        res.status(200).json({
            success: true,
            message: 'Backup deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Upload backup file
 */
exports.uploadBackup = async (req, res, next) => {
    try {
        if (!req.file) {
            return res.status(400).json({
                success: false,
                message: 'No file uploaded'
            });
        }

        await ensureBackupsDir();

        const filename = `uploaded_${Date.now()}_${req.file.originalname}`;
        const filepath = path.join(backupsDir, filename);

        // Move file to backups directory
        await fs.rename(req.file.path, filepath);

        res.status(200).json({
            success: true,
            message: 'Backup file uploaded successfully',
            data: {
                filename: filename
            }
        });
    } catch (error) {
        next(error);
    }
};

