// Commission Controller
const { query } = require('../config/database');

// Get all commissions with filters
exports.getAllCommissions = async (req, res, next) => {
    try {
        const { 
            referring_doctor_id, 
            payment_status, 
            start_date, 
            end_date,
            transaction_type,
            limit = 50, 
            offset = 0 
        } = req.query;
        
        let sql = `
            SELECT 
                rc.*,
                rd.name as referring_doctor_name,
                rd.specialty as doctor_specialty
            FROM tbl_referral_commission rc
            JOIN tbl_referring_doctor rd ON rc.referring_doctor_id = rd.id
            WHERE 1=1
        `;
        const params = [];
        
        if (referring_doctor_id) {
            sql += ' AND rc.referring_doctor_id = ?';
            params.push(referring_doctor_id);
        }
        
        if (payment_status) {
            sql += ' AND rc.payment_status = ?';
            params.push(payment_status);
        }
        
        if (transaction_type) {
            sql += ' AND rc.transaction_type = ?';
            params.push(transaction_type);
        }
        
        if (start_date && end_date) {
            sql += ' AND DATE(rc.transaction_date) BETWEEN DATE(?) AND DATE(?)';
            params.push(start_date, end_date);
        }
        
        sql += ' ORDER BY rc.transaction_date DESC, rc.created_at DESC LIMIT ? OFFSET ?';
        params.push(parseInt(limit), parseInt(offset));
        
        const commissions = await query(sql, params);
        
        // Get total count
        let countSql = 'SELECT COUNT(*) as total FROM tbl_referral_commission rc WHERE 1=1';
        const countParams = [];
        
        if (referring_doctor_id) {
            countSql += ' AND rc.referring_doctor_id = ?';
            countParams.push(referring_doctor_id);
        }
        
        if (payment_status) {
            countSql += ' AND rc.payment_status = ?';
            countParams.push(payment_status);
        }
        
        if (transaction_type) {
            countSql += ' AND rc.transaction_type = ?';
            countParams.push(transaction_type);
        }
        
        if (start_date && end_date) {
            countSql += ' AND DATE(rc.transaction_date) BETWEEN DATE(?) AND DATE(?)';
            countParams.push(start_date, end_date);
        }
        
        const [{ total }] = await query(countSql, countParams);
        
        res.json({
            success: true,
            data: commissions,
            pagination: {
                total,
                limit: parseInt(limit),
                offset: parseInt(offset),
                hasMore: (parseInt(offset) + commissions.length) < total
            }
        });
    } catch (error) {
        next(error);
    }
};

// Get commission by ID
exports.getCommissionById = async (req, res, next) => {
    try {
        const { id } = req.params;
        
        const [commission] = await query(
            `SELECT 
                rc.*,
                rd.name as referring_doctor_name,
                rd.specialty as doctor_specialty,
                rd.clinic_name
            FROM tbl_referral_commission rc
            JOIN tbl_referring_doctor rd ON rc.referring_doctor_id = rd.id
            WHERE rc.id = ?`,
            [id]
        );
        
        if (!commission) {
            return res.status(404).json({
                success: false,
                message: 'Commission record not found'
            });
        }
        
        res.json({
            success: true,
            data: commission
        });
    } catch (error) {
        next(error);
    }
};

// Create commission (usually automated, but can be manual)
exports.createCommission = async (req, res, next) => {
    try {
        const {
            referring_doctor_id,
            transaction_type,
            sale_id,
            lab_order_id,
            voucher_no,
            patient_id,
            patient_name,
            transaction_date,
            transaction_amount
        } = req.body;
        
        // Validation
        if (!referring_doctor_id || !transaction_type || !transaction_amount) {
            return res.status(400).json({
                success: false,
                message: 'Missing required fields'
            });
        }
        
        // Get referring doctor's commission settings
        const [doctor] = await query(
            'SELECT commission_type, commission_value FROM tbl_referring_doctor WHERE id = ? AND active = 1',
            [referring_doctor_id]
        );
        
        if (!doctor) {
            return res.status(404).json({
                success: false,
                message: 'Referring doctor not found or inactive'
            });
        }
        
        // Calculate commission
        let commission_amount = 0;
        let commission_rate = null;
        
        if (doctor.commission_type === 'percentage') {
            commission_amount = (transaction_amount * doctor.commission_value) / 100;
            commission_rate = doctor.commission_value;
        } else if (doctor.commission_type === 'fixed') {
            commission_amount = doctor.commission_value;
        }
        
        // Create commission record
        const result = await query(
            `INSERT INTO tbl_referral_commission 
            (referring_doctor_id, transaction_type, sale_id, lab_order_id, voucher_no,
             patient_id, patient_name, transaction_date, transaction_amount,
             commission_type, commission_rate, commission_amount, payment_status, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', ?)`,
            [referring_doctor_id, transaction_type, sale_id, lab_order_id, voucher_no,
             patient_id, patient_name, transaction_date || new Date(), transaction_amount,
             doctor.commission_type, commission_rate, commission_amount, req.user?.id]
        );
        
        // Update referring doctor totals
        await query(
            `UPDATE tbl_referring_doctor 
            SET total_referrals = total_referrals + 1,
                total_revenue = total_revenue + ?,
                total_commission_earned = total_commission_earned + ?
            WHERE id = ?`,
            [transaction_amount, commission_amount, referring_doctor_id]
        );
        
        const [newCommission] = await query(
            'SELECT * FROM tbl_referral_commission WHERE id = ?',
            [result.insertId]
        );
        
        res.status(201).json({
            success: true,
            message: 'Commission created successfully',
            data: newCommission
        });
    } catch (error) {
        next(error);
    }
};

// Approve commission
exports.approveCommission = async (req, res, next) => {
    try {
        const { id } = req.params;
        
        const [commission] = await query(
            'SELECT * FROM tbl_referral_commission WHERE id = ?',
            [id]
        );
        
        if (!commission) {
            return res.status(404).json({
                success: false,
                message: 'Commission not found'
            });
        }
        
        if (commission.payment_status !== 'pending') {
            return res.status(400).json({
                success: false,
                message: `Commission is already ${commission.payment_status}`
            });
        }
        
        await query(
            `UPDATE tbl_referral_commission 
            SET payment_status = 'approved', 
                approved_at = NOW(), 
                approved_by = ?
            WHERE id = ?`,
            [req.user?.id, id]
        );
        
        const [updated] = await query(
            'SELECT * FROM tbl_referral_commission WHERE id = ?',
            [id]
        );
        
        res.json({
            success: true,
            message: 'Commission approved successfully',
            data: updated
        });
    } catch (error) {
        next(error);
    }
};

// Mark commission as paid
exports.payCommission = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { payment_date, payment_method, payment_reference, payment_notes } = req.body;
        
        const [commission] = await query(
            'SELECT * FROM tbl_referral_commission WHERE id = ?',
            [id]
        );
        
        if (!commission) {
            return res.status(404).json({
                success: false,
                message: 'Commission not found'
            });
        }
        
        if (commission.payment_status === 'paid') {
            return res.status(400).json({
                success: false,
                message: 'Commission already paid'
            });
        }
        
        await query(
            `UPDATE tbl_referral_commission 
            SET payment_status = 'paid',
                payment_date = ?,
                payment_method = ?,
                payment_reference = ?,
                payment_notes = ?,
                paid_at = NOW(),
                paid_by = ?
            WHERE id = ?`,
            [payment_date || new Date(), payment_method, payment_reference, payment_notes, req.user?.id, id]
        );
        
        // Update referring doctor's paid total
        await query(
            `UPDATE tbl_referring_doctor 
            SET total_commission_paid = total_commission_paid + ?
            WHERE id = ?`,
            [commission.commission_amount, commission.referring_doctor_id]
        );
        
        const [updated] = await query(
            'SELECT * FROM tbl_referral_commission WHERE id = ?',
            [id]
        );
        
        res.json({
            success: true,
            message: 'Commission marked as paid successfully',
            data: updated
        });
    } catch (error) {
        next(error);
    }
};

// Batch pay commissions
exports.batchPayCommissions = async (req, res, next) => {
    try {
        const { commission_ids, payment_date, payment_method, payment_reference, payment_notes } = req.body;
        
        if (!commission_ids || !Array.isArray(commission_ids) || commission_ids.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'Invalid commission IDs'
            });
        }
        
        const placeholders = commission_ids.map(() => '?').join(',');
        
        // Get commissions
        const commissions = await query(
            `SELECT id, commission_amount, referring_doctor_id, payment_status 
            FROM tbl_referral_commission 
            WHERE id IN (${placeholders}) AND payment_status IN ('pending', 'approved')`,
            commission_ids
        );
        
        if (commissions.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'No eligible commissions found'
            });
        }
        
        // Update all commissions
        await query(
            `UPDATE tbl_referral_commission 
            SET payment_status = 'paid',
                payment_date = ?,
                payment_method = ?,
                payment_reference = ?,
                payment_notes = ?,
                paid_at = NOW(),
                paid_by = ?
            WHERE id IN (${placeholders})`,
            [payment_date || new Date(), payment_method, payment_reference, payment_notes, req.user?.id, ...commission_ids]
        );
        
        // Update referring doctors' paid totals
        const doctorTotals = {};
        commissions.forEach(c => {
            if (!doctorTotals[c.referring_doctor_id]) {
                doctorTotals[c.referring_doctor_id] = 0;
            }
            doctorTotals[c.referring_doctor_id] += parseFloat(c.commission_amount);
        });
        
        for (const [doctorId, totalAmount] of Object.entries(doctorTotals)) {
            await query(
                `UPDATE tbl_referring_doctor 
                SET total_commission_paid = total_commission_paid + ?
                WHERE id = ?`,
                [totalAmount, doctorId]
            );
        }
        
        res.json({
            success: true,
            message: `${commissions.length} commissions marked as paid successfully`,
            data: { count: commissions.length, total_amount: commissions.reduce((sum, c) => sum + parseFloat(c.commission_amount), 0) }
        });
    } catch (error) {
        next(error);
    }
};

// Cancel commission
exports.cancelCommission = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { reason } = req.body;
        
        const [commission] = await query(
            'SELECT * FROM tbl_referral_commission WHERE id = ?',
            [id]
        );
        
        if (!commission) {
            return res.status(404).json({
                success: false,
                message: 'Commission not found'
            });
        }
        
        if (commission.payment_status === 'paid') {
            return res.status(400).json({
                success: false,
                message: 'Cannot cancel paid commission'
            });
        }
        
        await query(
            `UPDATE tbl_referral_commission 
            SET payment_status = 'cancelled',
                notes = CONCAT(COALESCE(notes, ''), '\nCancelled: ', ?)
            WHERE id = ?`,
            [reason || 'No reason provided', id]
        );
        
        // Update referring doctor totals (subtract the commission)
        await query(
            `UPDATE tbl_referring_doctor 
            SET total_referrals = total_referrals - 1,
                total_revenue = total_revenue - ?,
                total_commission_earned = total_commission_earned - ?
            WHERE id = ?`,
            [commission.transaction_amount, commission.commission_amount, commission.referring_doctor_id]
        );
        
        const [updated] = await query(
            'SELECT * FROM tbl_referral_commission WHERE id = ?',
            [id]
        );
        
        res.json({
            success: true,
            message: 'Commission cancelled successfully',
            data: updated
        });
    } catch (error) {
        next(error);
    }
};

// Get commission statistics
exports.getCommissionStats = async (req, res, next) => {
    try {
        const { 
            start_date, 
            end_date, 
            referring_doctor_id, 
            payment_status, 
            transaction_type 
        } = req.query;
        
        let whereClause = ' WHERE 1=1';
        const params = [];
        
        if (start_date && end_date) {
            // Use DATE() to ensure proper date comparison
            whereClause += ' AND DATE(transaction_date) BETWEEN DATE(?) AND DATE(?)';
            params.push(start_date, end_date);
        } else if (start_date) {
            whereClause += ' AND DATE(transaction_date) >= DATE(?)';
            params.push(start_date);
        } else if (end_date) {
            whereClause += ' AND DATE(transaction_date) <= DATE(?)';
            params.push(end_date);
        }
        
        if (referring_doctor_id) {
            whereClause += ' AND referring_doctor_id = ?';
            params.push(referring_doctor_id);
        }
        
        if (payment_status) {
            whereClause += ' AND payment_status = ?';
            params.push(payment_status);
        }
        
        if (transaction_type) {
            whereClause += ' AND transaction_type = ?';
            params.push(transaction_type);
        }
        
        console.log('Getting commission stats with SQL:', whereClause, 'Params:', params);
        
        const statsResult = await query(
            `SELECT 
                COUNT(*) as total_commissions,
                COALESCE(SUM(transaction_amount), 0) as total_revenue,
                COALESCE(SUM(commission_amount), 0) as total_commission_amount,
                COALESCE(SUM(CASE WHEN payment_status = 'pending' THEN commission_amount ELSE 0 END), 0) as pending_amount,
                COALESCE(SUM(CASE WHEN payment_status = 'approved' THEN commission_amount ELSE 0 END), 0) as approved_amount,
                COALESCE(SUM(CASE WHEN payment_status = 'paid' THEN commission_amount ELSE 0 END), 0) as paid_amount,
                COUNT(DISTINCT referring_doctor_id) as active_referrers
            FROM tbl_referral_commission${whereClause}`,
            params
        );
        
        const stats = statsResult[0] || {};
        console.log('Commission stats result:', stats);
        
        res.json({
            success: true,
            data: stats
        });
    } catch (error) {
        console.error('Error getting commission stats:', error);
        next(error);
    }
};

