const { query, transaction } = require('../config/database');

/**
 * @desc    Get all deposit transactions
 * @route   GET /api/deposits
 * @access  Private
 */
exports.getDeposits = async (req, res, next) => {
    try {
        const limit = parseInt(req.query.limit) || 100;
        const offset = parseInt(req.query.offset) || 0;
        const { patient_id, transaction_type } = req.query;
        
        let sql = `
            SELECT 
                dt.*,
                p.name as patient_name,
                p.patient_id as patient_number,
                u.name as created_by_name
            FROM tbl_deposit_transaction dt
            LEFT JOIN tbl_patient p ON dt.patient_id = p.id
            LEFT JOIN tbl_user u ON dt.created_by = u.id
            WHERE dt.status = 'active'
        `;
        
        const params = [];
        
        if (patient_id) {
            sql += ' AND dt.patient_id = ?';
            params.push(patient_id);
        }
        
        if (transaction_type) {
            sql += ' AND dt.transaction_type = ?';
            params.push(transaction_type);
        }
        
        sql += ' ORDER BY dt.created_at DESC LIMIT ? OFFSET ?';
        params.push(limit, offset);
        
        const transactions = await query(sql, params);
        
        res.status(200).json({
            success: true,
            count: transactions.length,
            data: transactions
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get single deposit transaction
 * @route   GET /api/deposits/:id
 * @access  Private
 */
exports.getDeposit = async (req, res, next) => {
    try {
        const sql = `
            SELECT 
                dt.*,
                p.name as patient_name,
                p.patient_id as patient_number,
                u.name as created_by_name
            FROM tbl_deposit_transaction dt
            LEFT JOIN tbl_patient p ON dt.patient_id = p.id
            LEFT JOIN tbl_user u ON dt.created_by = u.id
            WHERE dt.id = ? AND dt.status = 'active'
        `;
        
        const transactions = await query(sql, [req.params.id]);
        
        if (transactions.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Deposit transaction not found'
            });
        }
        
        res.status(200).json({
            success: true,
            data: transactions[0]
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get patient deposit balance and history
 * @route   GET /api/deposits/patient/:patient_id
 * @access  Private
 */
exports.getPatientDeposits = async (req, res, next) => {
    try {
        const patientId = req.params.patient_id;
        
        // Get patient info
        const patientSql = 'SELECT id, name, patient_id, deposit_balance FROM tbl_patient WHERE id = ?';
        const patients = await query(patientSql, [patientId]);
        
        if (patients.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Patient not found'
            });
        }
        
        // Get deposit transactions
        const transactionsSql = `
            SELECT 
                dt.*,
                u.name as created_by_name
            FROM tbl_deposit_transaction dt
            LEFT JOIN tbl_user u ON dt.created_by = u.id
            WHERE dt.patient_id = ? AND dt.status = 'active'
            ORDER BY dt.created_at DESC
        `;
        
        const transactions = await query(transactionsSql, [patientId]);
        
        // Calculate totals
        const totals = transactions.reduce((acc, transaction) => {
            if (transaction.transaction_type === 'deposit' || transaction.transaction_type === 'refund') {
                acc.totalDeposits += parseFloat(transaction.amount);
            } else if (transaction.transaction_type === 'withdrawal') {
                acc.totalWithdrawals += parseFloat(transaction.amount);
            }
            return acc;
        }, { totalDeposits: 0, totalWithdrawals: 0 });
        
        res.status(200).json({
            success: true,
            data: {
                patient: patients[0],
                transactions,
                totals,
                currentBalance: parseFloat(patients[0].deposit_balance || 0)
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create new deposit transaction
 * @route   POST /api/deposits
 * @access  Private
 */
exports.createDeposit = async (req, res, next) => {
    try {
        const { 
            patient_id, 
            transaction_type, 
            amount, 
            description, 
            reference_no 
        } = req.body;
        
        const created_by = req.user.id;
        
        // Validate transaction type
        const validTypes = ['deposit', 'withdrawal', 'refund', 'adjustment'];
        if (!validTypes.includes(transaction_type)) {
            throw new Error('Invalid transaction type');
        }
        
        const result = await transaction(async (connection) => {
            // Get current patient balance
            const patientSql = 'SELECT deposit_balance FROM tbl_patient WHERE id = ?';
            const [patients] = await connection.execute(patientSql, [patient_id]);
            
            if (patients.length === 0) {
                throw new Error('Patient not found');
            }
            
            const currentBalance = parseFloat(patients[0].deposit_balance || 0);
            const transactionAmount = parseFloat(amount);
            
            // Calculate new balance
            let newBalance;
            if (transaction_type === 'deposit' || transaction_type === 'refund') {
                newBalance = currentBalance + transactionAmount;
            } else if (transaction_type === 'withdrawal') {
                if (currentBalance < transactionAmount) {
                    throw new Error('Insufficient deposit balance');
                }
                newBalance = currentBalance - transactionAmount;
            } else if (transaction_type === 'adjustment') {
                newBalance = transactionAmount; // For adjustments, amount is the new balance
            }
            
            // Create transaction record
            const transactionSql = `
                INSERT INTO tbl_deposit_transaction 
                (patient_id, transaction_type, amount, balance_after, description, reference_no, created_by)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            `;
            
            const [transactionResult] = await connection.execute(transactionSql, [
                patient_id,
                transaction_type,
                transactionAmount,
                newBalance,
                description,
                reference_no,
                created_by
            ]);
            
            // Update patient balance
            const updatePatientSql = 'UPDATE tbl_patient SET deposit_balance = ? WHERE id = ?';
            await connection.execute(updatePatientSql, [newBalance, patient_id]);
            
            return {
                id: transactionResult.insertId,
                patient_id,
                transaction_type,
                amount: transactionAmount,
                balance_after: newBalance,
                description,
                reference_no
            };
        });
        
        res.status(201).json({
            success: true,
            message: 'Deposit transaction created successfully',
            data: result
        });
        
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update deposit transaction
 * @route   PUT /api/deposits/:id
 * @access  Private
 */
exports.updateDeposit = async (req, res, next) => {
    try {
        const { description, reference_no } = req.body;
        
        const sql = `
            UPDATE tbl_deposit_transaction 
            SET description = ?, reference_no = ?, updated_at = CURRENT_TIMESTAMP
            WHERE id = ? AND status = 'active'
        `;
        
        const result = await query(sql, [description, reference_no, req.params.id]);
        
        if (result.affectedRows === 0) {
            return res.status(404).json({
                success: false,
                message: 'Deposit transaction not found'
            });
        }
        
        res.status(200).json({
            success: true,
            message: 'Deposit transaction updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Cancel deposit transaction
 * @route   DELETE /api/deposits/:id
 * @access  Private
 */
exports.cancelDeposit = async (req, res, next) => {
    try {
        await transaction(async (connection) => {
            // Get transaction details
            const transactionSql = `
                SELECT * FROM tbl_deposit_transaction 
                WHERE id = ? AND status = 'active'
            `;
            const [transactions] = await connection.execute(transactionSql, [req.params.id]);
            
            if (transactions.length === 0) {
                throw new Error('Transaction not found or already cancelled');
            }
            
            const transaction = transactions[0];
            
            // Get current patient balance
            const patientSql = 'SELECT deposit_balance FROM tbl_patient WHERE id = ?';
            const [patients] = await connection.execute(patientSql, [transaction.patient_id]);
            
            if (patients.length === 0) {
                throw new Error('Patient not found');
            }
            
            const currentBalance = parseFloat(patients[0].deposit_balance || 0);
            
            // Calculate reverse balance
            let reverseBalance;
            if (transaction.transaction_type === 'deposit' || transaction.transaction_type === 'refund') {
                reverseBalance = currentBalance - parseFloat(transaction.amount);
            } else if (transaction.transaction_type === 'withdrawal') {
                reverseBalance = currentBalance + parseFloat(transaction.amount);
            } else if (transaction.transaction_type === 'adjustment') {
                // For adjustments, we need to know the previous balance
                // This is complex, so we'll just set to 0 for now
                reverseBalance = 0;
            }
            
            // Mark transaction as cancelled
            const cancelSql = 'UPDATE tbl_deposit_transaction SET status = "cancelled" WHERE id = ?';
            await connection.execute(cancelSql, [req.params.id]);
            
            // Update patient balance
            const updatePatientSql = 'UPDATE tbl_patient SET deposit_balance = ? WHERE id = ?';
            await connection.execute(updatePatientSql, [reverseBalance, transaction.patient_id]);
        });
        
        res.status(200).json({
            success: true,
            message: 'Deposit transaction cancelled successfully'
        });
        
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get deposit statistics
 * @route   GET /api/deposits/stats
 * @access  Private
 */
exports.getDepositStats = async (req, res, next) => {
    try {
        const { period = '30' } = req.query; // days
        
        // Get total deposits, withdrawals, and current balance
        const statsSql = `
            SELECT 
                COUNT(*) as total_transactions,
                SUM(CASE WHEN transaction_type IN ('deposit', 'refund') THEN amount ELSE 0 END) as total_deposits,
                SUM(CASE WHEN transaction_type = 'withdrawal' THEN amount ELSE 0 END) as total_withdrawals,
                (SELECT SUM(deposit_balance) FROM tbl_patient) as total_balance
            FROM tbl_deposit_transaction 
            WHERE status = 'active' 
            AND created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
        `;
        
        const stats = await query(statsSql, [parseInt(period)]);
        
        // Get recent transactions
        const recentSql = `
            SELECT 
                dt.*,
                p.name as patient_name,
                p.patient_id as patient_number
            FROM tbl_deposit_transaction dt
            LEFT JOIN tbl_patient p ON dt.patient_id = p.id
            WHERE dt.status = 'active'
            ORDER BY dt.created_at DESC
            LIMIT 10
        `;
        
        const recentTransactions = await query(recentSql);
        
        res.status(200).json({
            success: true,
            data: {
                stats: stats[0],
                recent_transactions: recentTransactions
            }
        });
    } catch (error) {
        next(error);
    }
};
