/**
 * Discharged Patient Report Controller
 * Generates comprehensive reports for discharged patients
 */

const { query } = require('../config/database');

/**
 * @desc    Get comprehensive discharged patient report
 * @route   GET /api/reports/discharged-patient/:patientId
 * @access  Private
 */
exports.getDischargedPatientReport = async (req, res, next) => {
    try {
        const { patientId } = req.params;
        
        if (!patientId) {
            return res.status(400).json({
                success: false,
                message: 'Patient ID is required'
            });
        }

        // Get patient basic information from tbl_inpatient directly
        const patientSql = `
            SELECT 
                i.*,
                i.id as inpatient_id,
                i.start_date,
                i.end_date,
                i.discharge_date,
                i.final_bill,
                i.discharge_notes,
                i.dr_id,
                d.name as doctor_name,
                b.bed_number,
                b.room_number
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            WHERE i.id = ? AND i.end_date IS NOT NULL
            ORDER BY i.end_date DESC
            LIMIT 1
        `;
        
        const patientResult = await query(patientSql, [patientId]);
        
        if (patientResult.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Discharged patient not found'
            });
        }

        const patient = patientResult[0];
        const inpatientId = patient.inpatient_id;

        // Check if settled table exists first
        let settledTableExists = false;
        try {
            await query('SELECT 1 FROM tbl_inpatient_usage_settled LIMIT 1');
            settledTableExists = true;
        } catch (error) {
            if (error.code === 'ER_NO_SUCH_TABLE') {
                settledTableExists = false;
            } else {
                throw error;
            }
        }
        
        // Get medication usage from active table (and archived if available)
        let medicationSql = `
            SELECT 
                u.id,
                u.inpatient_id,
                u.item_type,
                u.item_id,
                u.quantity,
                u.price,
                u.notes,
                u.usage_date,
                u.created_at,
                u.created_by,
                COALESCE(u.is_settled, 0) as is_settled,
                NULL as settled_at,
                NULL as settlement_deposit_id,
                CASE 
                    WHEN u.item_type = 'product' THEN p.name
                    WHEN u.item_type = 'service' THEN s.service_name
                END as product_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.unit_name
                    WHEN u.item_type = 'service' THEN 'service'
                END as unit_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.cat_id
                    WHEN u.item_type = 'service' THEN s.cat_id
                END as category,
                CASE 
                    WHEN u.item_type = 'product' THEN p.expire_date
                    ELSE NULL
                END as expire_date,
                'current' as source_table
            FROM tbl_inpatient_usage u
            LEFT JOIN tbl_stock p ON u.item_type = 'product' AND u.item_id = p.id
            LEFT JOIN tbl_service s ON u.item_type = 'service' AND u.item_id = s.id
            WHERE u.inpatient_id = ?
        `;
        
        const medicationParams = [inpatientId];
        
        // Add UNION ALL for settled table if it exists
        if (settledTableExists) {
            medicationSql += `
            UNION ALL
            
            SELECT 
                us.original_usage_id as id,
                us.inpatient_id,
                us.item_type,
                us.item_id,
                us.quantity,
                us.price,
                us.notes,
                us.usage_date,
                us.created_at,
                NULL as created_by,
                1 as is_settled,
                us.created_at as settled_at,
                us.settlement_deposit_id,
                us.product_name,
                CASE 
                    WHEN us.item_type = 'product' THEN p.unit_name
                    WHEN us.item_type = 'service' THEN 'service'
                END as unit_name,
                CASE 
                    WHEN us.item_type = 'product' THEN p.cat_id
                    WHEN us.item_type = 'service' THEN s.cat_id
                END as category,
                CASE 
                    WHEN us.item_type = 'product' THEN p.expire_date
                    ELSE NULL
                END as expire_date,
                'archived' as source_table
            FROM tbl_inpatient_usage_settled us
            LEFT JOIN tbl_stock p ON us.item_type = 'product' AND us.item_id = p.id
            LEFT JOIN tbl_service s ON us.item_type = 'service' AND us.item_id = s.id
            WHERE us.inpatient_id = ?
            `;
            medicationParams.push(inpatientId);
        }
        
        medicationSql += ` ORDER BY usage_date DESC, created_at DESC`;
        
        const medications = await query(medicationSql, medicationParams);

        // Get services used from active table (and archived if available)
        let servicesSql = `
            SELECT 
                u.id,
                u.inpatient_id,
                u.item_type,
                u.item_id,
                u.quantity,
                u.price,
                u.notes,
                u.usage_date,
                u.created_at,
                u.created_by,
                COALESCE(u.is_settled, 0) as is_settled,
                NULL as settled_at,
                NULL as settlement_deposit_id,
                CASE 
                    WHEN u.item_type = 'product' THEN p.name
                    WHEN u.item_type = 'service' THEN s.service_name
                END as service_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.cat_id
                    WHEN u.item_type = 'service' THEN s.cat_id
                END as category,
                'current' as source_table
            FROM tbl_inpatient_usage u
            LEFT JOIN tbl_stock p ON u.item_type = 'product' AND u.item_id = p.id
            LEFT JOIN tbl_service s ON u.item_type = 'service' AND u.item_id = s.id
            WHERE u.inpatient_id = ? AND u.item_type = 'service'
        `;
        
        const servicesParams = [inpatientId];
        
        // Add UNION ALL for settled table if it exists
        if (settledTableExists) {
            servicesSql += `
            UNION ALL
            
            SELECT 
                us.original_usage_id as id,
                us.inpatient_id,
                us.item_type,
                us.item_id,
                us.quantity,
                us.price,
                us.notes,
                us.usage_date,
                us.created_at,
                NULL as created_by,
                1 as is_settled,
                us.created_at as settled_at,
                us.settlement_deposit_id,
                us.product_name as service_name,
                CASE 
                    WHEN us.item_type = 'product' THEN p.cat_id
                    WHEN us.item_type = 'service' THEN s.cat_id
                END as category,
                'archived' as source_table
            FROM tbl_inpatient_usage_settled us
            LEFT JOIN tbl_stock p ON us.item_type = 'product' AND us.item_id = p.id
            LEFT JOIN tbl_service s ON us.item_type = 'service' AND us.item_id = s.id
            WHERE us.inpatient_id = ? AND us.item_type = 'service'
            `;
            servicesParams.push(inpatientId);
        }
        
        servicesSql += ` ORDER BY usage_date DESC, created_at DESC`;
        
        const services = await query(servicesSql, servicesParams);

        // Get stock/equipment usage from active table (and archived if available)
        let stockSql = `
            SELECT 
                u.id,
                u.inpatient_id,
                u.item_type,
                u.item_id,
                u.quantity,
                u.price,
                u.notes,
                u.usage_date,
                u.created_at,
                u.created_by,
                COALESCE(u.is_settled, 0) as is_settled,
                NULL as settled_at,
                NULL as settlement_deposit_id,
                CASE 
                    WHEN u.item_type = 'product' THEN p.name
                    WHEN u.item_type = 'service' THEN s.service_name
                END as product_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.unit_name
                    WHEN u.item_type = 'service' THEN 'service'
                END as unit_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.cat_id
                    WHEN u.item_type = 'service' THEN s.cat_id
                END as category,
                CASE 
                    WHEN u.item_type = 'product' THEN p.expire_date
                    ELSE NULL
                END as expire_date,
                'current' as source_table
            FROM tbl_inpatient_usage u
            LEFT JOIN tbl_stock p ON u.item_type = 'product' AND u.item_id = p.id
            LEFT JOIN tbl_service s ON u.item_type = 'service' AND u.item_id = s.id
            WHERE u.inpatient_id = ? AND u.item_type = 'product'
        `;
        
        const stockParams = [inpatientId];
        
        // Add UNION ALL for settled table if it exists
        if (settledTableExists) {
            stockSql += `
            UNION ALL
            
            SELECT 
                us.original_usage_id as id,
                us.inpatient_id,
                us.item_type,
                us.item_id,
                us.quantity,
                us.price,
                us.notes,
                us.usage_date,
                us.created_at,
                NULL as created_by,
                1 as is_settled,
                us.created_at as settled_at,
                us.settlement_deposit_id,
                us.product_name,
                CASE 
                    WHEN us.item_type = 'product' THEN p.unit_name
                    WHEN us.item_type = 'service' THEN 'service'
                END as unit_name,
                CASE 
                    WHEN us.item_type = 'product' THEN p.cat_id
                    WHEN us.item_type = 'service' THEN s.cat_id
                END as category,
                CASE 
                    WHEN us.item_type = 'product' THEN p.expire_date
                    ELSE NULL
                END as expire_date,
                'archived' as source_table
            FROM tbl_inpatient_usage_settled us
            LEFT JOIN tbl_stock p ON us.item_type = 'product' AND us.item_id = p.id
            LEFT JOIN tbl_service s ON us.item_type = 'service' AND us.item_id = s.id
            WHERE us.inpatient_id = ? AND us.item_type = 'product'
            `;
            stockParams.push(inpatientId);
        }
        
        stockSql += ` ORDER BY usage_date DESC, created_at DESC`;
        
        const stockUsage = await query(stockSql, stockParams);

        // Get laboratory results (if table exists)
        let labResults = [];
        try {
            const labResultsSql = `
                SELECT 
                    lr.*,
                    lt.name as test_name,
                    lt.normal_range,
                    lt.unit,
                    lt.category
                FROM tbl_lab_reports lr
                LEFT JOIN tbl_lab_tests lt ON lr.test_id = lt.id
                WHERE lr.patient_id = ?
                ORDER BY lr.reported_at DESC
            `;
            labResults = await query(labResultsSql, [patientId]);
        } catch (error) {
            // Suppress expected table not found errors
            if (error.code === 'ER_NO_SUCH_TABLE') {
                console.log('Lab reports table not found, skipping...');
            } else {
                console.error('Unexpected error in lab results query:', error);
            }
            labResults = [];
        }

        // Get EMR records (if table exists)
        let emrRecords = [];
        try {
            const emrSql = `
                SELECT 
                    e.*,
                    d.name as doctor_name
                FROM tbl_emr e
                LEFT JOIN tbl_doctor d ON e.doctor_id = d.id
                WHERE e.patient_id = ?
                ORDER BY e.created_at DESC
            `;
            emrRecords = await query(emrSql, [patientId]);
        } catch (error) {
            // Suppress expected table not found errors
            if (error.code === 'ER_NO_SUCH_TABLE') {
                console.log('EMR table not found, skipping...');
            } else {
                console.error('Unexpected error in EMR query:', error);
            }
            emrRecords = [];
        }

        // Get deposits and payments (if table exists)
        let deposits = [];
        try {
            const depositsSql = `
                SELECT *
                FROM tbl_inpatient_deposits
                WHERE inpatient_id = ?
                ORDER BY created_at DESC
            `;
            deposits = await query(depositsSql, [inpatientId]);
        } catch (error) {
            // Suppress expected table not found errors
            if (error.code === 'ER_NO_SUCH_TABLE') {
                console.log('Deposits table not found, skipping...');
            } else {
                console.error('Unexpected error in deposits query:', error);
            }
            deposits = [];
        }

        // Get discharge summary
        const dischargeSummarySql = `
            SELECT *
            FROM tbl_inpatient_discharge
            WHERE inpatient_id = ?
            ORDER BY created_at DESC
            LIMIT 1
        `;
        
        const dischargeSummary = await query(dischargeSummarySql, [inpatientId]);

        // Calculate financial summary
        const totalUsage = medications.concat(services, stockUsage).reduce((sum, item) => {
            return sum + (parseFloat(item.quantity) * parseFloat(item.price));
        }, 0);

        const totalDeposits = deposits.reduce((sum, deposit) => {
            return sum + parseFloat(deposit.amount || 0);
        }, 0);

        const dailyRate = parseFloat(patient.daily_rate || 0);
        const startDate = new Date(patient.start_date);
        const endDate = new Date(patient.end_date);
        const daysStayed = Math.max(1, Math.ceil((endDate - startDate) / (1000 * 60 * 60 * 24)));
        const bedCharges = dailyRate * daysStayed;

        const totalCharges = totalUsage + bedCharges;
        const outstandingBalance = totalCharges - totalDeposits;

        // Group usage by date
        const allUsage = medications.concat(services, stockUsage);
        const usageByDate = {};
        allUsage.forEach(item => {
            const date = item.usage_date;
            if (!usageByDate[date]) {
                usageByDate[date] = [];
            }
            usageByDate[date].push(item);
        });

        // Calculate daily totals
        const dailyTotals = {};
        Object.keys(usageByDate).forEach(date => {
            dailyTotals[date] = usageByDate[date].reduce((sum, item) => {
                return sum + (parseFloat(item.quantity) * parseFloat(item.price));
            }, 0);
        });

        const report = {
            patient: {
                id: patient.id,
                patient_id: patient.patient_id,
                name: patient.name,
                age: patient.age,
                phone: patient.phone,
                address: patient.address,
                case_: patient.case_,
                remark: patient.remark,
                dr_name: patient.dr_name,
                room_no: patient.room_no,
                deposit: patient.deposit
            },
            admission: {
                inpatient_id: inpatientId,
                start_date: patient.start_date,
                end_date: patient.end_date,
                discharge_date: patient.discharge_date,
                days_stayed: daysStayed,
                doctor: {
                    id: patient.dr_id,
                    name: patient.doctor_name
                },
                bed: {
                    number: patient.bed_number,
                    room: patient.room_number
                }
            },
            medications: medications,
            services: services,
            stock_usage: stockUsage,
            lab_results: labResults,
            emr_records: emrRecords,
            deposits: deposits,
            discharge_summary: dischargeSummary[0] || null,
            financial: {
                total_usage: totalUsage,
                bed_charges: bedCharges,
                total_charges: totalCharges,
                total_deposits: totalDeposits,
                outstanding_balance: outstandingBalance,
                daily_rate: dailyRate
            },
            usage_by_date: usageByDate,
            daily_totals: dailyTotals,
            generated_at: new Date().toISOString()
        };

        res.json({
            success: true,
            data: report
        });

    } catch (error) {
        console.error('Error generating discharged patient report:', error);
        next(error);
    }
};

/**
 * @desc    Get list of discharged patients
 * @route   GET /api/reports/discharged-patients
 * @access  Private
 */
exports.getDischargedPatients = async (req, res, next) => {
    try {
        const { page = 1, limit = 50, search = '', start_date, end_date } = req.query;
        const offset = (page - 1) * limit;

        let sql = `
            SELECT 
                i.id,
                i.patient_id,
                i.name,
                i.age,
                i.phone,
                i.start_date,
                i.end_date,
                i.discharge_date,
                i.final_bill,
                d.name as doctor_name,
                b.bed_number,
                b.room_number
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            WHERE i.end_date IS NOT NULL
        `;

        const params = [];

        if (search) {
            sql += ` AND (i.name LIKE ? OR i.patient_id LIKE ? OR i.phone LIKE ?)`;
            const searchTerm = `%${search}%`;
            params.push(searchTerm, searchTerm, searchTerm);
        }

        // Date filtering by discharge date
        if (start_date && end_date) {
            sql += ` AND DATE(i.end_date) BETWEEN DATE(?) AND DATE(?)`;
            params.push(start_date, end_date);
        } else if (start_date) {
            sql += ` AND DATE(i.end_date) >= DATE(?)`;
            params.push(start_date);
        } else if (end_date) {
            sql += ` AND DATE(i.end_date) <= DATE(?)`;
            params.push(end_date);
        }

        sql += ` ORDER BY i.end_date DESC LIMIT ? OFFSET ?`;
        params.push(parseInt(limit), parseInt(offset));

        const patients = await query(sql, params);

        // Get total count
        let countSql = `
            SELECT COUNT(*) as total
            FROM tbl_inpatient i
            WHERE i.end_date IS NOT NULL
        `;

        const countParams = [];
        if (search) {
            countSql += ` AND (i.name LIKE ? OR i.patient_id LIKE ? OR i.phone LIKE ?)`;
            const searchTerm = `%${search}%`;
            countParams.push(searchTerm, searchTerm, searchTerm);
        }

        // Date filtering for count
        if (start_date && end_date) {
            countSql += ` AND DATE(i.end_date) BETWEEN DATE(?) AND DATE(?)`;
            countParams.push(start_date, end_date);
        } else if (start_date) {
            countSql += ` AND DATE(i.end_date) >= DATE(?)`;
            countParams.push(start_date);
        } else if (end_date) {
            countSql += ` AND DATE(i.end_date) <= DATE(?)`;
            countParams.push(end_date);
        }

        const countResult = await query(countSql, countParams);
        const total = countResult[0].total;

        res.json({
            success: true,
            data: {
                patients,
                pagination: {
                    current_page: parseInt(page),
                    per_page: parseInt(limit),
                    total,
                    total_pages: Math.ceil(total / limit)
                }
            }
        });

    } catch (error) {
        console.error('Error getting discharged patients:', error);
        next(error);
    }
};
