const { query } = require('../config/database');
const path = require('path');
const fs = require('fs').promises;

exports.getDoctors = async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_doctor ORDER BY name';
        const doctors = await query(sql);

        res.status(200).json({
            success: true,
            count: doctors.length,
            data: doctors
        });
    } catch (error) {
        next(error);
    }
};

exports.getDoctor = async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_doctor WHERE id = ?';
        const doctors = await query(sql, [req.params.id]);

        if (doctors.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Doctor not found'
            });
        }

        res.status(200).json({
            success: true,
            data: doctors[0]
        });
    } catch (error) {
        next(error);
    }
};

exports.createDoctor = async (req, res, next) => {
    try {
        const { name, phone, education, specialist, remark } = req.body;
        const photo = req.file ? `/uploads/${req.file.filename}` : null;

        const sql = `INSERT INTO tbl_doctor (name, phone, education, specialist, photo, remark) 
                     VALUES (?, ?, ?, ?, ?, ?)`;

        const result = await query(sql, [name, phone, education, specialist, photo, remark]);

        res.status(201).json({
            success: true,
            message: 'Doctor created successfully',
            data: { id: result.insertId, name, photo }
        });
    } catch (error) {
        next(error);
    }
};

exports.updateDoctor = async (req, res, next) => {
    try {
        const { name, phone, education, specialist, remark } = req.body;
        const doctorId = req.params.id;

        // Check if new photo is uploaded
        if (req.file) {
            const photo = `/uploads/${req.file.filename}`;
            
            // Get old photo to delete it
            const oldDoctor = await query('SELECT photo FROM tbl_doctor WHERE id = ?', [doctorId]);
            if (oldDoctor.length > 0 && oldDoctor[0].photo) {
                const oldPhotoPath = path.join(__dirname, '../../', oldDoctor[0].photo);
                try {
                    await fs.unlink(oldPhotoPath);
                } catch (err) {
                    console.log('Error deleting old photo:', err.message);
                }
            }

            const sql = `UPDATE tbl_doctor 
                         SET name = ?, phone = ?, education = ?, specialist = ?, photo = ?, remark = ?
                         WHERE id = ?`;

            await query(sql, [name, phone, education, specialist, photo, remark, doctorId]);
        } else {
            const sql = `UPDATE tbl_doctor 
                         SET name = ?, phone = ?, education = ?, specialist = ?, remark = ?
                         WHERE id = ?`;

            await query(sql, [name, phone, education, specialist, remark, doctorId]);
        }

        res.status(200).json({
            success: true,
            message: 'Doctor updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

exports.deleteDoctor = async (req, res, next) => {
    try {
        const doctorId = req.params.id;

        // Get doctor photo to delete it
        const doctor = await query('SELECT photo FROM tbl_doctor WHERE id = ?', [doctorId]);
        if (doctor.length > 0 && doctor[0].photo) {
            const photoPath = path.join(__dirname, '../../', doctor[0].photo);
            try {
                await fs.unlink(photoPath);
            } catch (err) {
                console.log('Error deleting photo:', err.message);
            }
        }

        const sql = 'DELETE FROM tbl_doctor WHERE id = ?';
        await query(sql, [doctorId]);

        res.status(200).json({
            success: true,
            message: 'Doctor deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

