/**
 * Electronic Medical Records (EMR) Controller
 * Handles patient treatment history with multiple images
 */

const { query, transaction } = require('../config/database');
const path = require('path');
const fs = require('fs');

/**
 * @desc    Get patient treatment history
 * @route   GET /api/emr/patient/:patientId/history
 * @access  Private
 */
exports.getPatientHistory = async (req, res, next) => {
    try {
        const { patientId } = req.params;

        // Get patient info
        const patientSql = 'SELECT * FROM tbl_patient WHERE patient_id = ?';
        const patients = await query(patientSql, [patientId]);

        if (patients.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Patient not found'
            });
        }

        // Get treatment history
        const treatmentSql = `SELECT t.*, d.name as doctor_name 
                             FROM tbl_treatment t
                             LEFT JOIN tbl_doctor d ON t.doctor_id = d.id
                             WHERE t.patient_id = ?
                             ORDER BY t.Date DESC`;
        const treatments = await query(treatmentSql, [patientId]);

        // Get images for each treatment
        for (let treatment of treatments) {
            const imagesSql = 'SELECT * FROM tbl_images WHERE patient_id = ? AND treatment_id = ?';
            treatment.images = await query(imagesSql, [patientId, treatment.id]);
        }

        res.json({
            success: true,
            patient: patients[0],
            treatments: treatments
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get single treatment record
 * @route   GET /api/emr/treatment/:id
 * @access  Private
 */
exports.getTreatment = async (req, res, next) => {
    try {
        const sql = `SELECT t.*, d.name as doctor_name, p.name as patient_name
                     FROM tbl_treatment t
                     LEFT JOIN tbl_doctor d ON t.doctor_id = d.id
                     LEFT JOIN tbl_patient p ON t.patient_id = p.patient_id
                     WHERE t.id = ?`;
        
        const treatments = await query(sql, [req.params.id]);

        if (treatments.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Treatment record not found'
            });
        }

        const treatment = treatments[0];

        // Get images
        const imagesSql = 'SELECT * FROM tbl_images WHERE treatment_id = ?';
        treatment.images = await query(imagesSql, [treatment.id]);

        res.json({
            success: true,
            data: treatment
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create treatment record with images
 * @route   POST /api/emr/treatment
 * @access  Private
 */
exports.createTreatment = async (req, res, next) => {
    try {
        const {
            patient_id,
            doctor_id,
            chief_complaint,
            present_illness,
            past_diagnoses,
            surgeries_procedures,
            hospitalizations,
            current_medications,
            dosages,
            allergen,
            treatment_type,
            diagnosis,
            prescription,
            recommendation,
            voucher_id,
            vital_signs,
            notes
        } = req.body;

        const result = await transaction(async (conn) => {
            // Insert treatment record
            const treatmentSql = `INSERT INTO tbl_treatment 
                                 (patient_id, doctor_id, chief_complaint, present_illness, 
                                  Past_Diagnoses, Surgeries_Procedures, Hospitalizations, 
                                  Current_Medications, Dosages, Allergen, Treatment_Type, 
                                  diagnosis, prescription, Recommendation, Voucher_id, 
                                  vital_signs, notes, Date)
                                 VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())`;

            const [treatmentResult] = await conn.query(treatmentSql, [
                patient_id,
                doctor_id || null,
                chief_complaint || null,
                present_illness || null,
                past_diagnoses || null,
                surgeries_procedures || null,
                hospitalizations || null,
                current_medications || null,
                dosages || null,
                allergen || null,
                treatment_type || 'Consultation',
                diagnosis || null,
                prescription || null,
                recommendation || null,
                voucher_id || null,
                vital_signs || null,
                notes || null
            ]);

            const treatmentId = treatmentResult.insertId;

            // Insert images if uploaded
            if (req.files && req.files.length > 0) {
                for (const file of req.files) {
                    const imagePath = `/uploads/medical/${file.filename}`;
                    await conn.query(
                        'INSERT INTO tbl_images (patient_id, treatment_id, image, image_path, upload_date) VALUES (?, ?, NULL, ?, NOW())',
                        [patient_id, treatmentId, imagePath]
                    );
                }
            }

            return { treatmentId };
        });

        res.status(201).json({
            success: true,
            message: 'Treatment record created successfully',
            data: result
        });

    } catch (error) {
        // Delete uploaded files if transaction fails
        if (req.files) {
            req.files.forEach(file => {
                const filePath = path.join(__dirname, '../uploads/medical', file.filename);
                if (fs.existsSync(filePath)) {
                    fs.unlinkSync(filePath);
                }
            });
        }
        next(error);
    }
};

/**
 * @desc    Update treatment record
 * @route   PUT /api/emr/treatment/:id
 * @access  Private
 */
exports.updateTreatment = async (req, res, next) => {
    try {
        const {
            doctor_id,
            chief_complaint,
            present_illness,
            past_diagnoses,
            surgeries_procedures,
            hospitalizations,
            current_medications,
            dosages,
            allergen,
            treatment_type,
            diagnosis,
            prescription,
            recommendation,
            vital_signs,
            notes
        } = req.body;

        const result = await transaction(async (conn) => {
            // Update treatment record
            const updateSql = `UPDATE tbl_treatment 
                              SET doctor_id = ?, chief_complaint = ?, present_illness = ?,
                                  Past_Diagnoses = ?, Surgeries_Procedures = ?, Hospitalizations = ?,
                                  Current_Medications = ?, Dosages = ?, Allergen = ?, Treatment_Type = ?,
                                  diagnosis = ?, prescription = ?, Recommendation = ?,
                                  vital_signs = ?, notes = ?
                              WHERE id = ?`;

            await conn.query(updateSql, [
                doctor_id,
                chief_complaint,
                present_illness,
                past_diagnoses,
                surgeries_procedures,
                hospitalizations,
                current_medications,
                dosages,
                allergen,
                treatment_type,
                diagnosis,
                prescription,
                recommendation,
                vital_signs,
                notes,
                req.params.id
            ]);

            // Add new images if uploaded
            if (req.files && req.files.length > 0) {
                const [treatments] = await conn.query('SELECT patient_id FROM tbl_treatment WHERE id = ?', [req.params.id]);
                
                if (treatments.length > 0) {
                    const patientId = treatments[0].patient_id;
                    
                    for (const file of req.files) {
                        const imagePath = `/uploads/medical/${file.filename}`;
                        await conn.query(
                            'INSERT INTO tbl_images (patient_id, treatment_id, image, image_path, upload_date) VALUES (?, ?, NULL, ?, NOW())',
                            [patientId, req.params.id, imagePath]
                        );
                    }
                }
            }

            return { id: req.params.id };
        });

        res.json({
            success: true,
            message: 'Treatment record updated successfully',
            data: result
        });

    } catch (error) {
        if (req.files) {
            req.files.forEach(file => {
                const filePath = path.join(__dirname, '../uploads/medical', file.filename);
                if (fs.existsSync(filePath)) {
                    fs.unlinkSync(filePath);
                }
            });
        }
        next(error);
    }
};

/**
 * @desc    Delete treatment record
 * @route   DELETE /api/emr/treatment/:id
 * @access  Private
 */
exports.deleteTreatment = async (req, res, next) => {
    try {
        await transaction(async (conn) => {
            // Get images to delete
            const [images] = await conn.query('SELECT image_path FROM tbl_images WHERE treatment_id = ?', [req.params.id]);

            // Delete images from filesystem
            images.forEach(img => {
                if (img.image_path) {
                    const imagePath = path.join(__dirname, '..', img.image_path);
                    if (fs.existsSync(imagePath)) {
                        fs.unlinkSync(imagePath);
                    }
                }
            });

            // Delete image records
            await conn.query('DELETE FROM tbl_images WHERE treatment_id = ?', [req.params.id]);

            // Delete treatment record
            await conn.query('DELETE FROM tbl_treatment WHERE id = ?', [req.params.id]);
        });

        res.json({
            success: true,
            message: 'Treatment record deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Delete image from treatment
 * @route   DELETE /api/emr/image/:id
 * @access  Private
 */
exports.deleteImage = async (req, res, next) => {
    try {
        // Get image info
        const imageSql = 'SELECT * FROM tbl_images WHERE id = ?';
        const images = await query(imageSql, [req.params.id]);

        if (images.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Image not found'
            });
        }

        const image = images[0];

        // Delete file from filesystem
        if (image.image_path) {
            const imagePath = path.join(__dirname, '..', image.image_path);
            if (fs.existsSync(imagePath)) {
                fs.unlinkSync(imagePath);
            }
        }

        // Delete from database
        await query('DELETE FROM tbl_images WHERE id = ?', [req.params.id]);

        res.json({
            success: true,
            message: 'Image deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get vital signs history
 * @route   GET /api/emr/patient/:patientId/vitals
 * @access  Private
 */
exports.getVitalsHistory = async (req, res, next) => {
    try {
        const sql = `SELECT vital_signs, Date, doctor_id 
                     FROM tbl_treatment 
                     WHERE patient_id = ? AND vital_signs IS NOT NULL
                     ORDER BY Date DESC
                     LIMIT 20`;
        
        const vitals = await query(sql, [req.params.patientId]);

        res.json({
            success: true,
            data: vitals
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get normalized vitals time series derived from treatment.vital_signs JSON
 * @route   GET /api/emr/patient/:patientId/vitals/chart
 * @access  Private
 */
exports.getVitalsChartFromEMR = async (req, res, next) => {
    try {
        const sql = `
            SELECT Date as date, vital_signs
            FROM tbl_treatment
            WHERE patient_id = ? AND vital_signs IS NOT NULL
            ORDER BY Date ASC
            LIMIT 200`;
        const rows = await query(sql, [req.params.patientId]);

        const series = rows.map(r => {
            let v = {};
            try { v = JSON.parse(r.vital_signs || '{}'); } catch (_) { v = {}; }
            return {
                date: r.date,
                bp_sys: parseInt((v.bp || '').split('/')[0]) || null,
                bp_dia: parseInt((v.bp || '').split('/')[1]) || null,
                heart_rate: v.pulse ? Number(v.pulse) : (v.heart_rate ? Number(v.heart_rate) : null),
                temperature: v.temp ? Number(v.temp) : (v.temperature ? Number(v.temperature) : null),
                respiratory_rate: v.respiratory ? Number(v.respiratory) : (v.respiratory_rate ? Number(v.respiratory_rate) : null),
                spo2: v.spo2 ? Number(v.spo2) : (v.oxygen_saturation ? Number(v.oxygen_saturation) : null),
                weight: v.weight ? Number(v.weight) : null,
                height: v.height ? Number(v.height) : null,
                bmi: v.bmi ? (typeof v.bmi === 'string' ? Number(v.bmi) : Number(v.bmi)) : null,
                glucose: v.glucose ? Number(v.glucose) : (v.blood_glucose ? Number(v.blood_glucose) : null)
            };
        });

        res.json({ success: true, data: series });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Search treatments
 * @route   GET /api/emr/search
 * @access  Private
 */
exports.searchTreatments = async (req, res, next) => {
    try {
        const { q, startDate, endDate, doctorId } = req.query;

        let sql = `SELECT t.*, p.name as patient_name, p.phone, d.name as doctor_name
                   FROM tbl_treatment t
                   LEFT JOIN tbl_patient p ON t.patient_id = p.patient_id
                   LEFT JOIN tbl_doctor d ON t.doctor_id = d.id
                   WHERE 1=1`;
        
        const params = [];

        if (q) {
            sql += ' AND (p.name LIKE ? OR p.patient_id LIKE ? OR t.diagnosis LIKE ?)';
            params.push(`%${q}%`, `%${q}%`, `%${q}%`);
        }

        if (startDate) {
            sql += ' AND t.Date >= ?';
            params.push(startDate);
        }

        if (endDate) {
            sql += ' AND t.Date <= ?';
            params.push(endDate);
        }

        if (doctorId) {
            sql += ' AND t.doctor_id = ?';
            params.push(doctorId);
        }

        sql += ' ORDER BY t.Date DESC LIMIT 100';

        const treatments = await query(sql, params);

        res.json({
            success: true,
            count: treatments.length,
            data: treatments
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get treatment statistics
 * @route   GET /api/emr/stats
 * @access  Private
 */
exports.getStats = async (req, res, next) => {
    try {
        const todayCount = await query(
            'SELECT COUNT(*) as count FROM tbl_treatment WHERE Date = CURDATE()'
        );

        const weekCount = await query(
            'SELECT COUNT(*) as count FROM tbl_treatment WHERE Date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)'
        );

        const monthCount = await query(
            'SELECT COUNT(*) as count FROM tbl_treatment WHERE Date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)'
        );

        const byType = await query(
            `SELECT Treatment_Type, COUNT(*) as count 
             FROM tbl_treatment 
             WHERE Date >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)
             GROUP BY Treatment_Type`
        );

        res.json({
            success: true,
            data: {
                today: todayCount[0].count,
                week: weekCount[0].count,
                month: monthCount[0].count,
                byType: byType
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get all treatments for EMR overview
 * @route   GET /api/emr/all-treatments
 * @access  Private
 */
exports.getAllTreatments = async (req, res, next) => {
    try {
        const sql = `
            SELECT 
                t.*,
                p.name as patient_name,
                p.age as patient_age,
                d.name as doctor_name
            FROM tbl_treatment t
            LEFT JOIN tbl_patient p ON t.patient_id = p.patient_id
            LEFT JOIN tbl_doctor d ON t.doctor_id = d.id
            ORDER BY t.Date DESC, t.id DESC
        `;

        const treatments = await query(sql);

        res.json({
            success: true,
            count: treatments.length,
            data: treatments
        });
    } catch (error) {
        next(error);
    }
};

