const { query } = require('../config/database');

// Expense Category CRUD
exports.getExpenseCategories = async (req, res, next) => {
    try {
        const categories = await query('SELECT id, name FROM tbl_cat_expenses ORDER BY name');
        res.json({
            success: true,
            data: categories
        });
    } catch (error) {
        next(error);
    }
};

exports.createExpenseCategory = async (req, res, next) => {
    try {
        const { name } = req.body;
        if (!name || !name.trim()) {
            return res.status(400).json({
                success: false,
                message: 'Category name is required'
            });
        }

        const existing = await query('SELECT id FROM tbl_cat_expenses WHERE LOWER(name) = LOWER(?)', [name.trim()]);
        if (existing.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Category already exists'
            });
        }

        const result = await query('INSERT INTO tbl_cat_expenses (name) VALUES (?)', [name.trim()]);
        res.status(201).json({
            success: true,
            message: 'Category created successfully',
            data: { id: result.insertId, name: name.trim() }
        });
    } catch (error) {
        next(error);
    }
};

exports.updateExpenseCategory = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { name } = req.body;

        if (!name || !name.trim()) {
            return res.status(400).json({
                success: false,
                message: 'Category name is required'
            });
        }

        const existing = await query('SELECT id FROM tbl_cat_expenses WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Category not found'
            });
        }

        const duplicate = await query('SELECT id FROM tbl_cat_expenses WHERE LOWER(name) = LOWER(?) AND id != ?', [name.trim(), id]);
        if (duplicate.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Another category with this name already exists'
            });
        }

        await query('UPDATE tbl_cat_expenses SET name = ? WHERE id = ?', [name.trim(), id]);
        res.json({
            success: true,
            message: 'Category updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

exports.deleteExpenseCategory = async (req, res, next) => {
    try {
        const { id } = req.params;
        const existing = await query('SELECT id FROM tbl_cat_expenses WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Category not found'
            });
        }

        const usage = await query('SELECT COUNT(*) as total FROM tbl_expenses WHERE cat_exp_id = ?', [id]);
        if (usage[0].total > 0) {
            return res.status(400).json({
                success: false,
                message: 'Cannot delete category while expenses are assigned to it'
            });
        }

        await query('DELETE FROM tbl_cat_expenses WHERE id = ?', [id]);
        res.json({
            success: true,
            message: 'Category deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

// Create new expense
exports.createExpense = async (req, res, next) => {
    try {
        const { cat_exp_id, amount, user_name, reason, remark, date, status, payment_method } = req.body;
        let normalizedPaymentMethod = payment_method ?? req.body.paymentMethod ?? null;
        if (typeof normalizedPaymentMethod === 'string') {
            normalizedPaymentMethod = normalizedPaymentMethod.trim() || null;
        }

        if (!cat_exp_id || !amount || !user_name || !reason || !date) {
            return res.status(400).json({
                success: false,
                message: 'Missing required fields: cat_exp_id, amount, user_name, reason, date'
            });
        }

        const sql = `
            INSERT INTO tbl_expenses (cat_exp_id, amount, user_name, reason, remark, date, status, payment_method)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        `;

        const result = await query(sql, [
            cat_exp_id,
            amount,
            user_name,
            reason,
            remark || null,
            date,
            status || 1,
            normalizedPaymentMethod
        ]);

        res.status(201).json({
            success: true,
            message: 'Expense created successfully',
            data: {
                id: result.insertId,
                cat_exp_id,
                amount,
                user_name,
                reason,
                remark,
                date,
                status: status || 1,
                payment_method: normalizedPaymentMethod
            }
        });
    } catch (error) {
        next(error);
    }
};

// Get all expenses
exports.getExpenses = async (req, res, next) => {
    try {
        const limit = req.query.limit ? parseInt(req.query.limit) : 1000;
        const offset = req.query.offset ? parseInt(req.query.offset) : 0;

        const sql = `
            SELECT e.*, COALESCE(c.name, e.cat_exp_id) AS category_name
            FROM tbl_expenses e
            LEFT JOIN tbl_cat_expenses c ON c.id = CAST(e.cat_exp_id AS UNSIGNED)
            ORDER BY e.date DESC, e.id DESC
            LIMIT ? OFFSET ?
        `;

        const expenses = await query(sql, [limit, offset]);

        // Get total count
        const countResult = await query('SELECT COUNT(*) as total FROM tbl_expenses');
        const total = countResult[0].total;

        res.json({
            success: true,
            data: expenses,
            pagination: {
                total,
                limit,
                offset,
                hasMore: offset + expenses.length < total
            }
        });
    } catch (error) {
        next(error);
    }
};

// Get expense by ID
exports.getExpenseById = async (req, res, next) => {
    try {
        const { id } = req.params;

        const sql = `
            SELECT e.*, COALESCE(c.name, e.cat_exp_id) AS category_name
            FROM tbl_expenses e
            LEFT JOIN tbl_cat_expenses c ON c.id = CAST(e.cat_exp_id AS UNSIGNED)
            WHERE e.id = ?
        `;
        const result = await query(sql, [id]);

        if (result.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Expense not found'
            });
        }

        res.json({
            success: true,
            data: result[0]
        });
    } catch (error) {
        next(error);
    }
};

// Update expense
exports.updateExpense = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { cat_exp_id, amount, user_name, reason, remark, date, status, payment_method } = req.body;
        let normalizedPaymentMethod = payment_method ?? req.body.paymentMethod ?? undefined;
        if (typeof normalizedPaymentMethod === 'string') {
            normalizedPaymentMethod = normalizedPaymentMethod.trim() || null;
        }

        // Check if expense exists
        const checkSql = 'SELECT * FROM tbl_expenses WHERE id = ?';
        const existing = await query(checkSql, [id]);

        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Expense not found'
            });
        }

        const sql = `
            UPDATE tbl_expenses
            SET cat_exp_id = ?, amount = ?, user_name = ?, reason = ?, remark = ?, date = ?, status = ?, payment_method = ?
            WHERE id = ?
        `;

        await query(sql, [
            cat_exp_id || existing[0].cat_exp_id,
            amount || existing[0].amount,
            user_name || existing[0].user_name,
            reason || existing[0].reason,
            remark !== undefined ? remark : existing[0].remark,
            date || existing[0].date,
            status !== undefined ? status : existing[0].status,
            normalizedPaymentMethod !== undefined ? normalizedPaymentMethod : existing[0].payment_method,
            id
        ]);

        res.json({
            success: true,
            message: 'Expense updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

// Delete expense
exports.deleteExpense = async (req, res, next) => {
    try {
        const { id } = req.params;

        // Check if expense exists
        const checkSql = 'SELECT * FROM tbl_expenses WHERE id = ?';
        const existing = await query(checkSql, [id]);

        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Expense not found'
            });
        }

        const sql = 'DELETE FROM tbl_expenses WHERE id = ?';
        await query(sql, [id]);

        res.json({
            success: true,
            message: 'Expense deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

// Get expense statistics
exports.getExpenseStats = async (req, res, next) => {
    try {
        // Today's expenses
        const todaySql = `
            SELECT COALESCE(SUM(amount), 0) as today_total
            FROM tbl_expenses
            WHERE date = CURDATE()
        `;
        const todayResult = await query(todaySql);

        // This month's expenses
        const monthSql = `
            SELECT COALESCE(SUM(amount), 0) as month_total
            FROM tbl_expenses
            WHERE YEAR(date) = YEAR(CURDATE()) AND MONTH(date) = MONTH(CURDATE())
        `;
        const monthResult = await query(monthSql);

        // This year's expenses
        const yearSql = `
            SELECT COALESCE(SUM(amount), 0) as year_total
            FROM tbl_expenses
            WHERE YEAR(date) = YEAR(CURDATE())
        `;
        const yearResult = await query(yearSql);

        // Total count
        const countSql = 'SELECT COUNT(*) as total_count FROM tbl_expenses';
        const countResult = await query(countSql);

        res.json({
            success: true,
            data: {
                today: todayResult[0].today_total,
                month: monthResult[0].month_total,
                year: yearResult[0].year_total,
                total_count: countResult[0].total_count
            }
        });
    } catch (error) {
        next(error);
    }
};

// Get expenses by date range
exports.getExpensesByDateRange = async (req, res, next) => {
    try {
        const { start_date, end_date } = req.query;

        if (!start_date || !end_date) {
            return res.status(400).json({
                success: false,
                message: 'start_date and end_date are required'
            });
        }

        const sql = `
            SELECT e.*, COALESCE(c.name, e.cat_exp_id) AS category_name
            FROM tbl_expenses e
            LEFT JOIN tbl_cat_expenses c ON c.id = CAST(e.cat_exp_id AS UNSIGNED)
            WHERE e.date BETWEEN ? AND ?
            ORDER BY e.date DESC, e.id DESC
        `;

        const expenses = await query(sql, [start_date, end_date]);

        // Calculate total
        const totalSql = `
            SELECT COALESCE(SUM(amount), 0) as total
            FROM tbl_expenses
            WHERE date BETWEEN ? AND ?
        `;
        const totalResult = await query(totalSql, [start_date, end_date]);

        res.json({
            success: true,
            data: expenses,
            summary: {
                total_amount: totalResult[0].total,
                total_count: expenses.length
            }
        });
    } catch (error) {
        next(error);
    }
};

