/**
 * Hospital Operations Controller
 * Bed Management, OR Scheduling, Emergency Department
 */

const { query } = require('../config/database');

// ============================================
// BED MANAGEMENT
// ============================================

/**
 * @desc    Get all beds
 * @route   GET /api/hospital/beds
 * @access  Private
 */
exports.getBeds = async (req, res, next) => {
    try {
        const { department_id, status, bed_type } = req.query;
        let sql = `SELECT b.*, d.dept_name as department_name
            FROM tbl_beds b
            LEFT JOIN tbl_departments d ON b.department_id = d.id
            WHERE 1=1`;
        const params = [];

        if (department_id) {
            sql += ' AND b.department_id = ?';
            params.push(department_id);
        }
        if (status) {
            sql += ' AND b.status = ?';
            params.push(status);
        }
        if (bed_type) {
            sql += ' AND b.bed_type = ?';
            params.push(bed_type);
        }

        sql += ' ORDER BY b.department_id, b.room_number, b.bed_number';
        const beds = await query(sql, params);

        res.json({
            success: true,
            count: beds.length,
            data: beds
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get bed availability summary
 * @route   GET /api/hospital/beds/availability
 * @access  Private
 */
exports.getBedAvailability = async (req, res, next) => {
    try {
        const sql = `SELECT
            d.dept_name,
            d.id as department_id,
            COUNT(b.id) as total_beds,
            SUM(CASE WHEN b.status = 'Available' THEN 1 ELSE 0 END) as available,
            SUM(CASE WHEN b.status = 'Occupied' THEN 1 ELSE 0 END) as occupied,
            SUM(CASE WHEN b.status = 'Reserved' THEN 1 ELSE 0 END) as reserved,
            SUM(CASE WHEN b.status IN ('Maintenance', 'Cleaning', 'Out_of_Service') THEN 1 ELSE 0 END) as unavailable,
            ROUND((SUM(CASE WHEN b.status = 'Occupied' THEN 1 ELSE 0 END) / COUNT(b.id)) * 100, 2) as occupancy_rate
            FROM tbl_departments d
            LEFT JOIN tbl_beds b ON d.id = b.department_id
            WHERE d.is_active = 1
            GROUP BY d.id, d.dept_name
            ORDER BY d.dept_name`;
        
        const availability = await query(sql);

        res.json({
            success: true,
            data: availability
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Assign bed to patient
 * @route   POST /api/hospital/beds/assign
 * @access  Private
 */
exports.assignBed = async (req, res, next) => {
    try {
        const { bed_id, patient_id, reason } = req.body;

        if (!bed_id || !patient_id) {
            return res.status(400).json({
                success: false,
                message: 'Bed ID and Patient ID are required'
            });
        }

        // Check if bed is available
        const bedCheck = await query('SELECT status FROM tbl_beds WHERE id = ?', [bed_id]);
        if (bedCheck.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Bed not found'
            });
        }
        if (bedCheck[0].status !== 'Available') {
            return res.status(400).json({
                success: false,
                message: 'Bed is not available'
            });
        }

        // Update bed status
        await query('UPDATE tbl_beds SET status = "Occupied", current_patient_id = ? WHERE id = ?', 
            [patient_id, bed_id]);

        // Create assignment record
        const assignSql = `INSERT INTO tbl_bed_assignments 
            (bed_id, patient_id, assigned_by, reason, status) 
            VALUES (?, ?, ?, ?, 'Active')`;
        const result = await query(assignSql, [bed_id, patient_id, req.user.id, reason]);

        res.status(201).json({
            success: true,
            message: 'Bed assigned successfully',
            data: { assignment_id: result.insertId }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Discharge patient from bed
 * @route   PATCH /api/hospital/beds/:bedId/discharge
 * @access  Private
 */
exports.dischargeBed = async (req, res, next) => {
    try {
        const { bedId } = req.params;

        // Get current assignment
        const assignment = await query(
            'SELECT id FROM tbl_bed_assignments WHERE bed_id = ? AND status = "Active"',
            [bedId]
        );

        if (assignment.length > 0) {
            // Update assignment
            await query(
                'UPDATE tbl_bed_assignments SET discharged_date = NOW(), status = "Completed" WHERE id = ?',
                [assignment[0].id]
            );
        }

        // Update bed status
        await query('UPDATE tbl_beds SET status = "Cleaning", current_patient_id = NULL WHERE id = ?', [bedId]);

        res.json({
            success: true,
            message: 'Patient discharged from bed'
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// OPERATING ROOM MANAGEMENT
// ============================================

/**
 * @desc    Get operating rooms
 * @route   GET /api/hospital/operating-rooms
 * @access  Private
 */
exports.getOperatingRooms = async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_operating_rooms WHERE is_active = 1 ORDER BY or_code';
        const rooms = await query(sql);

        res.json({
            success: true,
            count: rooms.length,
            data: rooms
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create surgery schedule
 * @route   POST /api/hospital/surgeries
 * @access  Private
 */
exports.scheduleSurgery = async (req, res, next) => {
    try {
        const {
            or_id,
            patient_id,
            surgeon_id,
            anesthesiologist_id,
            procedure_name,
            procedure_code,
            scheduled_date,
            scheduled_start_time,
            scheduled_end_time,
            priority = 'Elective',
            anesthesia_type,
            estimated_duration,
            pre_op_diagnosis,
            special_requirements
        } = req.body;

        if (!or_id || !patient_id || !surgeon_id || !procedure_name || !scheduled_date || !scheduled_start_time) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing'
            });
        }

        // Generate schedule number
        const year = new Date().getFullYear();
        await query('UPDATE tbl_surgery_sequence SET last_schedule_number = last_schedule_number + 1 WHERE year = ?', [year]);
        const seqResult = await query('SELECT last_schedule_number FROM tbl_surgery_sequence WHERE year = ?', [year]);
        const scheduleNum = seqResult[0].last_schedule_number;
        const schedule_number = `SUR${year}${String(scheduleNum).padStart(6, '0')}`;

        // Create surgery schedule
        const sql = `INSERT INTO tbl_surgery_schedules 
            (schedule_number, or_id, patient_id, surgeon_id, anesthesiologist_id, 
            procedure_name, procedure_code, scheduled_date, scheduled_start_time, scheduled_end_time,
            priority, anesthesia_type, estimated_duration, pre_op_diagnosis, special_requirements, 
            created_by, status) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Scheduled')`;
        
        const result = await query(sql, [
            schedule_number,
            or_id,
            patient_id,
            surgeon_id,
            anesthesiologist_id ?? null,
            procedure_name,
            procedure_code ?? null,
            scheduled_date,
            scheduled_start_time,
            scheduled_end_time ?? null,
            priority,
            anesthesia_type ?? null,
            estimated_duration ?? null,
            pre_op_diagnosis ?? null,
            special_requirements ?? null,
            req.user.id
        ]);

        res.status(201).json({
            success: true,
            data: {
                id: result.insertId,
                schedule_number: schedule_number
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get surgery schedules
 * @route   GET /api/hospital/surgeries
 * @access  Private
 */
exports.getSurgerySchedules = async (req, res, next) => {
    try {
        const { date, or_id, status } = req.query;
        let sql = `SELECT s.*, o.or_name, p.name as patient_name, 
            d.name as surgeon_name, a.name as anesthesiologist_name
            FROM tbl_surgery_schedules s
            LEFT JOIN tbl_operating_rooms o ON s.or_id = o.id
            LEFT JOIN tbl_patient p ON s.patient_id = p.patient_id
            LEFT JOIN tbl_doctor d ON s.surgeon_id = d.id
            LEFT JOIN tbl_doctor a ON s.anesthesiologist_id = a.id
            WHERE 1=1`;
        const params = [];

        if (date) {
            sql += ' AND s.scheduled_date = ?';
            params.push(date);
        }
        if (or_id) {
            sql += ' AND s.or_id = ?';
            params.push(or_id);
        }
        if (status) {
            sql += ' AND s.status = ?';
            params.push(status);
        }

        sql += ' ORDER BY s.scheduled_date, s.scheduled_start_time';
        const schedules = await query(sql, params);

        res.json({
            success: true,
            count: schedules.length,
            data: schedules
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update surgery status
 * @route   PATCH /api/hospital/surgeries/:id/status
 * @access  Private
 */
exports.updateSurgeryStatus = async (req, res, next) => {
    try {
        const { status, actual_start_time, actual_end_time } = req.body;
        let sql = 'UPDATE tbl_surgery_schedules SET status = ?';
        const params = [status];

        if (actual_start_time) {
            sql += ', actual_start_time = ?';
            params.push(actual_start_time);
        }
        if (actual_end_time) {
            sql += ', actual_end_time = ?';
            params.push(actual_end_time);
        }

        sql += ' WHERE id = ?';
        params.push(req.params.id);

        await query(sql, params);

        res.json({
            success: true,
            message: 'Surgery status updated'
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// EMERGENCY DEPARTMENT
// ============================================

/**
 * @desc    Create emergency admission
 * @route   POST /api/hospital/emergency
 * @access  Private
 */
exports.createEmergencyAdmission = async (req, res, next) => {
    try {
        const {
            patient_id,
            arrival_mode,
            chief_complaint,
            triage_level,
            vital_signs
        } = req.body;

        if (!arrival_mode || !chief_complaint || !triage_level) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing'
            });
        }

        // Generate ER number
        const year = new Date().getFullYear();
        await query('UPDATE tbl_er_sequence SET last_er_number = last_er_number + 1 WHERE year = ?', [year]);
        const seqResult = await query('SELECT last_er_number FROM tbl_er_sequence WHERE year = ?', [year]);
        const erNum = seqResult[0].last_er_number;
        const er_number = `ER${year}${String(erNum).padStart(6, '0')}`;

        // Create ER admission
        const sql = `INSERT INTO tbl_emergency_admissions 
            (er_number, patient_id, arrival_mode, chief_complaint, triage_level, 
            triage_time, triaged_by, vital_signs, status) 
            VALUES (?, ?, ?, ?, ?, NOW(), ?, ?, 'Waiting')`;
        
        const result = await query(sql, [
            er_number,
            patient_id ?? null,
            arrival_mode,
            chief_complaint,
            triage_level,
            req.user.id,
            vital_signs ? JSON.stringify(vital_signs) : null
        ]);

        res.status(201).json({
            success: true,
            data: {
                id: result.insertId,
                er_number: er_number
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get emergency admissions
 * @route   GET /api/hospital/emergency
 * @access  Private
 */
exports.getEmergencyAdmissions = async (req, res, next) => {
    try {
        const { status, date } = req.query;
        let sql = `SELECT e.*, p.name as patient_name, p.age, p.phone, d.name as doctor_name
            FROM tbl_emergency_admissions e
            LEFT JOIN tbl_patient p ON e.patient_id = p.patient_id
            LEFT JOIN tbl_doctor d ON e.assigned_doctor_id = d.id
            WHERE 1=1`;
        const params = [];

        if (status) {
            sql += ' AND e.status = ?';
            params.push(status);
        }
        if (date) {
            sql += ' AND DATE(e.arrival_date) = ?';
            params.push(date);
        } else {
            sql += ' AND DATE(e.arrival_date) = CURDATE()';
        }

        sql += ' ORDER BY e.triage_level, e.arrival_date';
        const admissions = await query(sql, params);

        res.json({
            success: true,
            count: admissions.length,
            data: admissions
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update ER patient status
 * @route   PATCH /api/hospital/emergency/:id
 * @access  Private
 */
exports.updateEmergencyAdmission = async (req, res, next) => {
    try {
        const { status, assigned_doctor_id, disposition } = req.body;
        let sql = 'UPDATE tbl_emergency_admissions SET ';
        const updates = [];
        const params = [];

        if (status) {
            updates.push('status = ?');
            params.push(status);
            if (status === 'In_Treatment' && !req.body.treatment_start_time) {
                updates.push('treatment_start_time = NOW()');
            }
        }
        if (assigned_doctor_id) {
            updates.push('assigned_doctor_id = ?');
            params.push(assigned_doctor_id);
            updates.push('assigned_time = NOW()');
        }
        if (disposition) {
            updates.push('disposition = ?');
            params.push(disposition);
            updates.push('disposition_time = NOW()');
        }

        if (updates.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'No updates provided'
            });
        }

        sql += updates.join(', ') + ' WHERE id = ?';
        params.push(req.params.id);

        await query(sql, params);

        res.json({
            success: true,
            message: 'Emergency admission updated'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get departments
 * @route   GET /api/hospital/departments
 * @access  Private
 */
exports.getDepartments = async (req, res, next) => {
    try {
        const sql = `SELECT d.*, doc.name as head_doctor_name
            FROM tbl_departments d
            LEFT JOIN tbl_doctor doc ON d.head_of_department = doc.id
            WHERE d.is_active = 1
            ORDER BY d.dept_name`;
        
        const departments = await query(sql);

        res.json({
            success: true,
            count: departments.length,
            data: departments
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create new department
 * @route   POST /api/hospital/departments
 * @access  Private
 */
exports.createDepartment = async (req, res, next) => {
    try {
        const {
            dept_code,
            dept_name,
            description,
            head_of_department,
            contact_number,
            location,
            bed_capacity
        } = req.body;

        // Basic validation
        if (!dept_code || !dept_name) {
            return res.status(400).json({
                success: false,
                message: 'Department code and name are required'
            });
        }

        // Check if department code already exists
        const existing = await query(
            'SELECT id FROM tbl_departments WHERE dept_code = ?',
            [dept_code.trim().toUpperCase()]
        );

        if (existing.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Department code already exists'
            });
        }

        const sql = `INSERT INTO tbl_departments 
            (dept_code, dept_name, description, head_of_department, contact_number, location, bed_capacity)
            VALUES (?, ?, ?, ?, ?, ?, ?)`;

        const result = await query(sql, [
            dept_code.trim().toUpperCase(),
            dept_name.trim(),
            description || null,
            head_of_department || null,
            contact_number || null,
            location || null,
            bed_capacity || 0
        ]);

        res.status(201).json({
            success: true,
            message: 'Department created successfully',
            data: {
                id: result.insertId,
                dept_code: dept_code.trim().toUpperCase(),
                dept_name: dept_name.trim()
            }
        });
    } catch (error) {
        if (error.code === 'ER_DUP_ENTRY') {
            return res.status(400).json({
                success: false,
                message: 'Department code already exists'
            });
        }
        next(error);
    }
};

/**
 * @desc    Update department
 * @route   PUT /api/hospital/departments/:id
 * @access  Private
 */
exports.updateDepartment = async (req, res, next) => {
    try {
        const { id } = req.params;
        const {
            dept_code,
            dept_name,
            description,
            head_of_department,
            contact_number,
            location,
            bed_capacity,
            is_active
        } = req.body;

        // Check if department exists
        const existing = await query('SELECT id FROM tbl_departments WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Department not found'
            });
        }

        // Check if new dept_code conflicts with another department
        if (dept_code) {
            const codeCheck = await query(
                'SELECT id FROM tbl_departments WHERE dept_code = ? AND id != ?',
                [dept_code.trim().toUpperCase(), id]
            );
            if (codeCheck.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Department code already exists'
                });
            }
        }

        const updateFields = [];
        const updateValues = [];

        if (dept_code) {
            updateFields.push('dept_code = ?');
            updateValues.push(dept_code.trim().toUpperCase());
        }
        if (dept_name) {
            updateFields.push('dept_name = ?');
            updateValues.push(dept_name.trim());
        }
        if (description !== undefined) {
            updateFields.push('description = ?');
            updateValues.push(description || null);
        }
        if (head_of_department !== undefined) {
            updateFields.push('head_of_department = ?');
            updateValues.push(head_of_department || null);
        }
        if (contact_number !== undefined) {
            updateFields.push('contact_number = ?');
            updateValues.push(contact_number || null);
        }
        if (location !== undefined) {
            updateFields.push('location = ?');
            updateValues.push(location || null);
        }
        if (bed_capacity !== undefined) {
            updateFields.push('bed_capacity = ?');
            updateValues.push(bed_capacity || 0);
        }
        if (is_active !== undefined) {
            updateFields.push('is_active = ?');
            updateValues.push(is_active ? 1 : 0);
        }

        if (updateFields.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'No fields to update'
            });
        }

        updateValues.push(id);
        const sql = `UPDATE tbl_departments SET ${updateFields.join(', ')} WHERE id = ?`;
        
        await query(sql, updateValues);

        res.json({
            success: true,
            message: 'Department updated successfully'
        });
    } catch (error) {
        if (error.code === 'ER_DUP_ENTRY') {
            return res.status(400).json({
                success: false,
                message: 'Department code already exists'
            });
        }
        next(error);
    }
};

/**
 * @desc    Delete department (soft delete)
 * @route   DELETE /api/hospital/departments/:id
 * @access  Private
 */
exports.deleteDepartment = async (req, res, next) => {
    try {
        const { id } = req.params;

        // Check if department exists
        const existing = await query('SELECT id FROM tbl_departments WHERE id = ?', [id]);
        if (existing.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Department not found'
            });
        }

        // Check if department has beds assigned
        const bedsCount = await query(
            'SELECT COUNT(*) as count FROM tbl_beds WHERE department_id = ?',
            [id]
        );

        if (bedsCount[0].count > 0) {
            // Soft delete instead of hard delete
            await query('UPDATE tbl_departments SET is_active = 0 WHERE id = ?', [id]);
            return res.json({
                success: true,
                message: 'Department deactivated (has assigned beds). It cannot be permanently deleted.',
                softDelete: true
            });
        }

        // Hard delete if no beds assigned
        await query('DELETE FROM tbl_departments WHERE id = ?', [id]);

        res.json({
            success: true,
            message: 'Department deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get hospital statistics
 * @route   GET /api/hospital/stats
 * @access  Private
 */
exports.getHospitalStats = async (req, res, next) => {
    try {
        const today = new Date().toISOString().split('T')[0];

        // Bed statistics
        const bedStats = await query(`
            SELECT 
                COUNT(*) as total_beds,
                SUM(CASE WHEN status = 'Available' THEN 1 ELSE 0 END) as available_beds,
                SUM(CASE WHEN status = 'Occupied' THEN 1 ELSE 0 END) as occupied_beds,
                ROUND((SUM(CASE WHEN status = 'Occupied' THEN 1 ELSE 0 END) / COUNT(*)) * 100, 2) as occupancy_rate
            FROM tbl_beds
        `);

        // OR statistics
        const orStats = await query(`
            SELECT 
                COUNT(*) as total_surgeries,
                SUM(CASE WHEN status = 'Scheduled' THEN 1 ELSE 0 END) as scheduled,
                SUM(CASE WHEN status = 'In_Progress' THEN 1 ELSE 0 END) as in_progress,
                SUM(CASE WHEN status = 'Completed' THEN 1 ELSE 0 END) as completed
            FROM tbl_surgery_schedules
            WHERE scheduled_date = '${today}'
        `);

        // ER statistics
        const erStats = await query(`
            SELECT
                COUNT(*) as total_patients,
                SUM(CASE WHEN status = 'Waiting' THEN 1 ELSE 0 END) as waiting,
                SUM(CASE WHEN status = 'In_Treatment' THEN 1 ELSE 0 END) as in_treatment,
                AVG(TIMESTAMPDIFF(MINUTE, arrival_date, COALESCE(treatment_start_time, NOW()))) as avg_wait_time
            FROM tbl_emergency_admissions
            WHERE DATE(arrival_date) = '${today}'
        `);

        res.json({
            success: true,
            data: {
                beds: bedStats[0],
                surgeries: orStats[0],
                emergency: erStats[0]
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create new bed
 * @route   POST /api/hospital/beds
 * @access  Private
 */
exports.createBed = async (req, res, next) => {
    try {
        const {
            bed_number,
            room_number,
            department_id,
            bed_type,
            status = 'Available',
            daily_rate = 0,
            equipment,
            notes
        } = req.body;

        // Basic validation
        if (!bed_number || !room_number || !department_id || !bed_type) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing: bed_number, room_number, department_id, bed_type'
            });
        }

        // Check if bed already exists
        const existingBed = await query(
            'SELECT id FROM tbl_beds WHERE bed_number = ? AND room_number = ? AND department_id = ?',
            [bed_number, room_number, department_id]
        );

        if (existingBed.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Bed already exists in this room and department'
            });
        }

        const sql = `INSERT INTO tbl_beds
            (bed_number, room_number, department_id, bed_type, status, daily_rate, equipment, notes)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)`;

        const result = await query(sql, [
            bed_number,
            room_number,
            department_id,
            bed_type,
            status,
            daily_rate,
            equipment || null,
            notes || null
        ]);

        res.status(201).json({
            success: true,
            data: {
                id: result.insertId,
                bed_number: bed_number,
                room_number: room_number
            },
            message: 'Bed created successfully'
        });
    } catch (error) {
        if (error.code === 'ER_DUP_ENTRY') {
            return res.status(400).json({
                success: false,
                message: 'Bed already exists'
            });
        }
        next(error);
    }
};

/**
 * @desc    Update bed
 * @route   PUT /api/hospital/beds/:bedId
 * @access  Private
 */
exports.updateBed = async (req, res, next) => {
    try {
        const { bedId } = req.params;
        const {
            bed_number,
            room_number,
            department_id,
            bed_type,
            status,
            daily_rate,
            equipment,
            notes
        } = req.body;

        // Basic validation
        if (!bed_number || !room_number || !department_id || !bed_type) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing'
            });
        }

        // Check if bed exists
        const existingBed = await query('SELECT id FROM tbl_beds WHERE id = ?', [bedId]);
        if (existingBed.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Bed not found'
            });
        }

        // Check if new bed number/room combination already exists (excluding current bed)
        const duplicateBed = await query(
            'SELECT id FROM tbl_beds WHERE bed_number = ? AND room_number = ? AND department_id = ? AND id != ?',
            [bed_number, room_number, department_id, bedId]
        );

        if (duplicateBed.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Another bed already exists with this number and room'
            });
        }

        const sql = `UPDATE tbl_beds SET
            bed_number = ?, room_number = ?, department_id = ?, bed_type = ?, status = ?,
            daily_rate = ?, equipment = ?, notes = ?
            WHERE id = ?`;

        await query(sql, [
            bed_number,
            room_number,
            department_id,
            bed_type,
            status,
            daily_rate,
            equipment || null,
            notes || null,
            bedId
        ]);

        res.json({
            success: true,
            message: 'Bed updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Delete bed
 * @route   DELETE /api/hospital/beds/:bedId
 * @access  Private
 */
exports.deleteBed = async (req, res, next) => {
    try {
        const { bedId } = req.params;

        // Check if bed exists
        const existingBed = await query('SELECT id, status, current_patient_id FROM tbl_beds WHERE id = ?', [bedId]);
        if (existingBed.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Bed not found'
            });
        }

        const bed = existingBed[0];

        // Check if bed is occupied
        if (bed.status === 'Occupied' && bed.current_patient_id) {
            return res.status(400).json({
                success: false,
                message: 'Cannot delete occupied bed. Please discharge patient first.'
            });
        }

        // Delete bed
        await query('DELETE FROM tbl_beds WHERE id = ?', [bedId]);

        res.json({
            success: true,
            message: 'Bed deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

