/**
 * Inpatient Deposit Management Controller
 * Handles deposit operations for inpatients
 */

const { query, transaction } = require('../config/database');

/**
 * @desc    Get inpatient deposit history
 * @route   GET /api/inpatient-deposits/:inpatientId
 * @access  Private
 */
exports.getInpatientDeposits = async (req, res, next) => {
    try {
        const { inpatientId } = req.params;
        
        // Get inpatient info
        const inpatientSql = `
            SELECT 
                i.id,
                i.patient_id,
                i.name,
                i.deposit,
                i.start_date,
                i.end_date,
                d.name as doctor_name,
                b.bed_number,
                b.room_number
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            WHERE i.id = ?
        `;
        const inpatients = await query(inpatientSql, [inpatientId]);
        
        if (inpatients.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Inpatient not found'
            });
        }
        
        // Get deposit transactions
        const depositsSql = `
            SELECT 
                id.*,
                u.name as created_by_name
            FROM tbl_inpatient_deposits id
            LEFT JOIN tbl_user u ON id.created_by = u.id
            WHERE id.inpatient_id = ?
            ORDER BY id.created_at DESC
        `;
        
        const deposits = await query(depositsSql, [inpatientId]);
        
        // Calculate totals
        const totals = deposits.reduce((acc, deposit) => {
            acc.totalDeposits += parseFloat(deposit.amount || 0);
            return acc;
        }, { totalDeposits: 0 });
        
        res.status(200).json({
            success: true,
            data: {
                inpatient: inpatients[0],
                deposits,
                totals,
                currentBalance: parseFloat(inpatients[0].deposit || 0)
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Add deposit to inpatient
 * @route   POST /api/inpatient-deposits
 * @access  Private
 */
exports.addInpatientDeposit = async (req, res, next) => {
    try {
        const { 
            inpatient_id, 
            amount, 
            payment_method = 'cash', 
            notes = '',
            created_by 
        } = req.body;
        
        if (!inpatient_id || !amount || amount <= 0) {
            return res.status(400).json({
                success: false,
                message: 'Invalid deposit data'
            });
        }
        
        const result = await transaction(async (conn) => {
            // Check if inpatient exists and is active
            const inpatientCheck = await conn.query(
                'SELECT id, patient_id, name, deposit FROM tbl_inpatient WHERE id = ? AND end_date IS NULL',
                [inpatient_id]
            );
            
            if (inpatientCheck.length === 0) {
                throw new Error('Inpatient not found or already discharged');
            }
            
            const inpatient = inpatientCheck[0];
            
            // Update inpatient deposit
            const newDeposit = (inpatient.deposit || 0) + parseFloat(amount);
            await conn.query(
                'UPDATE tbl_inpatient SET deposit = ? WHERE id = ?',
                [newDeposit, inpatient_id]
            );
            
            // Create deposit record
            const depositSql = `
                INSERT INTO tbl_inpatient_deposits 
                (inpatient_id, amount, payment_method, notes, created_by, created_at)
                VALUES (?, ?, ?, ?, ?, NOW())
            `;
            
            const depositResult = await conn.query(depositSql, [
                inpatient_id,
                parseFloat(amount),
                payment_method,
                notes,
                created_by || req.user?.id || 1
            ]);
            
            return {
                depositId: depositResult.insertId,
                newBalance: newDeposit,
                inpatient: inpatient
            };
        });
        
        res.status(201).json({
            success: true,
            message: 'Deposit added successfully',
            data: result
        });
        
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Withdraw from inpatient deposit
 * @route   POST /api/inpatient-deposits/withdraw
 * @access  Private
 */
exports.withdrawInpatientDeposit = async (req, res, next) => {
    try {
        const { 
            inpatient_id, 
            amount, 
            payment_method = 'cash', 
            notes = '',
            created_by 
        } = req.body;
        
        if (!inpatient_id || !amount || amount <= 0) {
            return res.status(400).json({
                success: false,
                message: 'Invalid withdrawal data'
            });
        }
        
        const result = await transaction(async (conn) => {
            // Check if inpatient exists and is active
            const inpatientCheck = await conn.query(
                'SELECT id, patient_id, name, deposit FROM tbl_inpatient WHERE id = ? AND end_date IS NULL',
                [inpatient_id]
            );
            
            if (inpatientCheck.length === 0) {
                throw new Error('Inpatient not found or already discharged');
            }
            
            const inpatient = inpatientCheck[0];
            const currentDeposit = parseFloat(inpatient.deposit || 0);
            const withdrawalAmount = parseFloat(amount);
            
            if (currentDeposit < withdrawalAmount) {
                throw new Error('Insufficient deposit balance');
            }
            
            // Update inpatient deposit
            const newDeposit = currentDeposit - withdrawalAmount;
            await conn.query(
                'UPDATE tbl_inpatient SET deposit = ? WHERE id = ?',
                [newDeposit, inpatient_id]
            );
            
            // Create withdrawal record (negative amount)
            const depositSql = `
                INSERT INTO tbl_inpatient_deposits 
                (inpatient_id, amount, payment_method, notes, created_by, created_at)
                VALUES (?, ?, ?, ?, ?, NOW())
            `;
            
            const depositResult = await conn.query(depositSql, [
                inpatient_id,
                -withdrawalAmount, // Negative amount for withdrawal
                payment_method,
                notes,
                created_by || req.user?.id || 1
            ]);
            
            return {
                depositId: depositResult.insertId,
                newBalance: newDeposit,
                inpatient: inpatient
            };
        });
        
        res.status(201).json({
            success: true,
            message: 'Withdrawal processed successfully',
            data: result
        });
        
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get inpatient deposit summary
 * @route   GET /api/inpatient-deposits/:inpatientId/summary
 * @access  Private
 */
exports.getInpatientDepositSummary = async (req, res, next) => {
    try {
        const { inpatientId } = req.params;
        
        // Get inpatient basic info
        const inpatientSql = `
            SELECT 
                i.id,
                i.patient_id,
                i.name,
                i.deposit,
                i.start_date,
                i.end_date,
                d.name as doctor_name,
                b.bed_number,
                b.room_number
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            WHERE i.id = ?
        `;
        const inpatients = await query(inpatientSql, [inpatientId]);
        
        if (inpatients.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Inpatient not found'
            });
        }
        
        const inpatient = inpatients[0];
        
        // Get deposit summary
        const summarySql = `
            SELECT 
                COUNT(*) as total_transactions,
                SUM(CASE WHEN amount > 0 THEN amount ELSE 0 END) as total_deposits,
                SUM(CASE WHEN amount < 0 THEN ABS(amount) ELSE 0 END) as total_withdrawals,
                MAX(created_at) as last_transaction_date
            FROM tbl_inpatient_deposits 
            WHERE inpatient_id = ?
        `;
        
        const summary = await query(summarySql, [inpatientId]);
        
        res.status(200).json({
            success: true,
            data: {
                inpatient,
                summary: summary[0] || {
                    total_transactions: 0,
                    total_deposits: 0,
                    total_withdrawals: 0,
                    last_transaction_date: null
                },
                currentBalance: parseFloat(inpatient.deposit || 0)
            }
        });
        
    } catch (error) {
        next(error);
    }
};
