/**
 * Inpatient Report Controller
 * Generates comprehensive reports for inpatients
 */

const { query } = require('../config/database');

/**
 * @desc    Get comprehensive inpatient report
 * @route   GET /api/reports/inpatient/:inpatientId
 * @access  Private
 */
exports.getInpatientReport = async (req, res, next) => {
    try {
        const { inpatientId } = req.params;
        
        if (!inpatientId) {
            return res.status(400).json({
                success: false,
                message: 'Inpatient ID is required'
            });
        }

        // Get inpatient basic information
        const inpatientSql = `
            SELECT 
                i.*,
                i.id as inpatient_id,
                i.start_date,
                i.end_date,
                i.discharge_date,
                i.final_bill,
                i.discharge_notes,
                i.dr_id,
                d.name as doctor_name,
                b.bed_number,
                b.room_number
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            WHERE i.id = ?
        `;
        
        const inpatientResult = await query(inpatientSql, [inpatientId]);
        
        if (inpatientResult.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Inpatient not found'
            });
        }
        
        const inpatient = inpatientResult[0];
        
        // Get daily usage records
        const dailyUsage = await getDailyUsage(inpatientId);
        
        // Get financial summary
        const financialSummary = await getFinancialSummary(inpatientId, dailyUsage);
        
        // Get deposit history
        const depositHistory = await getDepositHistory(inpatientId);
        
        res.status(200).json({
            success: true,
            data: {
                inpatient,
                dailyUsage,
                financialSummary,
                depositHistory
            }
        });
        
    } catch (error) {
        console.error('Error generating inpatient report:', error);
        next(error);
    }
};

/**
 * @desc    Get all inpatients for report listing
 * @route   GET /api/reports/inpatients
 * @access  Private
 */
exports.getInpatientsForReport = async (req, res, next) => {
    try {
        const { search, status, date } = req.query;
        
        let sql = `
            SELECT 
                i.*,
                i.id as inpatient_id,
                i.start_date,
                i.end_date,
                i.dr_id,
                d.name as dr_name,
                b.bed_number,
                b.room_number
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            WHERE 1=1
        `;
        
        const params = [];
        
        // Search filter
        if (search) {
            sql += ' AND (i.name LIKE ? OR i.patient_id LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }
        
        // Status filter
        if (status === 'admitted') {
            sql += ' AND i.end_date IS NULL';
        } else if (status === 'discharged') {
            sql += ' AND i.end_date IS NOT NULL';
        }
        
        // Date filter
        if (date) {
            sql += ' AND DATE(i.start_date) = ?';
            params.push(date);
        }
        
        sql += ' ORDER BY i.start_date DESC';
        
        const inpatients = await query(sql, params);
        
        res.status(200).json({
            success: true,
            data: inpatients
        });
        
    } catch (error) {
        console.error('Error loading inpatients for report:', error);
        next(error);
    }
};

/**
 * Get daily usage records for an inpatient
 */
async function getDailyUsage(inpatientId) {
    try {
        const usageSql = `
            SELECT 
                DATE(u.created_at) as usage_date,
                u.quantity,
                u.price,
                u.notes,
                u.item_type,
                CASE 
                    WHEN u.item_type = 'product' THEN p.name
                    WHEN u.item_type = 'service' THEN s.service_name
                END as product_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.cat_id
                    WHEN u.item_type = 'service' THEN s.cat_id
                END as category,
                CASE 
                    WHEN u.item_type = 'product' THEN p.unit_name
                    WHEN u.item_type = 'service' THEN 'service'
                END as unit_name
            FROM tbl_inpatient_usage u
            LEFT JOIN tbl_stock p ON u.item_type = 'product' AND u.item_id = p.id
            LEFT JOIN tbl_service s ON u.item_type = 'service' AND u.item_id = s.id
            WHERE u.inpatient_id = ?
            ORDER BY u.created_at DESC
        `;
        
        const usageRecords = await query(usageSql, [inpatientId]);
        
        // Group by date
        const dailyUsage = {};
        
        usageRecords.forEach(record => {
            const date = record.usage_date;
            if (!dailyUsage[date]) {
                dailyUsage[date] = {
                    medications: [],
                    services: [],
                    stock: [],
                    total: 0
                };
            }
            
            const totalPrice = parseFloat(record.quantity || 0) * parseFloat(record.price || 0);
            
            const item = {
                name: record.product_name || 'Unknown Product',
                quantity: record.quantity,
                unit: record.unit_name || 'unit',
                unit_price: record.price,
                total_price: totalPrice,
                category: record.category || 'General',
                notes: record.notes
            };
            
            // Categorize based on item_type
            if (record.item_type === 'service') {
                dailyUsage[date].services.push(item);
            } else {
                // For products, categorize based on category or name patterns
                const category = (record.category || '').toLowerCase();
                const name = (record.product_name || '').toLowerCase();
                
                if (category.includes('medicine') || category.includes('drug') || 
                    name.includes('tablet') || name.includes('capsule') || name.includes('syrup')) {
                    dailyUsage[date].medications.push(item);
                } else {
                    dailyUsage[date].stock.push(item);
                }
            }
            
            dailyUsage[date].total += totalPrice;
        });
        
        return dailyUsage;
        
    } catch (error) {
        console.error('Error getting daily usage:', error);
        return {};
    }
}

/**
 * Get financial summary for an inpatient
 */
async function getFinancialSummary(inpatientId, dailyUsage) {
    try {
        // Calculate total usage from daily usage
        let totalUsage = 0;
        Object.values(dailyUsage).forEach(dayUsage => {
            totalUsage += dayUsage.total || 0;
        });
        
        // Get initial deposit
        const inpatientSql = 'SELECT deposit FROM tbl_inpatient WHERE id = ?';
        const inpatientResult = await query(inpatientSql, [inpatientId]);
        const initialDeposit = parseFloat(inpatientResult[0]?.deposit || 0);
        
        // Calculate outstanding balance
        const outstandingBalance = totalUsage - initialDeposit;
        
        return {
            initialDeposit,
            totalUsage,
            outstandingBalance: Math.max(0, outstandingBalance)
        };
        
    } catch (error) {
        console.error('Error getting financial summary:', error);
        return {
            initialDeposit: 0,
            totalUsage: 0,
            outstandingBalance: 0
        };
    }
}

/**
 * Get deposit history for an inpatient
 */
async function getDepositHistory(inpatientId) {
    try {
        const depositsSql = `
            SELECT 
                id.*,
                u.name as created_by_name
            FROM tbl_inpatient_deposits id
            LEFT JOIN tbl_user u ON id.created_by = u.id
            WHERE id.inpatient_id = ?
            ORDER BY id.created_at DESC
        `;
        
        const deposits = await query(depositsSql, [inpatientId]);
        
        return deposits;
        
    } catch (error) {
        console.error('Error getting deposit history:', error);
        return [];
    }
}
