/**
 * Inpatient Sales Controller
 * Handles inpatient daily usage tracking, deposits, and discharge settlements
 */

const { query, transaction } = require('../config/database');

const normalizePaymentMethod = (method) => {
    if (!method) return 'cash';

    const value = method.toString().trim().toLowerCase();
    if (!value) return 'cash';

    if (value.includes('cash')) {
        return 'cash';
    }

    if (
        value.includes('card') ||
        value.includes('visa') ||
        value.includes('master') ||
        value.includes('mpu') ||
        value.includes('debit') ||
        value.includes('credit') ||
        value.includes('pos')
    ) {
        return 'card';
    }

    if (value.includes('bank') || value.includes('transfer') || value.includes('ft')) {
        return 'bank_transfer';
    }

    if (value.includes('cheq') || value.includes('check')) {
        return 'check';
    }

    return 'other';
};

let dischargeTableSupportsExtendedFields = null;

async function ensureDischargeTableSchema(conn) {
    if (dischargeTableSupportsExtendedFields !== null) {
        return dischargeTableSupportsExtendedFields;
    }

    try {
        const [columns] = await conn.query(
            "SHOW COLUMNS FROM tbl_inpatient_discharge LIKE 'discount_amount'"
        );
        dischargeTableSupportsExtendedFields = Array.isArray(columns) && columns.length > 0;
    } catch (error) {
        console.warn('Unable to inspect tbl_inpatient_discharge schema:', error.message);
        dischargeTableSupportsExtendedFields = false;
    }

    return dischargeTableSupportsExtendedFields;
}

/**
 * @desc    Get daily usage for an inpatient
 * @route   GET /api/inpatient-sales/usage/:inpatientId
 * @access  Private
 */
exports.getDailyUsage = async (req, res, next) => {
    try {
        const { inpatientId } = req.params;
        const { date } = req.query;
        
        let sql = `
            SELECT 
                u.*,
                CASE 
                    WHEN u.item_type = 'product' THEN p.name
                    WHEN u.item_type = 'service' THEN s.service_name
                END as product_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.sale_price_latli
                    WHEN u.item_type = 'service' THEN s.sale_price
                END as selling_price,
                CASE 
                    WHEN u.item_type = 'product' THEN p.image_path
                    ELSE NULL
                END as image_url
            FROM tbl_inpatient_usage u
            LEFT JOIN tbl_stock p ON u.item_type = 'product' AND u.item_id = p.id
            LEFT JOIN tbl_service s ON u.item_type = 'service' AND u.item_id = s.id
            WHERE u.inpatient_id = ?
        `;
        
        const params = [inpatientId];
        
        if (date) {
            sql += ' AND DATE(u.usage_date) = ?';
            params.push(date);
        }
        
        sql += ' ORDER BY u.created_at DESC';
        
        const usage = await query(sql, params);
        
        res.json({
            success: true,
            data: usage
        });
    } catch (error) {
        console.error('❌ Discharge settlement failed:', error);
        res.status(500).json({
            success: false,
            message: error.message || 'Failed to process discharge. Please contact support.'
        });
    }
};

/**
 * @desc    Add daily usage for an inpatient
 * @route   POST /api/inpatient-sales/usage
 * @access  Private
 */
exports.addDailyUsage = async (req, res, next) => {
    try {
        const usageData = req.body;
        
        if (!Array.isArray(usageData)) {
            return res.status(400).json({
                success: false,
                message: 'Usage data must be an array'
            });
        }
        
        const result = await transaction(async (conn) => {
            const insertedUsage = [];
            
            for (const usage of usageData) {
                const { inpatient_id, product_id, quantity, price, notes, usage_date, item_type, warehouse_id } = usage;
                
                // Validate required fields
                if (!inpatient_id || !product_id || !quantity || !price) {
                    throw new Error('Missing required fields: inpatient_id, product_id, quantity, price');
                }
                
                // Check if inpatient exists and is active
                const inpatientCheck = await conn.query(
                    'SELECT id, patient_id, name FROM tbl_inpatient WHERE id = ? AND end_date IS NULL',
                    [inpatient_id]
                );
                
                if (inpatientCheck.length === 0) {
                    throw new Error('Inpatient not found or already discharged');
                }
                
                // Determine item type and validate item exists
                let itemType = item_type || 'product';
                let itemName = '';
                let itemCheck = null;
                
                if (itemType === 'service') {
                    // Check service availability
                    itemCheck = await conn.query(
                        'SELECT id, service_name as name, sale_price as selling_price FROM tbl_service WHERE id = ?',
                        [product_id]
                    );
                } else {
                    // Check product availability
                    itemCheck = await conn.query(
                        'SELECT id, name, qty as stock_quantity, sale_price_latli as selling_price FROM tbl_stock WHERE id = ?',
                        [product_id]
                    );
                }
                
                if (itemCheck.length === 0) {
                    throw new Error(`${itemType === 'service' ? 'Service' : 'Product'} not found`);
                }
                
                const item = itemCheck[0];
                itemName = item.name;
                
                // Check stock availability for products only
                if (itemType === 'product') {
                    if (warehouse_id) {
                        // Check warehouse stock
                        const [warehouseStock] = await conn.query(
                            'SELECT * FROM tbl_warehouse_stock WHERE warehouse_id = ? AND stock_id = ?',
                            [warehouse_id, product_id]
                        );
                        
                        if (warehouseStock.length === 0) {
                            throw new Error(`Product ${itemName} not available in selected warehouse`);
                        }
                        
                        const ws = warehouseStock[0];
                        const availableQty = ws.quantity - (ws.reserved_quantity || 0);
                        
                        if (availableQty < quantity) {
                            throw new Error(`Insufficient stock for ${itemName} in warehouse. Available: ${availableQty}`);
                        }
                    } else {
                        // No warehouse selected - check main stock
                        if (item.stock_quantity < quantity) {
                            throw new Error(`Insufficient stock for ${itemName}. Available: ${item.stock_quantity}`);
                        }
                    }
                }
                
                // Insert usage record with new structure
                const insertSql = `
                    INSERT INTO tbl_inpatient_usage 
                    (inpatient_id, item_type, item_id, quantity, price, notes, usage_date, created_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
                `;
                
                const insertResult = await conn.query(insertSql, [
                    inpatient_id,
                    itemType,
                    product_id,
                    quantity,
                    price,
                    notes || null,
                    usage_date || new Date().toISOString().split('T')[0]
                ]);
                
                // Update stock quantity for products only
                if (itemType === 'product') {
                    if (warehouse_id) {
                        // Update warehouse stock
                        await conn.query(
                            'UPDATE tbl_warehouse_stock SET quantity = quantity - ? WHERE warehouse_id = ? AND stock_id = ?',
                            [quantity, warehouse_id, product_id]
                        );
                        // Also update main stock for backward compatibility
                        await conn.query(
                            'UPDATE tbl_stock SET qty = qty - ? WHERE id = ?',
                            [quantity, product_id]
                        );
                    } else {
                        // Update main stock only
                        await conn.query(
                            'UPDATE tbl_stock SET qty = qty - ? WHERE id = ?',
                            [quantity, product_id]
                        );
                    }
                }
                
                insertedUsage.push({
                    id: insertResult[0].insertId,
                    inpatient_id,
                    item_type: itemType,
                    item_id: product_id,
                    item_name: itemName,
                    quantity,
                    price,
                    notes,
                    usage_date: usage_date || new Date().toISOString().split('T')[0]
                });
            }
            
            return insertedUsage;
        });
        
        res.json({
            success: true,
            message: 'Usage added successfully',
            data: result
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Collect deposit from inpatient
 * @route   POST /api/inpatient-sales/deposit
 * @access  Private
 */
exports.collectDeposit = async (req, res, next) => {
    try {
        const { inpatient_id, amount, payment_method, notes, is_partial_settlement, cashier } = req.body;
        const normalizedPaymentMethod = normalizePaymentMethod(payment_method);
        
        if (!inpatient_id || !amount || amount <= 0) {
            return res.status(400).json({
                success: false,
                message: 'Invalid deposit data'
            });
        }
        
        // Get cashier name from request body or authenticated user
        const cashierName = cashier || (req.user ? req.user.name : 'Unknown User');
        const createdBy = req.user ? req.user.id : null;
        
        const result = await transaction(async (conn) => {
            // Check if inpatient exists and is active
            const [inpatientCheck] = await conn.query(
                'SELECT id, patient_id, name, deposit FROM tbl_inpatient WHERE id = ? AND end_date IS NULL',
                [inpatient_id]
            );
            
            if (inpatientCheck.length === 0) {
                throw new Error('Inpatient not found or already discharged');
            }
            
            const inpatient = inpatientCheck[0];
            
            // Update inpatient deposit
            const newDeposit = (inpatient.deposit || 0) + amount;
            await conn.query(
                'UPDATE tbl_inpatient SET deposit = ? WHERE id = ?',
                [newDeposit, inpatient_id]
            );
            
            // Create deposit record with cashier name and created_by user ID
            const depositSql = `
                INSERT INTO tbl_inpatient_deposits 
                (inpatient_id, amount, payment_method, notes, cashier_name, created_by, created_at)
                VALUES (?, ?, ?, ?, ?, ?, NOW())
            `;
            
            const [depositResult] = await conn.query(depositSql, [
                inpatient_id,
                amount,
                normalizedPaymentMethod,
                notes || null,
                cashierName,
                createdBy
            ]);
            
            const depositId = depositResult.insertId;
            
            // If this is a partial settlement, archive and delete ALL current usage items
            if (is_partial_settlement) {
                // Get all current usage items (settled or not) to archive and remove
                const [usageItems] = await conn.query(`
                    SELECT u.*, 
                           CASE 
                               WHEN u.item_type = 'product' THEN p.name
                               WHEN u.item_type = 'service' THEN s.service_name
                               ELSE 'Unknown Item'
                           END as product_name
                    FROM tbl_inpatient_usage u
                    LEFT JOIN tbl_stock p ON u.item_type = 'product' AND u.item_id = p.id
                    LEFT JOIN tbl_service s ON u.item_type = 'service' AND u.item_id = s.id
                    WHERE u.inpatient_id = ?
                `, [inpatient_id]);
                
                console.log('Usage items to archive:', usageItems ? usageItems.length : 0);
                
                if (usageItems.length > 0) {
                    let archivedCount = 0;
                    let skippedCount = 0;
                    
                    // Archive all items to settled table (for history/invoice generation)
                    for (const item of usageItems) {
                        // Skip items with NULL or missing critical fields
                        if (!item.id || !item.item_id) {
                            console.warn('Skipping invalid usage item (missing id or item_id):', item);
                            skippedCount++;
                            continue;
                        }
                        
                        console.log('Archiving item:', item.id, item.product_name);
                        
                        // Safely extract values with defaults
                        const originalId = item.id;
                        const itemType = item.item_type || 'product';
                        const itemId = item.item_id;
                        const productName = item.product_name || 'Unknown Item';
                        const qty = parseFloat(item.quantity) || 0;
                        const price = parseFloat(item.price) || 0;
                        const usageDate = item.usage_date || new Date().toISOString().split('T')[0];
                        const createdAt = item.created_at || new Date();
                        
                        try {
                            // Archive to settled table
                            await conn.query(`
                                INSERT INTO tbl_inpatient_usage_settled 
                                (original_usage_id, inpatient_id, item_type, item_id, product_name, 
                                 quantity, price, notes, usage_date, settlement_deposit_id, created_at)
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                            `, [
                                originalId,
                                inpatient_id,
                                itemType,
                                itemId,
                                productName,
                                qty,
                                price,
                                item.notes || null,
                                usageDate,
                                depositId,
                                createdAt
                            ]);
                            archivedCount++;
                        } catch (error) {
                            console.error('Error archiving item:', item.id, error.message);
                            skippedCount++;
                        }
                    }
                    
                    console.log(`Archive summary: ${archivedCount} archived, ${skippedCount} skipped`);
                    
                    // DELETE all usage items to start fresh (both archived and invalid ones)
                    await conn.query(`
                        DELETE FROM tbl_inpatient_usage 
                        WHERE inpatient_id = ?
                    `, [inpatient_id]);
                    
                    console.log(`✅ Archived ${archivedCount} items and cleared all ${usageItems.length} usage items for inpatient ${inpatient_id}`);
                } else {
                    console.log(`No usage items to clear for inpatient ${inpatient_id}`);
                }
            }
            
            return {
                id: depositId,
                inpatient_id,
                amount,
                payment_method: normalizedPaymentMethod,
                original_payment_method: payment_method || normalizedPaymentMethod,
                notes,
                total_deposit: newDeposit,
                items_settled: is_partial_settlement
            };
        });
        
        res.json({
            success: true,
            message: is_partial_settlement ? 'Partial settlement processed successfully' : 'Deposit collected successfully',
            data: result
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get inpatient outstanding balance
 * @route   GET /api/inpatient-sales/outstanding/:inpatientId
 * @access  Private
 */
exports.getOutstandingBalance = async (req, res, next) => {
    try {
        const { inpatientId } = req.params;
        
        // Get total usage cost (ONLY unsettled items)
        const usageSql = `
            SELECT 
                COALESCE(SUM(quantity * price), 0) as total_usage
            FROM tbl_inpatient_usage 
            WHERE inpatient_id = ? AND (is_settled IS NULL OR is_settled = 0)
        `;
        const usageResult = await query(usageSql, [inpatientId]);
        const totalUsage = usageResult[0].total_usage;
        
        // Get total deposits
        const depositSql = `
            SELECT 
                COALESCE(SUM(amount), 0) as total_deposits
            FROM tbl_inpatient_deposits 
            WHERE inpatient_id = ?
        `;
        const depositResult = await query(depositSql, [inpatientId]);
        const totalDeposits = depositResult[0].total_deposits;
        
        // Get bed charges (if applicable)
        const bedSql = `
            SELECT 
                i.daily_rate,
                DATEDIFF(CURDATE(), i.start_date) as days_stayed
            FROM tbl_inpatient i
            WHERE i.id = ? AND i.end_date IS NULL
        `;
        const bedResult = await query(bedSql, [inpatientId]);
        const bedCharges = bedResult.length > 0 ? 
            (bedResult[0].daily_rate || 0) * (bedResult[0].days_stayed || 0) : 0;
        
        const totalCharges = parseFloat(totalUsage) + parseFloat(bedCharges);
        const outstandingBalance = totalCharges - parseFloat(totalDeposits);
        
        res.json({
            success: true,
            data: {
                total_usage: totalUsage,
                bed_charges: bedCharges,
                total_charges: totalCharges,
                total_deposits: totalDeposits,
                outstanding_balance: outstandingBalance
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Settle discharge and generate final bill
 * @route   POST /api/inpatient-sales/discharge
 * @access  Private
 */
exports.settleDischarge = async (req, res, next) => {
    try {
        const { inpatient_id, final_payment, payment_method, notes, discount, tax_rate, cashier } = req.body;
        const normalizedPaymentMethod = normalizePaymentMethod(payment_method);
        
        // Get cashier name from request body or authenticated user
        const cashierName = cashier || (req.user ? req.user.name : 'System');
        
        if (!inpatient_id) {
            return res.status(400).json({
                success: false,
                message: 'Inpatient ID is required'
            });
        }
        
        const result = await transaction(async (conn) => {
            // Get inpatient details
            const inpatientSql = `
                SELECT 
                    i.*,
                    d.name as doctor_name,
                    b.bed_number,
                    b.room_number
                FROM tbl_inpatient i
                LEFT JOIN tbl_doctor d ON i.dr_id = d.id
                LEFT JOIN tbl_beds b ON i.bed_id = b.id
                WHERE i.id = ? AND i.end_date IS NULL
            `;
            const inpatientResult = await conn.query(inpatientSql, [inpatient_id]);
            
            if (inpatientResult.length === 0) {
                throw new Error('Inpatient not found or already discharged');
            }
            
            const inpatient = inpatientResult[0];
            
            console.log('Inpatient data for discharge:', {
                id: inpatient.id,
                patient_id: inpatient.patient_id,
                start_date: inpatient.start_date,
                start_date_type: typeof inpatient.start_date,
                daily_rate: inpatient.daily_rate,
                daily_rate_type: typeof inpatient.daily_rate
            });
            
            // Get total usage
            const usageResult = await conn.query(`
                SELECT COALESCE(SUM(quantity * price), 0) as total_usage
                FROM tbl_inpatient_usage
                WHERE inpatient_id = ?
            `, [inpatient_id]);
            
            // Get total deposits
            const depositsResult = await conn.query(`
                SELECT COALESCE(SUM(amount), 0) as total_deposits
                FROM tbl_inpatient_deposits
                WHERE inpatient_id = ?
            `, [inpatient_id]);
            
            const totalUsage = parseFloat(usageResult[0].total_usage) || 0;
            const totalDeposits = parseFloat(depositsResult[0].total_deposits) || 0;
            
            // Calculate bed charges
            const dailyRate = parseFloat(inpatient.daily_rate || 0);
            if (isNaN(dailyRate)) {
                console.warn('Invalid daily rate for inpatient:', inpatient.daily_rate);
            }
            
            // Handle start date - use current date if invalid
            let startDate;
            if (inpatient.start_date) {
                // Handle both string and Date object inputs
                if (typeof inpatient.start_date === 'string') {
                    startDate = new Date(inpatient.start_date);
                } else if (inpatient.start_date instanceof Date) {
                    startDate = inpatient.start_date;
                } else {
                    startDate = new Date(inpatient.start_date);
                }
                
                if (isNaN(startDate.getTime())) {
                    console.warn('Invalid start date, using current date:', inpatient.start_date);
                    startDate = new Date();
                }
            } else {
                console.warn('No start date found, using current date');
                startDate = new Date();
            }
            
            const currentDate = new Date();
            
            const daysStayed = Math.max(1, Math.ceil((currentDate - startDate) / (1000 * 60 * 60 * 24)));
            const bedCharges = (isNaN(dailyRate) ? 0 : dailyRate) * daysStayed;
            
            const totalCharges = totalUsage + bedCharges;
            const discountAmount = parseFloat(discount) || 0;
            const taxRate = parseFloat(tax_rate) || 0;
            
            // Apply discount to outstanding balance
            const amountAfterDiscount = Math.max(0, totalCharges - totalDeposits - discountAmount);
            
            // Calculate tax on the amount after discount
            const taxAmount = (amountAfterDiscount * taxRate) / 100;
            const outstandingBalance = amountAfterDiscount + taxAmount;
            
            console.log('Discharge calculation details:', {
                inpatient_id,
                dailyRate,
                originalStartDate: inpatient.start_date,
                parsedStartDate: startDate.toISOString(),
                daysStayed,
                totalUsage,
                bedCharges,
                totalCharges,
                totalDeposits,
                discountAmount,
                taxRate,
                taxAmount,
                amountAfterDiscount,
                outstandingBalance
            });
            
            // Validate values before database update
            if (isNaN(totalCharges) || isNaN(outstandingBalance)) {
                throw new Error(`Invalid calculation: totalCharges=${totalCharges}, outstandingBalance=${outstandingBalance}`);
            }
            
            console.log('Discharge calculation:', {
                totalUsage,
                bedCharges,
                totalCharges,
                totalDeposits,
                outstandingBalance
            });
            
            // Update inpatient discharge
            const dischargeDate = new Date().toISOString().split('T')[0];
            await conn.query(
                'UPDATE tbl_inpatient SET end_date = ?, final_bill = ?, discharge_notes = ? WHERE id = ?',
                [dischargeDate, Math.round(outstandingBalance), notes || null, inpatient_id]
            );
            
            const hasExtendedFields = await ensureDischargeTableSchema(conn);

            if (hasExtendedFields) {
                await conn.query(`
                    INSERT INTO tbl_inpatient_discharge 
                    (inpatient_id, discharge_date, total_usage, bed_charges, total_charges, total_deposits, 
                     discount_amount, tax_rate, tax_amount, amount_after_discount, outstanding_balance, 
                     final_payment, payment_method, discharge_notes, created_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
                    ON DUPLICATE KEY UPDATE
                    discharge_date = VALUES(discharge_date),
                    total_usage = VALUES(total_usage),
                    bed_charges = VALUES(bed_charges),
                    total_charges = VALUES(total_charges),
                    total_deposits = VALUES(total_deposits),
                    discount_amount = VALUES(discount_amount),
                    tax_rate = VALUES(tax_rate),
                    tax_amount = VALUES(tax_amount),
                    amount_after_discount = VALUES(amount_after_discount),
                    outstanding_balance = VALUES(outstanding_balance),
                    final_payment = VALUES(final_payment),
                    payment_method = VALUES(payment_method),
                    discharge_notes = VALUES(discharge_notes)
                `, [
                    inpatient_id, dischargeDate, Math.round(totalUsage), Math.round(bedCharges), 
                    Math.round(totalCharges), Math.round(totalDeposits), Math.round(discountAmount), 
                    taxRate, Math.round(taxAmount), Math.round(amountAfterDiscount), 
                    Math.round(outstandingBalance), Math.round(final_payment || 0), 
                    normalizedPaymentMethod, notes || null
                ]);
            } else {
                await conn.query(`
                    INSERT INTO tbl_inpatient_discharge 
                    (inpatient_id, discharge_date, total_usage, bed_charges, total_charges, total_deposits, 
                     outstanding_balance, final_payment, payment_method, discharge_notes, created_at)
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW())
                    ON DUPLICATE KEY UPDATE
                    discharge_date = VALUES(discharge_date),
                    total_usage = VALUES(total_usage),
                    bed_charges = VALUES(bed_charges),
                    total_charges = VALUES(total_charges),
                    total_deposits = VALUES(total_deposits),
                    outstanding_balance = VALUES(outstanding_balance),
                    final_payment = VALUES(final_payment),
                    payment_method = VALUES(payment_method),
                    discharge_notes = VALUES(discharge_notes)
                `, [
                    inpatient_id, dischargeDate, Math.round(totalUsage), Math.round(bedCharges), 
                    Math.round(totalCharges), Math.round(totalDeposits), 
                    Math.round(outstandingBalance), Math.round(final_payment || 0), 
                    normalizedPaymentMethod, notes || null
                ]);
            }
            
            // Mark all remaining unsettled usage items as settled (for final discharge)
            try {
            await conn.query(`
                UPDATE tbl_inpatient_usage 
                SET is_settled = 1, settled_at = NOW()
                WHERE inpatient_id = ? AND (is_settled IS NULL OR is_settled = 0)
            `, [inpatient_id]);
            } catch (usageError) {
                if (usageError.code === 'ER_BAD_FIELD_ERROR') {
                    await conn.query(`
                        UPDATE tbl_inpatient_usage 
                        SET is_settled = 1
                        WHERE inpatient_id = ? AND (is_settled IS NULL OR is_settled = 0)
                    `, [inpatient_id]);
                } else {
                    throw usageError;
                }
            }
            
            // Create tbl_sale entries for inpatient sales to include in sales reports
            try {
                const customerName = inpatient.name ? `${inpatient.name} (PID: ${inpatient.patient_id || inpatient_id})` : `Inpatient ${inpatient_id}`;
                await createInpatientSalesEntries(
                    conn, 
                    inpatient_id, 
                    inpatient, 
                    totalUsage, 
                    bedCharges, 
                    discountAmount, 
                    taxAmount,
                    amountAfterDiscount,
                    normalizedPaymentMethod,
                    cashierName,
                    customerName
                );
            } catch (entryError) {
                console.error('⚠️ Failed to create inpatient sales entries:', entryError.message);
            }
            
            // Free up the bed
            if (inpatient.bed_id) {
                await conn.query(
                    'UPDATE tbl_beds SET status = "Available", current_patient_id = NULL, admission_date = NULL WHERE id = ?',
                    [inpatient.bed_id]
                );
            }
            
            // Record final payment if provided
            if (final_payment && final_payment > 0) {
                await conn.query(`
                    INSERT INTO tbl_inpatient_deposits 
                    (inpatient_id, amount, payment_method, notes, created_at)
                    VALUES (?, ?, ?, ?, NOW())
                `, [inpatient_id, final_payment, normalizedPaymentMethod, 'Final payment on discharge']);
            }
            
            return {
                inpatient_id,
                patient_name: inpatient.name,
                patient_id: inpatient.patient_id,
                discharge_date: dischargeDate,
                total_usage: Math.round(totalUsage),
                bed_charges: Math.round(bedCharges),
                total_charges: Math.round(totalCharges),
                total_deposits: Math.round(totalDeposits),
                discount_amount: Math.round(discountAmount),
                tax_rate: taxRate,
                tax_amount: Math.round(taxAmount),
                amount_after_discount: Math.round(amountAfterDiscount),
                outstanding_balance: Math.round(outstandingBalance - (final_payment || 0)),
                final_payment: Math.round(final_payment || 0),
                payment_method: normalizedPaymentMethod,
                original_payment_method: payment_method || normalizedPaymentMethod
            };
        });
        
        res.json({
            success: true,
            message: 'Patient discharged successfully',
            data: result
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get discharge report
 * @route   GET /api/inpatient-sales/discharge-report/:inpatientId
 * @access  Private
 */
exports.getDischargeReport = async (req, res, next) => {
    try {
        const { inpatientId } = req.params;
        
        // Get inpatient details
        const inpatientSql = `
            SELECT 
                i.*,
                d.name as doctor_name,
                b.bed_number,
                b.room_number
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            WHERE i.id = ?
        `;
        const inpatientResult = await query(inpatientSql, [inpatientId]);
        
        if (inpatientResult.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Inpatient not found'
            });
        }
        
        const inpatient = inpatientResult[0];
        
        // Get usage details
        const usageSql = `
            SELECT 
                u.*,
                CASE 
                    WHEN u.item_type = 'product' THEN p.name
                    WHEN u.item_type = 'service' THEN s.service_name
                END as product_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.sale_price_latli
                    WHEN u.item_type = 'service' THEN s.sale_price
                END as selling_price
            FROM tbl_inpatient_usage u
            LEFT JOIN tbl_stock p ON u.item_type = 'product' AND u.item_id = p.id
            LEFT JOIN tbl_service s ON u.item_type = 'service' AND u.item_id = s.id
            WHERE u.inpatient_id = ?
            ORDER BY u.usage_date, u.created_at
        `;
        const usageResult = await query(usageSql, [inpatientId]);
        
        // Get deposit history
        const depositSql = `
            SELECT * FROM tbl_inpatient_deposits 
            WHERE inpatient_id = ?
            ORDER BY created_at
        `;
        const depositResult = await query(depositSql, [inpatientId]);
        
        // Get discharge record with discount and tax information
        // Try to get discharge record, but handle case where columns might not exist
        let dischargeResult = [];
        try {
            const dischargeSql = `
                SELECT * FROM tbl_inpatient_discharge 
                WHERE inpatient_id = ?
            `;
            dischargeResult = await query(dischargeSql, [inpatientId]);
        } catch (error) {
            console.log('Discharge table or columns might not exist, using defaults:', error.message);
            dischargeResult = [];
        }
        
        // Calculate totals
        const totalUsage = usageResult.reduce((sum, item) => sum + (item.quantity * item.price), 0);
        const totalDeposits = depositResult.reduce((sum, item) => sum + item.amount, 0);
        const bedCharges = (inpatient.daily_rate || 0) * Math.max(1, Math.ceil((new Date() - new Date(inpatient.start_date)) / (1000 * 60 * 60 * 24)));
        const totalCharges = totalUsage + bedCharges;
        
        // Use discharge record data if available, otherwise calculate
        let discountAmount = 0;
        let taxRate = 0;
        let taxAmount = 0;
        let amountAfterDiscount = 0;
        let outstandingBalance = totalCharges - totalDeposits;
        
        if (dischargeResult.length > 0) {
            const discharge = dischargeResult[0];
            discountAmount = parseFloat(discharge.discount_amount) || 0;
            taxRate = parseFloat(discharge.tax_rate) || 0;
            taxAmount = parseFloat(discharge.tax_amount) || 0;
            amountAfterDiscount = parseFloat(discharge.amount_after_discount) || 0;
            outstandingBalance = parseFloat(discharge.outstanding_balance) || 0;
            
            console.log('Discharge record found:', {
                discount_amount: discharge.discount_amount,
                tax_rate: discharge.tax_rate,
                tax_amount: discharge.tax_amount,
                amount_after_discount: discharge.amount_after_discount
            });
        } else {
            console.log('No discharge record found, using calculated values');
        }
        
        res.json({
            success: true,
            data: {
                inpatient,
                usage: usageResult,
                deposits: depositResult,
                summary: {
                    total_usage: totalUsage,
                    bed_charges: bedCharges,
                    total_charges: totalCharges,
                    total_deposits: totalDeposits,
                    discount_amount: discountAmount,
                    tax_rate: taxRate,
                    tax_amount: taxAmount,
                    amount_after_discount: amountAfterDiscount,
                    outstanding_balance: outstandingBalance
                }
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update usage record
 * @route   PUT /api/inpatient-sales/usage/:id
 * @access  Private
 */
exports.updateUsage = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { quantity, price, notes, usage_date } = req.body;
        
        // Validate required fields
        if (!quantity || !price) {
            return res.status(400).json({
                success: false,
                message: 'Quantity and price are required'
            });
        }
        
        const result = await transaction(async (conn) => {
            // Get current usage record
            const currentUsage = await conn.query(
                'SELECT * FROM tbl_inpatient_usage WHERE id = ?',
                [id]
            );
            
            if (currentUsage.length === 0) {
                throw new Error('Usage record not found');
            }
            
            const usage = currentUsage[0];
            
            // Calculate quantity difference
            const oldQuantity = parseFloat(usage.quantity);
            const newQuantity = parseFloat(quantity);
            const quantityDiff = newQuantity - oldQuantity;
            
            // Update stock for products only
            if (usage.item_type === 'product') {
                // Check if there's enough stock for the increase
                if (quantityDiff > 0) {
                    const stockCheck = await conn.query(
                        'SELECT qty FROM tbl_stock WHERE id = ?',
                        [usage.item_id]
                    );
                    
                    if (stockCheck.length === 0) {
                        throw new Error('Product not found');
                    }
                    
                    if (stockCheck[0].qty < quantityDiff) {
                        throw new Error(`Insufficient stock. Available: ${stockCheck[0].qty}`);
                    }
                }
                
                // Update stock quantity
                await conn.query(
                    'UPDATE tbl_stock SET qty = qty - ? WHERE id = ?',
                    [quantityDiff, usage.item_id]
                );
            }
            
            // Update usage record
            await conn.query(
                'UPDATE tbl_inpatient_usage SET quantity = ?, price = ?, notes = ?, usage_date = ? WHERE id = ?',
                [quantity, price, notes || null, usage_date || usage.usage_date, id]
            );
            
            return { id, quantityDiff };
        });
        
        res.json({
            success: true,
            message: 'Usage record updated successfully',
            data: result
        });
        
    } catch (error) {
        next(error);
    }
}

/**
 * @desc    Delete usage record
 * @route   DELETE /api/inpatient-sales/usage/:id
 * @access  Private
 */
exports.deleteUsage = async (req, res, next) => {
    try {
        const { id } = req.params;
        
        const result = await transaction(async (conn) => {
            // Get usage record
            const usage = await conn.query(
                'SELECT * FROM tbl_inpatient_usage WHERE id = ?',
                [id]
            );
            
            if (usage.length === 0) {
                throw new Error('Usage record not found');
            }
            
            const usageRecord = usage[0];
            
            // Restore stock for products only
            if (usageRecord.item_type === 'product') {
                await conn.query(
                    'UPDATE tbl_stock SET qty = qty + ? WHERE id = ?',
                    [usageRecord.quantity, usageRecord.item_id]
                );
            }
            
            // Delete usage record
            await conn.query(
                'DELETE FROM tbl_inpatient_usage WHERE id = ?',
                [id]
            );
            
            return { id, restoredQuantity: usageRecord.quantity };
        });
        
        res.json({
            success: true,
            message: 'Usage record deleted successfully',
            data: result
        });
        
    } catch (error) {
        next(error);
    }
};

/**
 * Create tbl_sale entries for inpatient sales to include in sales reports
 * @param {Object} conn - Database connection
 * @param {number} inpatient_id - Inpatient ID
 * @param {Object} inpatient - Inpatient data
 * @param {number} totalUsage - Total usage amount
 * @param {number} bedCharges - Bed charges
 * @param {number} discountAmount - Discount amount
 * @param {number} taxAmount - Tax amount
 */
async function createInpatientSalesEntries(conn, inpatient_id, inpatient, totalUsage, bedCharges, discountAmount, taxAmount, totalAmount = null, paymentMethod = 'Cash', cashierName = 'System', customerName = null) {
    try {
        const toNumber = (value) => {
            const num = parseFloat(value);
            return Number.isFinite(num) ? num : 0;
        };

        // Get next voucher number for inpatient sales
        const [voucherResult] = await conn.query('SELECT COALESCE(MAX(VNo), 0) + 1 as voucher FROM tbl_sale');
        const voucherNo = (voucherResult && voucherResult[0] && voucherResult[0].voucher) ? voucherResult[0].voucher : 1;
        
        // Format customer name
        const formattedCustomerName = customerName || (inpatient.name ? `${inpatient.name} (PID: ${inpatient.patient_id || inpatient_id})` : `Inpatient ${inpatient_id}`);
        
        // Calculate totals if not provided
        const subtotal = toNumber(totalUsage) + toNumber(bedCharges);
        const finalTotal = totalAmount !== null ? toNumber(totalAmount) : (subtotal - toNumber(discountAmount) + toNumber(taxAmount));
        
        // Get all usage items for this inpatient
        const [usageItems] = await conn.query(`
            SELECT 
                u.*,
                CASE 
                    WHEN u.item_type = 'product' THEN p.name
                    WHEN u.item_type = 'service' THEN s.service_name
                END as product_name,
                CASE 
                    WHEN u.item_type = 'product' THEN p.barcode
                    WHEN u.item_type = 'service' THEN s.barcode
                END as barcode,
                CASE 
                    WHEN u.item_type = 'product' THEN p.cat_id
                    WHEN u.item_type = 'service' THEN s.cat_id
                END as cat_id,
                CASE 
                    WHEN u.item_type = 'product' THEN p.cost_price
                    WHEN u.item_type = 'service' THEN s.buy_price
                END as cost_price
            FROM tbl_inpatient_usage u
            LEFT JOIN tbl_stock p ON u.item_type = 'product' AND u.item_id = p.id
            LEFT JOIN tbl_service s ON u.item_type = 'service' AND u.item_id = s.id
            WHERE u.inpatient_id = ?
            ORDER BY u.created_at ASC
        `, [inpatient_id]);
        
        // Create sale entries for each usage item
        for (const item of usageItems) {
            const quantity = toNumber(item.quantity);
            const salePrice = toNumber(item.price);
            const costPrice = toNumber(item.cost_price);
            const revenue = Number((quantity * salePrice).toFixed(2));
            const profit = Number((revenue - (costPrice * quantity)).toFixed(2));
            
            // Determine the correct SaleType based on item_type
            const saleType = item.item_type === 'service' ? 'service' : 'Inpatient';
            
            await conn.query(`
                INSERT INTO tbl_sale 
                (cat_id, Barcode, Name, SaleType, SalePrice, dis, Qty, Total, Profit, VNo, Date, Cashier, cost_price, Refer)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURDATE(), ?, ?, ?)
            `, [
                item.cat_id || 0,
                item.barcode || `INP-${inpatient_id}-${item.id}`,
                item.product_name || 'Inpatient Item',
                saleType,
                salePrice,
                0, // No individual item discount
                quantity,
                revenue,
                profit,
                voucherNo,
                cashierName,
                costPrice,
                `Inpatient ID: ${inpatient_id}`
            ]);
        }
        
        // Create a summary entry for bed charges if applicable
        const sanitizedBedCharges = toNumber(bedCharges);
        if (sanitizedBedCharges > 0) {
            await conn.query(`
                INSERT INTO tbl_sale 
                (cat_id, Barcode, Name, SaleType, SalePrice, dis, Qty, Total, Profit, VNo, Date, Cashier, cost_price, Refer)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURDATE(), ?, ?, ?)
            `, [
                0, // No category for bed charges
                `BED-${inpatient_id}`,
                `Bed Charges (${inpatient.bed_number || 'N/A'})`,
                'Inpatient',
                sanitizedBedCharges,
                0,
                1,
                sanitizedBedCharges,
                0, // No profit on bed charges
                voucherNo,
                cashierName,
                0,
                `Inpatient ID: ${inpatient_id} - Bed Charges`
            ]);
        }
        
        // Create discount entry if applicable
        const sanitizedDiscount = toNumber(discountAmount);
        if (sanitizedDiscount > 0) {
            await conn.query(`
                INSERT INTO tbl_sale 
                (cat_id, Barcode, Name, SaleType, SalePrice, dis, Qty, Total, Profit, VNo, Date, Cashier, cost_price, Refer)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURDATE(), ?, ?, ?)
            `, [
                0,
                `DISC-${inpatient_id}`,
                'Discount Applied',
                'Inpatient',
                -sanitizedDiscount, // Negative amount for discount
                0,
                1,
                -sanitizedDiscount,
                0,
                voucherNo,
                cashierName,
                0,
                `Inpatient ID: ${inpatient_id} - Discount`
            ]);
        }
        
        // Create tax entry if applicable
        const sanitizedTax = toNumber(taxAmount);
        if (sanitizedTax > 0) {
            await conn.query(`
                INSERT INTO tbl_sale 
                (cat_id, Barcode, Name, SaleType, SalePrice, dis, Qty, Total, Profit, VNo, Date, Cashier, cost_price, Refer)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURDATE(), ?, ?, ?)
            `, [
                0,
                `TAX-${inpatient_id}`,
                'Tax Applied',
                'Inpatient',
                sanitizedTax,
                0,
                1,
                sanitizedTax,
                0,
                voucherNo,
                cashierName,
                0,
                `Inpatient ID: ${inpatient_id} - Tax`
            ]);
        }
        
        // Create voucher entry in tbl_voucher table (similar to regular POS sales)
        try {
            const voucherSql = `INSERT INTO tbl_voucher 
                               (id, user_name, sub_total, actual_cost, dis, tax, customer_name, Date, status, payment, doctor_id, doctor_name, patient_id)
                               VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), 1, ?, ?, ?, ?)`;
            
            await conn.query(voucherSql, [
                voucherNo,
                cashierName,
                subtotal,
                finalTotal,
                toNumber(discountAmount),
                toNumber(taxAmount),
                formattedCustomerName,
                paymentMethod || 'Cash',
                inpatient.dr_id || null,
                inpatient.doctor_name || null,
                inpatient.patient_id || null
            ]);
            
            console.log(`Created voucher entry for inpatient sale: Voucher ${voucherNo}, Customer: ${formattedCustomerName}`);
        } catch (voucherError) {
            console.error('Error creating voucher entry for inpatient sale:', voucherError);
            // Don't throw error - voucher creation failure shouldn't break the discharge process
        }
        
        console.log(`Created ${usageItems.length + (sanitizedBedCharges > 0 ? 1 : 0) + (sanitizedDiscount > 0 ? 1 : 0) + (sanitizedTax > 0 ? 1 : 0)} sale entries for inpatient ${inpatient_id} with voucher ${voucherNo}`);
        console.log('Inpatient sales entries created:', {
            usageItems: usageItems.length,
            bedCharges: sanitizedBedCharges > 0 ? 1 : 0,
            discountAmount: sanitizedDiscount > 0 ? 1 : 0,
            taxAmount: sanitizedTax > 0 ? 1 : 0,
            voucherNo: voucherNo,
            customerName: formattedCustomerName
        });
        
    } catch (error) {
        console.error('Error creating inpatient sales entries:', error);
        // Don't throw error to avoid breaking the discharge process
        // Just log the error and continue
    }
}