/**
 * Inpatient Controller
 * Handles inpatient admissions with bed assignment
 */

const { query, transaction } = require('../config/database');

/**
 * @desc    Get all inpatients
 * @route   GET /api/inpatients
 * @access  Private
 */
exports.getAllInpatients = async (req, res, next) => {
    try {
        const limit = parseInt(req.query.limit) || 100;
        const page = parseInt(req.query.page) || 1;
        const offset = (page - 1) * limit;

        const sql = `
            SELECT 
                i.*,
                d.name as dr_name,
                b.bed_number,
                b.room_number,
                b.bed_type,
                b.daily_rate,
                dept.dept_name as department_name
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            LEFT JOIN tbl_departments dept ON b.department_id = dept.id
            WHERE i.end_date IS NULL
            ORDER BY i.start_date DESC
            LIMIT ? OFFSET ?
        `;

        const inpatients = await query(sql, [limit, offset]);

        // Get total count
        const countSql = 'SELECT COUNT(*) as total FROM tbl_inpatient WHERE end_date IS NULL';
        const countResult = await query(countSql);

        res.json({
            success: true,
            count: inpatients.length,
            total: countResult[0].total,
            data: inpatients
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get single inpatient
 * @route   GET /api/inpatients/:id
 * @access  Private
 */
exports.getInpatient = async (req, res, next) => {
    try {
        const sql = `
            SELECT 
                i.*,
                d.name as dr_name,
                b.bed_number,
                b.room_number,
                b.bed_type,
                b.daily_rate,
                dept.dept_name as department_name
            FROM tbl_inpatient i
            LEFT JOIN tbl_doctor d ON i.dr_id = d.id
            LEFT JOIN tbl_beds b ON i.bed_id = b.id
            LEFT JOIN tbl_departments dept ON b.department_id = dept.id
            WHERE i.id = ?
        `;

        const inpatients = await query(sql, [req.params.id]);

        if (inpatients.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Inpatient not found'
            });
        }

        res.json({
            success: true,
            data: inpatients[0]
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create new inpatient admission
 * @route   POST /api/inpatients
 * @access  Private
 */
exports.createInpatient = async (req, res, next) => {
    try {
        const {
            name,
            age,
            phone,
            address,
            room_no,
            bed_id,
            bed_number,
            bed_type,
            daily_rate,
            dr_id,
            deposit,
            start_date,
            remark
        } = req.body;

        // Validate required fields
        if (!name || !bed_id) {
            return res.status(400).json({
                success: false,
                message: 'Name and bed selection are required'
            });
        }

        // Check if patient is already admitted (by name and phone)
        if (phone) {
            const existingPatient = await query(
                'SELECT id, name, patient_id, bed_number, room_no FROM tbl_inpatient WHERE phone = ? AND end_date IS NULL',
                [phone]
            );
            
            if (existingPatient.length > 0) {
                const patient = existingPatient[0];
                return res.status(400).json({
                    success: false,
                    message: `⚠️ Patient ${patient.name} (ID: ${patient.patient_id}) is already admitted in Bed ${patient.bed_number}, Room ${patient.room_no || 'N/A'}. Please discharge the patient first before creating a new admission.`,
                    data: {
                        existing_patient: patient,
                        action_required: 'discharge_existing'
                    }
                });
            }
        }

        // Check bed availability BEFORE starting transaction
        console.log(`🔍 Checking bed availability for bed_id: ${bed_id}`);
        
        const bedCheck = await query(
            'SELECT bed_number, room_number, status, current_patient_id FROM tbl_beds WHERE id = ?',
            [bed_id]
        );
        
        console.log('📊 Bed check result:', bedCheck);

        if (bedCheck.length === 0) {
            return res.status(404).json({
                success: false,
                message: '❌ Bed not found. Please select a valid bed.'
            });
        }

        const bed = bedCheck[0];
        
        // Check if bed is available - return appropriate warning
        if (bed.status !== 'Available') {
            let warningMessage = `⚠️ Bed ${bed.bed_number} in Room ${bed.room_number} is not available`;
            let statusInfo = {
                bed_id: bed_id,
                bed_number: bed.bed_number,
                room_number: bed.room_number,
                current_status: bed.status
            };
            
            if (bed.status === 'Occupied' && bed.current_patient_id) {
                // Get patient details for better warning message
                const patientInfo = await query(
                    'SELECT name, patient_id, phone FROM tbl_inpatient WHERE patient_id = ? AND end_date IS NULL',
                    [bed.current_patient_id]
                );
                
                if (patientInfo.length > 0) {
                    const currentPatient = patientInfo[0];
                    warningMessage = `⚠️ Bed ${bed.bed_number} in Room ${bed.room_number} is already occupied`;
                    statusInfo.current_patient = {
                        name: currentPatient.name,
                        patient_id: currentPatient.patient_id,
                        phone: currentPatient.phone
                    };
                    statusInfo.message = `This bed is currently occupied by ${currentPatient.name} (ID: ${currentPatient.patient_id})`;
                    statusInfo.action_required = 'Please select a different bed or discharge the current patient first';
                } else {
                    warningMessage = `⚠️ Bed ${bed.bed_number} in Room ${bed.room_number} is marked as occupied`;
                    statusInfo.message = `This bed is marked as occupied by patient ${bed.current_patient_id}`;
                    statusInfo.action_required = 'Please select a different bed';
                }
            } else if (bed.status === 'Cleaning') {
                warningMessage = `⚠️ Bed ${bed.bed_number} in Room ${bed.room_number} is being cleaned`;
                statusInfo.message = 'This bed is currently being cleaned and sanitized';
                statusInfo.action_required = 'Please wait a few moments for cleaning to complete or select a different bed';
                statusInfo.estimated_time = 'Available in approximately 2 minutes';
            } else if (bed.status === 'Maintenance') {
                warningMessage = `⚠️ Bed ${bed.bed_number} in Room ${bed.room_number} is under maintenance`;
                statusInfo.message = 'This bed is currently under maintenance and cannot be used';
                statusInfo.action_required = 'Please select a different bed';
            } else if (bed.status === 'Reserved') {
                warningMessage = `⚠️ Bed ${bed.bed_number} in Room ${bed.room_number} is reserved`;
                statusInfo.message = 'This bed has been reserved for another patient';
                statusInfo.action_required = 'Please select a different bed';
            }
            
            return res.status(400).json({
                success: false,
                message: warningMessage,
                warning: statusInfo
            });
        }

        // Use transaction to ensure data consistency
        const result = await transaction(async (conn) => {

            // Generate patient ID if not provided
            const countSql = 'SELECT COUNT(*) as count FROM tbl_inpatient';
            const countResult = await conn.query(countSql);
            const patient_id = `IP${String(countResult[0][0].count + 1).padStart(6, '0')}`;

            // Create inpatient record
            const inpatientSql = `
                INSERT INTO tbl_inpatient 
                (name, age, phone, address, room_no, bed_id, bed_number, bed_type, daily_rate, 
                 dr_id, deposit, start_date, remark, patient_id)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            `;

            const inpatientResult = await conn.query(inpatientSql, [
                name,
                age || null,
                phone || null,
                address || null,
                room_no || null,
                bed_id,
                bed_number || null,
                bed_type || null,
                daily_rate || 0,
                dr_id || null,
                deposit || 0,
                start_date || new Date().toISOString().split('T')[0],
                remark || null,
                patient_id
            ]);

            // Update bed status to occupied
            await conn.query(
                'UPDATE tbl_beds SET status = ?, current_patient_id = ?, admission_date = NOW() WHERE id = ?',
                ['Occupied', patient_id, bed_id]
            );

            return {
                id: inpatientResult[0].insertId,
                patient_id,
                bed_id,
                bed_number,
                room_no
            };
        });

        res.status(201).json({
            success: true,
            message: 'Patient admitted successfully',
            data: result
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update inpatient
 * @route   PUT /api/inpatients/:id
 * @access  Private
 */
exports.updateInpatient = async (req, res, next) => {
    try {
        const {
            name,
            age,
            phone,
            address,
            room_no,
            bed_id,
            bed_number,
            bed_type,
            daily_rate,
            dr_id,
            deposit,
            start_date,
            remark
        } = req.body;

        // Use transaction for bed changes
        const result = await transaction(async (conn) => {
            // Get current inpatient data
            const currentInpatient = await conn.query(
                'SELECT bed_id, patient_id FROM tbl_inpatient WHERE id = ?',
                [req.params.id]
            );

            if (currentInpatient[0].length === 0) {
                throw new Error('Inpatient not found');
            }

            const currentBedId = currentInpatient[0][0].bed_id;
            const patientId = currentInpatient[0][0].patient_id;

            // If bed is being changed
            if (bed_id && bed_id !== currentBedId) {
                // Free up current bed
                await conn.query(
                    'UPDATE tbl_beds SET status = ?, current_patient_id = NULL, admission_date = NULL WHERE id = ?',
                    ['Available', currentBedId]
                );

                // Check if new bed is available
                const bedCheck = await conn.query(
                    'SELECT status, current_patient_id FROM tbl_beds WHERE id = ?',
                    [bed_id]
                );

                if (bedCheck[0].length === 0) {
                    throw new Error('New bed not found');
                }

                if (bedCheck[0][0].status !== 'Available') {
                    throw new Error('New bed is not available');
                }

                if (bedCheck[0][0].current_patient_id) {
                    throw new Error('New bed is already occupied');
                }

                // Assign new bed
                await conn.query(
                    'UPDATE tbl_beds SET status = ?, current_patient_id = ?, admission_date = NOW() WHERE id = ?',
                    ['Occupied', patientId, bed_id]
                );
            }

            // Update inpatient record
            const updateSql = `
                UPDATE tbl_inpatient SET 
                name = ?, age = ?, phone = ?, address = ?, room_no = ?, 
                bed_id = ?, bed_number = ?, bed_type = ?, daily_rate = ?, 
                dr_id = ?, deposit = ?, start_date = ?, remark = ?
                WHERE id = ?
            `;

            await conn.query(updateSql, [
                name,
                age || null,
                phone || null,
                address || null,
                room_no || null,
                bed_id || currentBedId,
                bed_number || null,
                bed_type || null,
                daily_rate || 0,
                dr_id || null,
                deposit || 0,
                start_date || null,
                remark || null,
                req.params.id
            ]);

            return { id: req.params.id };
        });

        res.json({
            success: true,
            message: 'Inpatient updated successfully',
            data: result
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Discharge inpatient
 * @route   DELETE /api/inpatients/:id
 * @access  Private
 */
exports.dischargeInpatient = async (req, res, next) => {
    try {
        const { discharge_date, discharge_notes } = req.body;

        // Use transaction to free up bed
        const result = await transaction(async (conn) => {
            // Get inpatient data
            const inpatient = await conn.query(
                'SELECT bed_id, patient_id FROM tbl_inpatient WHERE id = ?',
                [req.params.id]
            );

            if (inpatient[0].length === 0) {
                throw new Error('Inpatient not found');
            }

            const bedId = inpatient[0][0].bed_id;
            const patientId = inpatient[0][0].patient_id;

            // Update inpatient record with discharge info
            await conn.query(
                'UPDATE tbl_inpatient SET end_date = ?, remark = CONCAT(COALESCE(remark, ""), " | Discharged: ", ?) WHERE id = ?',
                [
                    discharge_date || new Date().toISOString().split('T')[0],
                    discharge_notes || 'Discharged',
                    req.params.id
                ]
            );

            // Free up the bed and set to a safe status
            if (bedId) {
                let nextStatus = 'Cleaning';
                try {
                    const statusCol = await conn.query("SHOW COLUMNS FROM tbl_beds LIKE 'status'");
                    const colType = statusCol && statusCol[0] && statusCol[0][0] && statusCol[0][0].Type ? statusCol[0][0].Type : '';
                    if (colType && colType.startsWith('enum(')) {
                        const match = colType.match(/enum\((.*)\)/i);
                        if (match && match[1]) {
                            const allowed = match[1]
                                .split(',')
                                .map(s => s.trim())
                                .map(s => s.startsWith("'") ? s.slice(1) : s)
                                .map(s => s.endsWith("'") ? s.slice(0, -1) : s);
                            // Prefer Cleaning -> Maintenance -> Available -> first
                            nextStatus = allowed.includes('Cleaning') ? 'Cleaning' : (allowed.includes('Maintenance') ? 'Maintenance' : (allowed.includes('Available') ? 'Available' : allowed[0]));
                        }
                    }
                } catch (e) {
                    nextStatus = 'Available';
                }

                await conn.query(
                    'UPDATE tbl_beds SET status = ?, current_patient_id = NULL, admission_date = NULL WHERE id = ?',
                    [nextStatus, bedId]
                );

                // Auto-switch to Available after delay if we set Cleaning/Maintenance
                if (nextStatus !== 'Available') {
                    setTimeout(async () => {
                        try {
                            await query(
                                "UPDATE tbl_beds SET status = 'Available' WHERE id = ?",
                                [bedId]
                            );
                            console.log(`✅ Bed ${bedId} set to Available after discharge delay`);
                        } catch (error) {
                            console.error('Error auto-setting bed available:', error);
                        }
                    }, 2 * 60 * 1000);
                }
            }

            return { id: req.params.id, patient_id: patientId };
        });

        res.json({
            success: true,
            message: 'Patient discharged successfully',
            data: result
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get available beds for admission
 * @route   GET /api/inpatients/available-beds
 * @access  Private
 */
exports.getAvailableBeds = async (req, res, next) => {
    try {
        const { department_id } = req.query;
        
        let sql = `
            SELECT b.id, b.bed_number, b.room_number, b.bed_type, b.daily_rate, 
                   d.dept_name as department_name
            FROM tbl_beds b
            LEFT JOIN tbl_departments d ON b.department_id = d.id
            WHERE b.status = 'Available'
        `;
        const params = [];
        
        if (department_id) {
            sql += ' AND b.department_id = ?';
            params.push(department_id);
        }
        
        sql += ' ORDER BY d.dept_name, b.room_number, b.bed_number';
        
        const availableBeds = await query(sql, params);
        
        res.json({
            success: true,
            count: availableBeds.length,
            data: availableBeds
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Clean bed manually
 * @route   PATCH /api/inpatients/beds/:bedId/clean
 * @access  Private
 */
exports.cleanBed = async (req, res, next) => {
    try {
        const { bedId } = req.params;
        
        // Check if bed exists and is in cleaning status
        const bedCheck = await query(
            'SELECT id, bed_number, room_number, status FROM tbl_beds WHERE id = ?',
            [bedId]
        );
        
        if (bedCheck.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Bed not found'
            });
        }
        
        const bed = bedCheck[0];
        
        if (bed.status !== 'Cleaning') {
            return res.status(400).json({
                success: false,
                message: `Bed ${bed.bed_number} in Room ${bed.room_number} is not in cleaning status (Current: ${bed.status})`
            });
        }
        
        // Set bed to available
        await query(
            'UPDATE tbl_beds SET status = ? WHERE id = ?',
            ['Available', bedId]
        );
        
        res.json({
            success: true,
            message: `Bed ${bed.bed_number} in Room ${bed.room_number} has been cleaned and is now available`,
            data: {
                bed_id: bedId,
                bed_number: bed.bed_number,
                room_number: bed.room_number,
                status: 'Available'
            }
        });
    } catch (error) {
        next(error);
    }
};
