const { query, transaction } = require('../config/database');

/**
 * @desc    Generate invoice data for a completed sale
 * @route   GET /api/invoice/:voucherNo
 * @access  Private
 */
exports.getInvoiceData = async (req, res, next) => {
    try {
        const { voucherNo } = req.params;

        if (!voucherNo) {
            return res.status(400).json({
                success: false,
                message: 'Voucher number is required'
            });
        }

        // Get voucher information
        const voucherSql = `
            SELECT v.*, 
                   d.name as doctor_name,
                   p.name as patient_name
            FROM tbl_voucher v
            LEFT JOIN tbl_doctor d ON v.doctor_id = d.id
            LEFT JOIN tbl_patient p ON v.patient_id = p.id
            WHERE v.id = ?
        `;
        
        const [voucherResult] = await query(voucherSql, [voucherNo]);
        
        if (voucherResult.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Voucher not found'
            });
        }

        const voucher = voucherResult[0];

        // Get sale items for this voucher
        const itemsSql = `
            SELECT s.*, 
                   st.name as product_name,
                   st.barcode,
                   st.sale_price,
                   st.cost_price
            FROM tbl_sale s
            LEFT JOIN tbl_stock st ON s.Barcode = st.barcode
            WHERE s.VNo = ?
            ORDER BY s.id
        `;
        
        const [itemsResult] = await query(itemsSql, [voucherNo]);

        // Get shop information from settings
        const shopInfoSql = 'SELECT * FROM tbl_shopinfo WHERE id = 1';
        const [shopInfoResult] = await query(shopInfoSql);
        
        const shopInfo = shopInfoResult.length > 0 ? {
            shopName: shopInfoResult[0].name || 'Clinic Pro',
            shopType: shopInfoResult[0].type || 'Medical Center',
            shopAddress: shopInfoResult[0].address || '123 Medical Street, Health City',
            shopPhone: shopInfoResult[0].ph_no ? `Phone: ${shopInfoResult[0].ph_no}` : 'Phone: +1-234-567-8900',
            shopEmail: shopInfoResult[0].email || '',
            logoPath: shopInfoResult[0].photo_path || '',
            thankYouMessage: shopInfoResult[0].thanks || 'Thank you for your visit!',
            voucherFooter: shopInfoResult[0].remark || 'Please keep this receipt for your records',
            printerType: shopInfoResult[0].printer || '80mm',
            vouchersPerPage: shopInfoResult[0].printer_voucher_count || 1
        } : {
            shopName: 'Clinic Pro',
            shopType: 'Medical Center',
            shopAddress: '123 Medical Street, Health City',
            shopPhone: 'Phone: +1-234-567-8900',
            shopEmail: '',
            logoPath: '',
            thankYouMessage: 'Thank you for your visit!',
            voucherFooter: 'Please keep this receipt for your records',
            printerType: '80mm',
            vouchersPerPage: 1
        };

        // Format items for invoice
        const items = itemsResult.map(item => ({
            name: item.Name || item.product_name || 'Unknown Item',
            barcode: item.Barcode,
            quantity: item.Qty,
            price: parseFloat(item.SalePrice),
            amount: parseFloat(item.Total),
            cost_price: parseFloat(item.cost_price || 0)
        }));

        // Calculate totals
        const subtotal = items.reduce((sum, item) => sum + item.amount, 0);
        const discount = parseFloat(voucher.dis || 0);
        const tax = parseFloat(voucher.tax || 0);
        const total = parseFloat(voucher.actual_cost || 0);

        const invoiceData = {
            voucherNo: voucher.id,
            date: voucher.Date,
            cashier: voucher.user_name,
            customerName: voucher.customer_name || 'Walk-in',
            doctorName: voucher.doctor_name || '',
            patientName: voucher.patient_name || '',
            items: items,
            subtotal: subtotal,
            discount: discount,
            tax: tax,
            total: total,
            paid: total, // Assuming full payment for now
            paymentMethod: voucher.payment || 'Cash',
            shopInfo: shopInfo
        };

        res.json({
            success: true,
            data: invoiceData
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Generate invoice HTML with data
 * @route   POST /api/invoice/generate
 * @access  Private
 */
exports.generateInvoice = async (req, res, next) => {
    try {
        const { voucherNo } = req.body;

        if (!voucherNo) {
            return res.status(400).json({
                success: false,
                message: 'Voucher number is required'
            });
        }

        // Get invoice data
        const invoiceResponse = await this.getInvoiceData({ params: { voucherNo } }, res, next);
        
        if (!invoiceResponse) {
            return; // Response already sent by getInvoiceData
        }

        // Return the invoice data for frontend processing
        res.json({
            success: true,
            data: invoiceResponse.data
        });

    } catch (error) {
        next(error);
    }
};
