/**
 * Laboratory Management Controller
 * Professional lab ordering, result management, and quality control
 */

const { query, transaction } = require('../config/database');

/**
 * @desc    Get all lab tests from catalog
 * @route   GET /api/laboratory/catalog
 * @access  Private
 */
exports.getLabCatalog = async (req, res, next) => {
    try {
        const { category, active } = req.query;
        let sql = 'SELECT * FROM tbl_lab_test_catalog WHERE 1=1';
        const params = [];

        if (category) {
            sql += ' AND category = ?';
            params.push(category);
        }
        if (active !== undefined) {
            sql += ' AND is_active = ?';
            params.push(active === 'true' ? 1 : 0);
        }

        sql += ' ORDER BY category, test_name';
        const tests = await query(sql, params);

        res.json({
            success: true,
            count: tests.length,
            data: tests
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get lab test categories
 * @route   GET /api/laboratory/categories
 * @access  Private
 */
exports.getLabCategories = async (req, res, next) => {
    try {
        const sql = 'SELECT DISTINCT category FROM tbl_lab_test_catalog WHERE category IS NOT NULL AND is_active = 1 ORDER BY category';
        const categories = await query(sql);

        res.json({
            success: true,
            data: categories.map(c => c.category)
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create lab order
 * @route   POST /api/laboratory/orders
 * @access  Private
 */
exports.createLabOrder = async (req, res, next) => {
    try {
        const {
            patient_id,
            doctor_id,
            priority = 'Routine',
            clinical_info,
            tests, // Array of test IDs
            referring_doctor_id,
            referring_doctor_name
        } = req.body;

        if (!patient_id || !tests || tests.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'Patient ID and at least one test are required'
            });
        }

        // Generate order number
        const year = new Date().getFullYear();
        const seqSql = 'UPDATE tbl_lab_sequence SET last_order_number = last_order_number + 1 WHERE year = ?';
        await query(seqSql, [year]);

        const seqResult = await query('SELECT last_order_number FROM tbl_lab_sequence WHERE year = ?', [year]);
        const orderNum = seqResult[0].last_order_number;
        const order_number = `LAB${year}${String(orderNum).padStart(6, '0')}`;

        // Calculate total amount
        const testIds = tests.map(t => t.test_id || t);
        const testsSql = `SELECT SUM(price) as total FROM tbl_lab_test_catalog WHERE id IN (${testIds.join(',')})`;
        const totalResult = await query(testsSql);
        const total_amount = totalResult[0].total || 0;

        // Insert order
        const orderSql = `INSERT INTO tbl_lab_orders 
            (order_number, patient_id, doctor_id, priority, clinical_info, total_amount, created_by, status, referring_doctor_id, referring_doctor_name) 
            VALUES (?, ?, ?, ?, ?, ?, ?, 'Ordered', ?, ?)`;
        const orderResult = await query(orderSql, [
            order_number,
            patient_id,
            doctor_id ?? null,
            priority,
            clinical_info ?? null,
            total_amount,
            req.user.id,
            referring_doctor_id || null,
            referring_doctor_name || null
        ]);

        // Insert order details
        for (const test of tests) {
            const test_id = test.test_id || test;
            const testInfo = await query('SELECT test_code, test_name, price FROM tbl_lab_test_catalog WHERE id = ?', [test_id]);
            
            if (testInfo.length > 0) {
                const detailSql = `INSERT INTO tbl_lab_order_details 
                    (order_id, test_id, test_code, test_name, price) 
                    VALUES (?, ?, ?, ?, ?)`;
                await query(detailSql, [
                    orderResult.insertId,
                    test_id,
                    testInfo[0].test_code,
                    testInfo[0].test_name,
                    testInfo[0].price
                ]);
            }
        }

        // Create commission if there's a referring doctor
        if (referring_doctor_id && referring_doctor_id !== '' && referring_doctor_id !== 'null') {
            try {
                // Get referring doctor's commission settings
                const [referringDoctor] = await query(
                    'SELECT commission_type, commission_value FROM tbl_referring_doctor WHERE id = ? AND active = 1',
                    [referring_doctor_id]
                );
                
                if (referringDoctor && referringDoctor.length > 0) {
                    const doctor = referringDoctor[0];
                    let commission_amount = 0;
                    let commission_rate = null;
                    
                    if (doctor.commission_type === 'percentage') {
                        commission_amount = (total_amount * doctor.commission_value) / 100;
                        commission_rate = doctor.commission_value;
                    } else if (doctor.commission_type === 'fixed') {
                        commission_amount = doctor.commission_value;
                    }
                    
                    if (commission_amount > 0) {
                        // Get patient name for commission record
                        const [patientInfo] = await query('SELECT name FROM tbl_patient WHERE patient_id = ?', [patient_id]);
                        const patient_name = patientInfo && patientInfo.length > 0 ? patientInfo[0].name : 'Unknown';

                        // Create commission record
                        await query(
                            `INSERT INTO tbl_referral_commission 
                            (referring_doctor_id, transaction_type, lab_order_id, voucher_no, patient_id, patient_name,
                             transaction_date, transaction_amount, commission_type, commission_rate, commission_amount,
                             payment_status, created_by)
                            VALUES (?, 'lab', ?, ?, ?, ?, CURDATE(), ?, ?, ?, ?, 'pending', ?)`,
                            [referring_doctor_id, orderResult.insertId, order_number, patient_id, patient_name,
                             total_amount, doctor.commission_type, commission_rate, commission_amount, req.user.id]
                        );
                        
                        // Update referring doctor totals
                        await query(
                            `UPDATE tbl_referring_doctor 
                            SET total_referrals = total_referrals + 1,
                                total_revenue = total_revenue + ?,
                                total_commission_earned = total_commission_earned + ?
                            WHERE id = ?`,
                            [total_amount, commission_amount, referring_doctor_id]
                        );
                    }
                }
            } catch (commissionError) {
                console.error('Error creating lab commission:', commissionError);
                // Don't fail the order if commission creation fails
            }
        }

        res.status(201).json({
            success: true,
            data: {
                id: orderResult.insertId,
                order_number: order_number,
                total_amount: total_amount
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get lab orders
 * @route   GET /api/laboratory/orders
 * @access  Private
 */
exports.getLabOrders = async (req, res, next) => {
    try {
        const { patient_id, status, from, to, limit = 100 } = req.query;
        let sql = `SELECT o.*, p.name as patient_name, d.name as doctor_name, 
            o.referring_doctor_name
            FROM tbl_lab_orders o
            LEFT JOIN tbl_patient p ON o.patient_id = p.patient_id
            LEFT JOIN tbl_doctor d ON o.doctor_id = d.id
            WHERE 1=1`;
        const params = [];

        if (patient_id) {
            sql += ' AND o.patient_id = ?';
            params.push(patient_id);
        }
        if (status) {
            sql += ' AND o.status = ?';
            params.push(status);
        }
        if (from) {
            sql += ' AND o.order_date >= ?';
            params.push(from);
        }
        if (to) {
            sql += ' AND o.order_date <= ?';
            params.push(to);
        }

        sql += ' ORDER BY o.order_date DESC LIMIT ?';
        params.push(parseInt(limit));

        const orders = await query(sql, params);

        // Get order details for each order
        for (let order of orders) {
            const detailsSql = `SELECT * FROM tbl_lab_order_details WHERE order_id = ?`;
            order.tests = await query(detailsSql, [order.id]);
        }

        res.json({
            success: true,
            count: orders.length,
            data: orders
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get single lab order
 * @route   GET /api/laboratory/orders/:id
 * @access  Private
 */
exports.getLabOrder = async (req, res, next) => {
    try {
        const sql = `SELECT o.*, p.name as patient_name, p.age, p.phone, 
            d.name as doctor_name, u.name as created_by_name,
            o.referring_doctor_name
            FROM tbl_lab_orders o
            LEFT JOIN tbl_patient p ON o.patient_id = p.patient_id
            LEFT JOIN tbl_doctor d ON o.doctor_id = d.id
            LEFT JOIN tbl_user u ON o.created_by = u.id
            WHERE o.id = ?`;
        
        const orders = await query(sql, [req.params.id]);
        
        if (orders.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Lab order not found'
            });
        }

        const order = orders[0];

        // Get order details with results
        const detailsSql = `SELECT d.*, r.result_value, r.flag, r.status as result_status,
            r.performed_date, r.verified_date
            FROM tbl_lab_order_details d
            LEFT JOIN tbl_lab_results r ON d.id = r.order_detail_id
            WHERE d.order_id = ?`;
        order.tests = await query(detailsSql, [order.id]);

        res.json({
            success: true,
            data: order
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update specimen collection
 * @route   PATCH /api/laboratory/orders/:id/collect
 * @access  Private
 */
exports.collectSpecimen = async (req, res, next) => {
    try {
        const { specimen_ids } = req.body; // Array of {order_detail_id, specimen_id}
        
        const sql = `UPDATE tbl_lab_orders SET 
            status = 'Specimen_Collected',
            collection_date = NOW(),
            collected_by = ?
            WHERE id = ?`;
        
        await query(sql, [req.user.id, req.params.id]);

        // Update specimen IDs for tests
        if (specimen_ids && specimen_ids.length > 0) {
            for (const spec of specimen_ids) {
                await query(
                    'UPDATE tbl_lab_order_details SET specimen_id = ?, status = "In_Progress" WHERE id = ?',
                    [spec.specimen_id, spec.order_detail_id]
                );
            }
        }

        res.json({
            success: true,
            message: 'Specimen collection recorded'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Enter lab results
 * @route   POST /api/laboratory/results
 * @access  Private
 */
exports.enterLabResult = async (req, res, next) => {
    try {
        const {
            order_detail_id,
            order_id,
            patient_id,
            test_id,
            result_value,
            result_numeric,
            result_text,
            unit,
            notes
        } = req.body;

        if (!order_detail_id || !test_id) {
            return res.status(400).json({
                success: false,
                message: 'Order detail ID and test ID are required'
            });
        }

        // Get test reference ranges
        const testInfo = await query(
            'SELECT normal_range_min, normal_range_max, critical_low, critical_high, unit FROM tbl_lab_test_catalog WHERE id = ?',
            [test_id]
        );

        let flag = 'Normal';
        let is_critical = 0;
        let normal_range = null;

        if (testInfo.length > 0 && result_numeric !== null && result_numeric !== undefined) {
            const { normal_range_min, normal_range_max, critical_low, critical_high } = testInfo[0];
            normal_range = `${normal_range_min || ''} - ${normal_range_max || ''}`;

            // Determine flag
            if (critical_low && result_numeric < critical_low) {
                flag = 'Critical_Low';
                is_critical = 1;
            } else if (critical_high && result_numeric > critical_high) {
                flag = 'Critical_High';
                is_critical = 1;
            } else if (normal_range_min && result_numeric < normal_range_min) {
                flag = 'Low';
            } else if (normal_range_max && result_numeric > normal_range_max) {
                flag = 'High';
            }
        }

        // Insert result
        const sql = `INSERT INTO tbl_lab_results 
            (order_detail_id, order_id, patient_id, test_id, result_value, result_numeric, 
            result_text, unit, normal_range, flag, is_critical, notes, performed_by, performed_date, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), 'Preliminary')`;
        
        const result = await query(sql, [
            order_detail_id ?? null,
            order_id ?? null,
            patient_id ?? null,
            test_id ?? null,
            result_value ?? null,
            result_numeric ?? null,
            result_text ?? null,
            unit || testInfo[0]?.unit || null,
            normal_range ?? null,
            flag ?? null,
            is_critical ?? 0,
            notes ?? null,
            req.user.id
        ]);

        // Update order detail status
        await query('UPDATE tbl_lab_order_details SET status = "Completed" WHERE id = ?', [order_detail_id]);

        // Check if all tests completed, update order
        const remainingTests = await query(
            'SELECT COUNT(*) as count FROM tbl_lab_order_details WHERE order_id = ? AND status != "Completed"',
            [order_id]
        );
        
        if (remainingTests[0].count === 0) {
            await query('UPDATE tbl_lab_orders SET status = "Completed" WHERE id = ?', [order_id]);
        } else {
            await query('UPDATE tbl_lab_orders SET status = "In_Progress" WHERE id = ?', [order_id]);
        }

        // Create critical alert if needed
        if (is_critical) {
            const testName = await query('SELECT test_name FROM tbl_lab_test_catalog WHERE id = ?', [test_id]);
            const alertSql = `INSERT INTO tbl_lab_critical_alerts 
                (result_id, patient_id, test_name, critical_value, status)
                VALUES (?, ?, ?, ?, 'Pending')`;
            await query(alertSql, [
                result.insertId,
                patient_id,
                testName[0].test_name,
                `${result_value} ${unit || testInfo[0]?.unit || ''}`
            ]);
        }

        res.status(201).json({
            success: true,
            data: {
                id: result.insertId,
                flag: flag,
                is_critical: is_critical
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get lab results for patient
 * @route   GET /api/laboratory/results/patient/:patientId
 * @access  Private
 */
exports.getPatientResults = async (req, res, next) => {
    try {
        const sql = `SELECT r.*, t.test_name, t.category, o.order_number, o.order_date
            FROM tbl_lab_results r
            JOIN tbl_lab_test_catalog t ON r.test_id = t.id
            JOIN tbl_lab_orders o ON r.order_id = o.id
            WHERE r.patient_id = ?
            ORDER BY r.performed_date DESC`;
        
        const results = await query(sql, [req.params.patientId]);

        res.json({
            success: true,
            count: results.length,
            data: results
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Verify lab result (supervisor approval)
 * @route   PATCH /api/laboratory/results/:id/verify
 * @access  Private
 */
exports.verifyLabResult = async (req, res, next) => {
    try {
        const sql = `UPDATE tbl_lab_results SET 
            verified_by = ?,
            verified_date = NOW(),
            status = 'Final'
            WHERE id = ?`;
        
        await query(sql, [req.user.id, req.params.id]);

        res.json({
            success: true,
            message: 'Result verified successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get critical alerts
 * @route   GET /api/laboratory/alerts
 * @access  Private
 */
exports.getCriticalAlerts = async (req, res, next) => {
    try {
        const { status = 'Pending' } = req.query;
        
        const sql = `SELECT a.*, p.name as patient_name, p.phone
            FROM tbl_lab_critical_alerts a
            LEFT JOIN tbl_patient p ON a.patient_id = p.patient_id
            WHERE a.status = ?
            ORDER BY a.alert_date DESC
            LIMIT 50`;
        
        const alerts = await query(sql, [status]);

        res.json({
            success: true,
            count: alerts.length,
            data: alerts
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Acknowledge critical alert
 * @route   PATCH /api/laboratory/alerts/:id/acknowledge
 * @access  Private
 */
exports.acknowledgeAlert = async (req, res, next) => {
    try {
        const { action_taken } = req.body;
        
        const sql = `UPDATE tbl_lab_critical_alerts SET 
            acknowledged_by = ?,
            acknowledged_date = NOW(),
            action_taken = ?,
            status = 'Acknowledged'
            WHERE id = ?`;
        
        await query(sql, [req.user.id, action_taken, req.params.id]);

        res.json({
            success: true,
            message: 'Alert acknowledged'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get lab statistics
 * @route   GET /api/laboratory/stats
 * @access  Private
 */
exports.getLabStats = async (req, res, next) => {
    try {
        const { from, to } = req.query;
        const today = new Date().toISOString().split('T')[0];
        
        const whereClause = from && to 
            ? `WHERE order_date BETWEEN '${from}' AND '${to}'`
            : `WHERE DATE(order_date) = '${today}'`;

        const stats = await query(`
            SELECT 
                COUNT(*) as total_orders,
                SUM(CASE WHEN status = 'Completed' THEN 1 ELSE 0 END) as completed,
                SUM(CASE WHEN status = 'In_Progress' THEN 1 ELSE 0 END) as in_progress,
                SUM(CASE WHEN status = 'Ordered' THEN 1 ELSE 0 END) as pending,
                SUM(total_amount) as total_revenue
            FROM tbl_lab_orders
            ${whereClause}
        `);

        const criticalAlerts = await query(`
            SELECT COUNT(*) as count 
            FROM tbl_lab_critical_alerts 
            WHERE status = 'Pending' AND DATE(alert_date) = '${today}'
        `);

        res.json({
            success: true,
            data: {
                ...stats[0],
                critical_alerts: criticalAlerts[0].count
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Add test to catalog
 * @route   POST /api/laboratory/catalog
 * @access  Private
 */
exports.addTestToCatalog = async (req, res, next) => {
    try {
        const {
            test_code,
            test_name,
            category,
            sample_type,
            price,
            tat_hours,
            unit,
            method,
            normal_range_male,
            normal_range_female,
            critical_values,
            preparation_required,
            is_active
        } = req.body;

        if (!test_code || !test_name || !category || !sample_type || !price) {
            return res.status(400).json({
                success: false,
                message: 'Required fields: test_code, test_name, category, sample_type, price'
            });
        }

        const sql = `INSERT INTO tbl_lab_test_catalog 
            (test_code, test_name, category, sample_type, price, tat_hours, unit, method, 
            normal_range_male, normal_range_female, critical_values, preparation_required, is_active)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;

        const result = await query(sql, [
            test_code,
            test_name,
            category,
            sample_type,
            price,
            tat_hours || null,
            unit || null,
            method || null,
            normal_range_male || null,
            normal_range_female || null,
            critical_values || null,
            preparation_required || null,
            is_active !== undefined ? is_active : 1
        ]);

        res.status(201).json({
            success: true,
            data: { id: result.insertId },
            message: 'Test added to catalog successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update test in catalog
 * @route   PUT /api/laboratory/catalog/:id
 * @access  Private
 */
exports.updateTestInCatalog = async (req, res, next) => {
    try {
        const { id } = req.params;
        const {
            test_code,
            test_name,
            category,
            sample_type,
            price,
            tat_hours,
            unit,
            method,
            normal_range_male,
            normal_range_female,
            critical_values,
            preparation_required,
            is_active
        } = req.body;

        const sql = `UPDATE tbl_lab_test_catalog SET
            test_code = ?, test_name = ?, category = ?, sample_type = ?, price = ?,
            tat_hours = ?, unit = ?, method = ?, normal_range_male = ?, normal_range_female = ?,
            critical_values = ?, preparation_required = ?, is_active = ?
            WHERE id = ?`;

        await query(sql, [
            test_code,
            test_name,
            category,
            sample_type,
            price,
            tat_hours || null,
            unit || null,
            method || null,
            normal_range_male || null,
            normal_range_female || null,
            critical_values || null,
            preparation_required || null,
            is_active !== undefined ? is_active : 1,
            id
        ]);

        res.json({
            success: true,
            message: 'Test updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Delete test from catalog
 * @route   DELETE /api/laboratory/catalog/:id
 * @access  Private
 */
exports.deleteTestFromCatalog = async (req, res, next) => {
    try {
        const { id } = req.params;

        // Check if test is being used in any orders
        const usageCheck = await query(
            'SELECT COUNT(*) as count FROM tbl_lab_order_details WHERE test_id = ?',
            [id]
        );

        if (usageCheck[0].count > 0) {
            return res.status(400).json({
                success: false,
                message: `Cannot delete test. It is being used in ${usageCheck[0].count} lab order(s). Consider deactivating instead.`
            });
        }

        await query('DELETE FROM tbl_lab_test_catalog WHERE id = ?', [id]);

        res.json({
            success: true,
            message: 'Test deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

