const { query } = require('../config/database');
const fs = require('fs').promises;
const path = require('path');

const backupsDir = path.join(__dirname, '../../backups');

async function ensureBackupsDir() {
    try {
        await fs.access(backupsDir);
    } catch {
        await fs.mkdir(backupsDir, { recursive: true });
    }
}

// Create a backup file and return filename and path
async function createBackupFile() {
    await ensureBackupsDir();

    const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
    const filename = `clinic_backup_${timestamp}.sql`;
    const filepath = path.join(backupsDir, filename);

    // Get all tables
    const tables = await query('SHOW TABLES');
    const tableName = Object.keys(tables[0])[0];
    const tableList = tables.map(t => t[tableName]);

    let sqlDump = `-- Clinic Pro V3 Database Backup\n-- Created: ${new Date().toISOString()}\n-- Tables: ${tableList.length}\n\nSET FOREIGN_KEY_CHECKS=0;\n\n`;

    for (const table of tableList) {
        const createTableResult = await query(`SHOW CREATE TABLE \`${table}\``);
        const createTableSql = createTableResult[0]['Create Table'];
        sqlDump += `-- Table: ${table}\n`;
        sqlDump += `DROP TABLE IF EXISTS \`${table}\`;\n`;
        sqlDump += `${createTableSql};\n\n`;

        const rows = await query(`SELECT * FROM \`${table}\``);
        if (rows.length > 0) {
            sqlDump += `-- Data for table: ${table}\n`;
            const columns = Object.keys(rows[0]);
            const columnList = columns.map(col => `\`${col}\``).join(', ');
            for (let i = 0; i < rows.length; i += 100) {
                const batch = rows.slice(i, i + 100);
                sqlDump += `INSERT INTO \`${table}\` (${columnList}) VALUES\n`;
                const values = batch.map(row => {
                    const vals = columns.map(col => {
                        const val = row[col];
                        if (val === null) return 'NULL';
                        if (typeof val === 'number') return val;
                        if (val instanceof Date) {
                            if (isNaN(val.getTime())) return 'NULL';
                            return `'${val.toISOString().slice(0, 19).replace('T', ' ')}'`;
                        }
                        const escaped = String(val).replace(/\\/g, '\\\\').replace(/'/g, "\\'");
                        return `'${escaped}'`;
                    });
                    return `(${vals.join(', ')})`;
                });
                sqlDump += values.join(',\n');
                sqlDump += ';\n\n';
            }
        }
    }

    sqlDump += 'SET FOREIGN_KEY_CHECKS=1;\n';
    await fs.writeFile(filepath, sqlDump, 'utf8');

    return { filename, filepath, tables: tableList.length };
}

/**
 * @desc Reset database: create backup, truncate all tables except auth/settings
 * @route POST /api/maintenance/reset-database
 * @access Private (admin)
 */
exports.resetDatabase = async (req, res, next) => {
    try {
        // 1) Create backup
        const backupInfo = await createBackupFile();

        // 2) Find all tables
        const tables = await query('SHOW TABLES');
        const key = Object.keys(tables[0])[0];
        const tableList = tables.map(t => t[key]);

        // 3) Exclusions: keep users/auth and shop info
        const exclude = new Set([
            'tbl_user', 'tbl_users', 'users', 'user', 'tbl_login', 'tbl_roles', 'tbl_permissions',
            'tbl_shopinfo', 'migrations', 'sequelize_meta'
        ]);

        // 4) Disable FK checks during truncate
        await query('SET FOREIGN_KEY_CHECKS=0');
        let truncated = 0;
        for (const table of tableList) {
            if (exclude.has(table)) continue;
            try {
                await query(`TRUNCATE TABLE \`${table}\``);
                truncated++;
            } catch (e) {
                // fallback: delete rows if truncate fails due to permissions
                await query(`DELETE FROM \`${table}\``);
            }
        }
        await query('SET FOREIGN_KEY_CHECKS=1');

        res.json({
            success: true,
            message: 'Database reset completed',
            data: {
                backup: backupInfo.filename,
                truncated_tables: truncated
            }
        });
    } catch (error) {
        next(error);
    }
};


