const { query } = require('../config/database');

/**
 * Get all notifications for the current user
 */
exports.getNotifications = async (req, res, next) => {
    try {
        const userId = req.user?.id || req.user?.user_id;
        const notifications = [];
        
        // Get read notification IDs from database
        let readNotificationIds = [];
        try {
            const readNotifications = await query(`
                SELECT notification_id 
                FROM tbl_notification_read 
                WHERE user_id = ?
            `, [userId || 0]);
            readNotificationIds = readNotifications.map(r => r.notification_id);
        } catch (dbError) {
            // Table might not exist yet, that's okay
            console.warn('Could not fetch read notifications:', dbError.message);
        }
        
        // 1. Low Stock Alerts (quantity <= 10)
        const lowStockProducts = await query(`
            SELECT id, name, barcode, qty, unit_name 
            FROM tbl_stock 
            WHERE qty <= 10 AND qty > 0
            ORDER BY qty ASC
            LIMIT 10
        `);
        
        lowStockProducts.forEach(product => {
            const notificationId = `low-stock-${product.id}`;
            notifications.push({
                id: notificationId,
                type: 'low_stock',
                severity: product.qty <= 5 ? 'critical' : 'warning',
                title: 'Low Stock Alert',
                message: `${product.name} is running low (${product.qty} ${product.unit_name || 'units'} remaining)`,
                data: product,
                timestamp: new Date(),
                read: readNotificationIds.includes(notificationId)
            });
        });

        // 2. Out of Stock Alerts
        const outOfStockProducts = await query(`
            SELECT id, name, barcode 
            FROM tbl_stock 
            WHERE qty = 0
            LIMIT 10
        `);
        
        outOfStockProducts.forEach(product => {
            const notificationId = `out-of-stock-${product.id}`;
            notifications.push({
                id: notificationId,
                type: 'out_of_stock',
                severity: 'critical',
                title: 'Out of Stock',
                message: `${product.name} is out of stock`,
                data: product,
                timestamp: new Date(),
                read: readNotificationIds.includes(notificationId)
            });
        });

        // 3. Expired Products
        const expiredProducts = await query(`
            SELECT p.id, p.name, p.barcode, p.expire_date, p.qty
            FROM tbl_purchase p
            WHERE p.expire_date < CURDATE() AND p.qty > 0
            ORDER BY p.expire_date DESC
            LIMIT 10
        `);
        
        expiredProducts.forEach(product => {
            const notificationId = `expired-${product.id}`;
            notifications.push({
                id: notificationId,
                type: 'expired',
                severity: 'critical',
                title: 'Expired Product',
                message: `${product.name} expired on ${new Date(product.expire_date).toLocaleDateString()}`,
                data: product,
                timestamp: product.expire_date,
                read: readNotificationIds.includes(notificationId)
            });
        });

        // 4. Products Expiring Soon (within 30 days)
        const expiringProducts = await query(`
            SELECT p.id, p.name, p.barcode, p.expire_date, p.qty,
                   DATEDIFF(p.expire_date, CURDATE()) as days_remaining
            FROM tbl_purchase p
            WHERE p.expire_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)
            AND p.qty > 0
            ORDER BY p.expire_date ASC
            LIMIT 10
        `);
        
        expiringProducts.forEach(product => {
            const daysRemaining = product.days_remaining;
            const severity = daysRemaining <= 7 ? 'critical' : daysRemaining <= 15 ? 'warning' : 'info';
            const notificationId = `expiring-${product.id}`;
            
            notifications.push({
                id: notificationId,
                type: 'expiring_soon',
                severity: severity,
                title: 'Expiring Soon',
                message: `${product.name} will expire in ${daysRemaining} days (${new Date(product.expire_date).toLocaleDateString()})`,
                data: product,
                timestamp: product.expire_date,
                read: readNotificationIds.includes(notificationId)
            });
        });

        // 5. Today's Appointments
        const todayAppointments = await query(`
            SELECT id, name, phone, time, dr_name as doctor_name
            FROM tbl_appointment
            WHERE DATE(time) = CURDATE()
            ORDER BY time ASC
        `);
        
        todayAppointments.forEach(appointment => {
            const notificationId = `appointment-${appointment.id}`;
            notifications.push({
                id: notificationId,
                type: 'appointment',
                severity: 'info',
                title: 'Today\'s Appointment',
                message: `${appointment.name} at ${new Date(appointment.time).toLocaleTimeString()} with ${appointment.doctor_name || 'Doctor'}`,
                data: appointment,
                timestamp: new Date(),
                read: readNotificationIds.includes(notificationId)
            });
        });

        // 6. Pending Inpatient Payments
        const pendingPayments = await query(`
            SELECT id, name, room_no, deposit, patient_id
            FROM tbl_inpatient
            WHERE end_date IS NULL AND deposit < 100000
            ORDER BY deposit ASC
            LIMIT 5
        `);
        
        pendingPayments.forEach(patient => {
            const notificationId = `payment-${patient.id}`;
            notifications.push({
                id: notificationId,
                type: 'payment',
                severity: 'warning',
                title: 'Low Deposit Alert',
                message: `Patient ${patient.name} (${patient.patient_id}) in Room ${patient.room_no} has low deposit: ${patient.deposit} Ks`,
                data: patient,
                timestamp: new Date(),
                read: readNotificationIds.includes(notificationId)
            });
        });

        // 7. Recent Sales (Today)
        const recentSalesCount = await query(`
            SELECT COUNT(*) as count, SUM(Total) as total_amount
            FROM tbl_sale
            WHERE DATE(Date) = CURDATE()
        `);
        
        if (recentSalesCount[0].count > 0) {
            const notificationId = 'sales-today';
            notifications.push({
                id: notificationId,
                type: 'sales_summary',
                severity: 'success',
                title: 'Today\'s Sales',
                message: `${recentSalesCount[0].count} sales completed today. Total: ${recentSalesCount[0].total_amount.toLocaleString()} Ks`,
                data: recentSalesCount[0],
                timestamp: new Date(),
                read: readNotificationIds.includes(notificationId)
            });
        }

        // 8. System Messages
        const systemNotificationId = 'system-welcome';
        notifications.push({
            id: systemNotificationId,
            type: 'system',
            severity: 'info',
            title: 'Welcome to Clinic Pro V3',
            message: 'Your healthcare management system is running smoothly',
            timestamp: new Date(),
            read: readNotificationIds.includes(systemNotificationId)
        });

        // Sort by severity and timestamp
        const severityOrder = { critical: 0, warning: 1, success: 2, info: 3 };
        notifications.sort((a, b) => {
            if (a.severity !== b.severity) {
                return severityOrder[a.severity] - severityOrder[b.severity];
            }
            return new Date(b.timestamp) - new Date(a.timestamp);
        });

        // Get counts by type
        const counts = {
            total: notifications.length,
            critical: notifications.filter(n => n.severity === 'critical').length,
            warning: notifications.filter(n => n.severity === 'warning').length,
            info: notifications.filter(n => n.severity === 'info').length,
            unread: notifications.filter(n => !n.read).length
        };

        res.status(200).json({
            success: true,
            counts: counts,
            data: notifications
        });
    } catch (error) {
        console.error('Notification error:', error);
        next(error);
    }
};

/**
 * Get notification counts only
 */
exports.getNotificationCounts = async (req, res, next) => {
    try {
        const counts = {
            low_stock: 0,
            out_of_stock: 0,
            expired: 0,
            expiring_soon: 0,
            appointments_today: 0,
            total: 0
        };

        // Low stock
        const lowStock = await query('SELECT COUNT(*) as count FROM tbl_stock WHERE qty <= 10 AND qty > 0');
        counts.low_stock = lowStock[0].count;

        // Out of stock
        const outOfStock = await query('SELECT COUNT(*) as count FROM tbl_stock WHERE qty = 0');
        counts.out_of_stock = outOfStock[0].count;

        // Expired
        const expired = await query('SELECT COUNT(*) as count FROM tbl_purchase WHERE expire_date < CURDATE() AND qty > 0');
        counts.expired = expired[0].count;

        // Expiring soon
        const expiring = await query('SELECT COUNT(*) as count FROM tbl_purchase WHERE expire_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) AND qty > 0');
        counts.expiring_soon = expiring[0].count;

        // Today's appointments
        const appointments = await query('SELECT COUNT(*) as count FROM tbl_appointment WHERE DATE(time) = CURDATE()');
        counts.appointments_today = appointments[0].count;

        // Total
        counts.total = counts.low_stock + counts.out_of_stock + counts.expired + counts.expiring_soon + counts.appointments_today;

        res.status(200).json({
            success: true,
            data: counts
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Get stock alerts specifically
 */
exports.getStockAlerts = async (req, res, next) => {
    try {
        const alerts = [];

        // Low stock
        const lowStock = await query(`
            SELECT id, name, barcode, qty, unit_name, cost_price, sale_price_latli
            FROM tbl_stock 
            WHERE qty <= 10 AND qty > 0
            ORDER BY qty ASC
        `);

        // Out of stock
        const outOfStock = await query(`
            SELECT id, name, barcode, unit_name
            FROM tbl_stock 
            WHERE qty = 0
        `);

        res.status(200).json({
            success: true,
            data: {
                low_stock: lowStock,
                out_of_stock: outOfStock,
                total_alerts: lowStock.length + outOfStock.length
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Get expiration alerts specifically
 */
exports.getExpirationAlerts = async (req, res, next) => {
    try {
        // Expired products
        const expired = await query(`
            SELECT p.id, p.name, p.barcode, p.expire_date, p.qty,
                   DATEDIFF(CURDATE(), p.expire_date) as days_expired
            FROM tbl_purchase p
            WHERE p.expire_date < CURDATE() AND p.qty > 0
            ORDER BY p.expire_date DESC
        `);

        // Expiring within 7 days
        const expiring7Days = await query(`
            SELECT p.id, p.name, p.barcode, p.expire_date, p.qty,
                   DATEDIFF(p.expire_date, CURDATE()) as days_remaining
            FROM tbl_purchase p
            WHERE p.expire_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 7 DAY)
            AND p.qty > 0
            ORDER BY p.expire_date ASC
        `);

        // Expiring within 30 days
        const expiring30Days = await query(`
            SELECT p.id, p.name, p.barcode, p.expire_date, p.qty,
                   DATEDIFF(p.expire_date, CURDATE()) as days_remaining
            FROM tbl_purchase p
            WHERE p.expire_date BETWEEN DATE_ADD(CURDATE(), INTERVAL 8 DAY) AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)
            AND p.qty > 0
            ORDER BY p.expire_date ASC
        `);

        res.status(200).json({
            success: true,
            data: {
                expired: expired,
                expiring_7_days: expiring7Days,
                expiring_30_days: expiring30Days,
                total_alerts: expired.length + expiring7Days.length + expiring30Days.length
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Mark notification as read
 */
exports.markAsRead = async (req, res, next) => {
    try {
        const { notificationId } = req.params;
        const userId = req.user?.id || req.user?.user_id;

        // Store read status in database if user is logged in
        // For now, we'll use a simple approach with a read_notifications table
        // If table doesn't exist, we'll just return success (client-side will handle it)
        try {
            // Check if read_notifications table exists, if not create it
            await query(`
                CREATE TABLE IF NOT EXISTS tbl_notification_read (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    notification_id VARCHAR(255) NOT NULL,
                    user_id INT,
                    read_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    UNIQUE KEY unique_notification_user (notification_id, user_id),
                    INDEX idx_user_id (user_id),
                    INDEX idx_notification_id (notification_id)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            `);

            // Insert read status
            if (userId) {
                await query(`
                    INSERT INTO tbl_notification_read (notification_id, user_id)
                    VALUES (?, ?)
                    ON DUPLICATE KEY UPDATE read_at = CURRENT_TIMESTAMP
                `, [notificationId, userId]);
            }
        } catch (dbError) {
            // If table creation fails, just log and continue (client-side will handle it)
            console.warn('Could not store read status in database:', dbError.message);
        }

        res.status(200).json({
            success: true,
            message: 'Notification marked as read'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Mark all notifications as read
 */
exports.markAllAsRead = async (req, res, next) => {
    try {
        const userId = req.user?.id || req.user?.user_id;

        // Get all current notification IDs by calling getNotifications logic
        // We'll create a temporary request object to get notifications
        const tempReq = { ...req };
        const notifications = [];
        
        // Get all notification IDs that would be generated
        const lowStockProducts = await query(`SELECT id FROM tbl_stock WHERE qty <= 10 AND qty > 0 LIMIT 10`);
        const outOfStockProducts = await query(`SELECT id FROM tbl_stock WHERE qty = 0 LIMIT 10`);
        const expiredProducts = await query(`SELECT id FROM tbl_purchase WHERE expire_date < CURDATE() AND qty > 0 LIMIT 10`);
        const expiringProducts = await query(`SELECT id FROM tbl_purchase WHERE expire_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY) AND qty > 0 LIMIT 10`);
        const todayAppointments = await query(`SELECT id FROM tbl_appointment WHERE DATE(time) = CURDATE()`);
        const pendingPayments = await query(`SELECT id FROM tbl_inpatient WHERE end_date IS NULL AND deposit < 100000 LIMIT 5`);
        const recentSalesCount = await query(`SELECT COUNT(*) as count FROM tbl_sale WHERE DATE(Date) = CURDATE()`);
        
        const notificationIds = [
            ...lowStockProducts.map(p => `low-stock-${p.id}`),
            ...outOfStockProducts.map(p => `out-of-stock-${p.id}`),
            ...expiredProducts.map(p => `expired-${p.id}`),
            ...expiringProducts.map(p => `expiring-${p.id}`),
            ...todayAppointments.map(a => `appointment-${a.id}`),
            ...pendingPayments.map(p => `payment-${p.id}`),
            ...(recentSalesCount[0].count > 0 ? ['sales-today'] : []),
            'system-welcome'
        ];
        
        // Store read status for all notifications
        try {
            // Ensure table exists
            await query(`
                CREATE TABLE IF NOT EXISTS tbl_notification_read (
                    id INT AUTO_INCREMENT PRIMARY KEY,
                    notification_id VARCHAR(255) NOT NULL,
                    user_id INT,
                    read_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    UNIQUE KEY unique_notification_user (notification_id, user_id),
                    INDEX idx_user_id (user_id),
                    INDEX idx_notification_id (notification_id)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
            `);

            if (userId) {
                // Mark all notifications as read
                for (const notificationId of notificationIds) {
                    await query(`
                        INSERT INTO tbl_notification_read (notification_id, user_id)
                        VALUES (?, ?)
                        ON DUPLICATE KEY UPDATE read_at = CURRENT_TIMESTAMP
                    `, [notificationId, userId]);
                }
            }
        } catch (dbError) {
            console.warn('Could not store read status in database:', dbError.message);
        }

        res.status(200).json({
            success: true,
            message: 'All notifications marked as read'
        });
    } catch (error) {
        next(error);
    }
};

