const { query } = require('../config/database');

// @desc    Get all patients
// @route   GET /api/patients
// @access  Private
exports.getPatients = async (req, res, next) => {
    try {
        const page = parseInt(req.query.page) || 1;
        const limit = parseInt(req.query.limit) || 20;
        const { date, date_from, date_to, date_range, doctor, patient_type, sort_by, sort_order } = req.query;
        const offset = (page - 1) * limit;

        let sql = `SELECT p.*, 
                   CASE 
                       WHEN p.phone IS NOT NULL AND p.phone != '' THEN 
                           (SELECT COUNT(*) FROM tbl_patient WHERE phone = p.phone)
                       ELSE 1
                   END as visit_count
                   FROM tbl_patient p`;
        let params = [];
        let whereClauses = [];

        // Handle date filtering
        if (date_range) {
            // Quick date range filters
            const now = new Date();
            const year = now.getFullYear();
            const month = now.getMonth(); // 0-11
            
            // Format date as YYYY-MM-DD for MySQL
            const formatDate = (dateObj) => {
                const y = dateObj.getFullYear();
                const m = String(dateObj.getMonth() + 1).padStart(2, '0');
                const d = String(dateObj.getDate()).padStart(2, '0');
                return `${y}-${m}-${d}`;
            };
            
            if (date_range === 'today') {
                const todayStr = formatDate(now);
                whereClauses.push('p.date = ?');
                params.push(todayStr);
            } else if (date_range === 'month') {
                // First day of current month
                const firstDay = new Date(year, month, 1);
                const firstDayStr = formatDate(firstDay);
                // Last day of current month
                const lastDay = new Date(year, month + 1, 0);
                const lastDayStr = formatDate(lastDay);
                whereClauses.push('p.date >= ? AND p.date <= ?');
                params.push(firstDayStr, lastDayStr);
            } else if (date_range === 'year') {
                // First day of current year
                const firstDay = new Date(year, 0, 1);
                const firstDayStr = formatDate(firstDay);
                // Last day of current year
                const lastDay = new Date(year, 11, 31);
                const lastDayStr = formatDate(lastDay);
                whereClauses.push('p.date >= ? AND p.date <= ?');
                params.push(firstDayStr, lastDayStr);
            }
        } else if (date_from && date_to) {
            // Custom date range
            whereClauses.push('p.date >= ? AND p.date <= ?');
            params.push(date_from, date_to);
        } else if (date_from) {
            // From date only
            whereClauses.push('p.date >= ?');
            params.push(date_from);
        } else if (date_to) {
            // To date only
            whereClauses.push('p.date <= ?');
            params.push(date_to);
        } else if (date) {
            // Single date (existing functionality)
            whereClauses.push('p.date = ?');
            params.push(date);
        }

        if (doctor) {
            whereClauses.push('p.dr_name LIKE ?');
            params.push(`%${doctor}%`);
        }

        if (whereClauses.length > 0) {
            sql += ' WHERE ' + whereClauses.join(' AND ');
        }

        // Handle sorting (sanitize to prevent SQL injection)
        const validSortColumns = ['id', 'name', 'age', 'phone', 'case_', 'dr_name', 'date', 'patient_id'];
        const sortColumn = sort_by && validSortColumns.includes(sort_by) ? sort_by : 'date';
        const sortDirection = (sort_order && sort_order.toUpperCase() === 'ASC') ? 'ASC' : 'DESC';
        
        // Use parameterized query for column name (safe because we validate against whitelist)
        sql += ` ORDER BY p.\`${sortColumn}\` ${sortDirection} LIMIT ? OFFSET ?`;
        params.push(limit, offset);

        // Debug logging
        console.log('Query params:', { sql, params, types: params.map(p => typeof p) });

        const patients = await query(sql, params);

        // Process patients to add patient_type
        const processedPatients = patients.map(patient => ({
            ...patient,
            visit_count: parseInt(patient.visit_count) || 1,
            patient_type: parseInt(patient.visit_count) > 1 ? 'returning' : 'new',
            is_returning: parseInt(patient.visit_count) > 1
        }));

        // Filter by patient_type if specified
        let filteredPatients = processedPatients;
        if (patient_type === 'new') {
            filteredPatients = processedPatients.filter(p => p.visit_count === 1);
        } else if (patient_type === 'returning') {
            filteredPatients = processedPatients.filter(p => p.visit_count > 1);
        }

        // Get total count
        let countSql = 'SELECT COUNT(*) as total FROM tbl_patient p';
        if (whereClauses.length > 0) {
            countSql += ' WHERE ' + whereClauses.join(' AND ');
        }
        const countResult = await query(countSql, params.slice(0, -2));

        res.status(200).json({
            success: true,
            count: filteredPatients.length,
            total: countResult[0].total,
            page: page,
            pages: Math.ceil(countResult[0].total / limit),
            data: filteredPatients
        });
    } catch (error) {
        next(error);
    }
};

// @desc    Get single patient
// @route   GET /api/patients/:id
// @access  Private
exports.getPatient = async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_patient WHERE id = ?';
        const patients = await query(sql, [req.params.id]);

        if (patients.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Patient not found'
            });
        }

        res.status(200).json({
            success: true,
            data: patients[0]
        });
    } catch (error) {
        next(error);
    }
};

// @desc    Check if patient exists
// @route   GET /api/patients/check
// @access  Private
exports.checkPatientExists = async (req, res, next) => {
    try {
        const { phone, name } = req.query;

        if (!phone && !name) {
            return res.status(400).json({
                success: false,
                message: 'Phone number or name is required'
            });
        }

        let sql = 'SELECT * FROM tbl_patient WHERE ';
        let params = [];

        if (phone && name) {
            sql += '(phone = ? OR name LIKE ?)';
            params.push(phone, `%${name}%`);
        } else if (phone) {
            sql += 'phone = ?';
            params.push(phone);
        } else {
            sql += 'name LIKE ?';
            params.push(`%${name}%`);
        }

        sql += ' ORDER BY date DESC LIMIT 10';

        const patients = await query(sql, params);

        // Get visit count for each patient
        const patientsWithVisitCount = await Promise.all(patients.map(async (patient) => {
            let visitCount = 1;
            if (patient.phone) {
                const visitCountResult = await query(
                    'SELECT COUNT(*) as visit_count FROM tbl_patient WHERE phone = ?',
                    [patient.phone]
                );
                visitCount = visitCountResult[0]?.visit_count || 1;
            }
            return {
                ...patient,
                visit_count: visitCount,
                is_returning: visitCount > 1,
                patient_type: visitCount > 1 ? 'returning' : 'new'
            };
        }));

        res.status(200).json({
            success: true,
            exists: patients.length > 0,
            count: patients.length,
            data: patientsWithVisitCount
        });
    } catch (error) {
        next(error);
    }
};

// @desc    Create new patient
// @route   POST /api/patients
// @access  Private
exports.createPatient = async (req, res, next) => {
    try {
        const { name, age, phone, address, case_, remark, dr_name, room_no, deposit, force_new } = req.body;

        // Check if patient already exists (unless force_new is true)
        if (!force_new && phone) {
            const existingCheck = await query(
                'SELECT id, name, patient_id, date FROM tbl_patient WHERE phone = ? ORDER BY date DESC LIMIT 1',
                [phone]
            );

            if (existingCheck.length > 0) {
                const existingPatient = existingCheck[0];
                // Get visit count
                const visitCountResult = await query(
                    'SELECT COUNT(*) as visit_count FROM tbl_patient WHERE phone = ?',
                    [phone]
                );
                const visit_count = visitCountResult[0]?.visit_count || 1;

                return res.status(409).json({
                    success: false,
                    message: 'Patient already exists in the system',
                    data: {
                        existing_patient: existingPatient,
                        visit_count: visit_count,
                        is_returning: visit_count > 1,
                        patient_type: visit_count > 1 ? 'returning' : 'existing'
                    },
                    action_required: 'use_existing_or_force_new'
                });
            }
        }

        // Check by name if no phone (less strict check)
        if (!force_new && !phone && name) {
            const existingByName = await query(
                'SELECT id, name, patient_id, phone, date FROM tbl_patient WHERE name LIKE ? ORDER BY date DESC LIMIT 5',
                [`%${name}%`]
            );

            if (existingByName.length > 0) {
                return res.status(409).json({
                    success: false,
                    message: 'Similar patient names found. Please verify if this is a new patient or use existing.',
                    data: {
                        similar_patients: existingByName,
                        patient_type: 'possible_duplicate'
                    },
                    action_required: 'verify_or_force_new'
                });
            }
        }

        // Generate patient ID
        const countSql = 'SELECT COUNT(*) as count FROM tbl_patient';
        const countResult = await query(countSql);
        const patient_id = `P${String(countResult[0].count + 1).padStart(6, '0')}`;

        const sql = `INSERT INTO tbl_patient 
                     (name, age, phone, address, case_, remark, date, dr_name, patient_id, room_no, deposit) 
                     VALUES (?, ?, ?, ?, ?, ?, CURDATE(), ?, ?, ?, ?)`;

        const result = await query(sql, [name, age, phone, address, case_, remark, dr_name, patient_id, room_no, deposit || 0]);

        res.status(201).json({
            success: true,
            message: 'New patient created successfully',
            data: {
                id: result.insertId,
                patient_id,
                name,
                patient_type: 'new'
            }
        });
    } catch (error) {
        next(error);
    }
};

// @desc    Update patient
// @route   PUT /api/patients/:id
// @access  Private
exports.updatePatient = async (req, res, next) => {
    try {
        const { name, age, phone, address, case_, remark, dr_name, room_no, deposit } = req.body;

        const sql = `UPDATE tbl_patient 
                     SET name = ?, age = ?, phone = ?, address = ?, case_ = ?, 
                         remark = ?, dr_name = ?, room_no = ?, deposit = ?
                     WHERE id = ?`;

        await query(sql, [name, age, phone, address, case_, remark, dr_name, room_no, deposit, req.params.id]);

        res.status(200).json({
            success: true,
            message: 'Patient updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

// @desc    Delete patient
// @route   DELETE /api/patients/:id
// @access  Private
exports.deletePatient = async (req, res, next) => {
    try {
        const sql = 'DELETE FROM tbl_patient WHERE id = ?';
        await query(sql, [req.params.id]);

        res.status(200).json({
            success: true,
            message: 'Patient deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

// @desc    Search patients
// @route   GET /api/patients/search
// @access  Private
exports.searchPatients = async (req, res, next) => {
    try {
        const { q } = req.query;

        if (!q) {
            return res.status(400).json({
                success: false,
                message: 'Search query is required'
            });
        }

        const sql = `SELECT * FROM tbl_patient 
                     WHERE name LIKE ? OR patient_id LIKE ? OR phone LIKE ?
                     LIMIT 50`;

        const patients = await query(sql, [`%${q}%`, `%${q}%`, `%${q}%`]);

        res.status(200).json({
            success: true,
            count: patients.length,
            data: patients
        });
    } catch (error) {
        next(error);
    }
};

// @desc    Get patient history
// @route   GET /api/patients/:id/history
// @access  Private
exports.getPatientHistory = async (req, res, next) => {
    try {
        const patientSql = 'SELECT patient_id FROM tbl_patient WHERE id = ?';
        const patientResult = await query(patientSql, [req.params.id]);

        if (patientResult.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Patient not found'
            });
        }

        const patient_id = patientResult[0].patient_id;

        // Get sales history
        const salesSql = `SELECT * FROM tbl_patient_sale 
                          WHERE patient_id = ? 
                          ORDER BY Date DESC`;
        const sales = await query(salesSql, [patient_id]);

        // Get treatment history
        const treatmentSql = `SELECT * FROM tbl_treatment 
                              WHERE patient_id = ? 
                              ORDER BY Date DESC`;
        const treatments = await query(treatmentSql, [patient_id]);

        res.status(200).json({
            success: true,
            data: {
                sales,
                treatments
            }
        });
    } catch (error) {
        next(error);
    }
};

