/**
 * Permission Management Controller
 * Handles CRUD operations for roles, permissions, and assignments
 */

const { Role, Permission, RolePermission, User } = require('../models/sequelize');
const { clearPermissionCache } = require('../utils/permissions');

/**
 * @desc    Get all roles
 * @route   GET /api/permissions/roles
 * @access  Private (Admin only)
 */
exports.getRoles = async (req, res, next) => {
    try {
        const roles = await Role.findAll({
            include: [{
                model: Permission,
                as: 'permissions',
                through: {
                    where: { active: true, granted: true },
                    attributes: []
                }
            }],
            order: [['level', 'DESC']]
        });

        const rolesWithCounts = await Promise.all(roles.map(async (role) => {
            const userCount = await User.count({ where: { roleId: role.id } });
            return {
                id: role.id,
                name: role.name,
                level: role.level,
                description: role.description,
                active: role.active,
                metadata: typeof role.metadata === 'string' ? JSON.parse(role.metadata) : role.metadata,
                permissionCount: role.permissions?.length || 0,
                userCount: userCount,
                createdAt: role.createdAt,
                updatedAt: role.updatedAt
            };
        }));

        res.json({
            success: true,
            data: rolesWithCounts
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get single role with permissions
 * @route   GET /api/permissions/roles/:id
 * @access  Private (Admin only)
 */
exports.getRole = async (req, res, next) => {
    try {
        const role = await Role.findByPk(req.params.id, {
            include: [{
                model: Permission,
                as: 'permissions',
                through: {
                    attributes: ['granted', 'active', 'metadata']
                }
            }]
        });

        if (!role) {
            return res.status(404).json({
                success: false,
                message: 'Role not found'
            });
        }

        res.json({
            success: true,
            data: {
                ...role.toJSON(),
                metadata: typeof role.metadata === 'string' ? JSON.parse(role.metadata) : role.metadata
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create role
 * @route   POST /api/permissions/roles
 * @access  Private (Admin only)
 */
exports.createRole = async (req, res, next) => {
    try {
        const { name, level, description, active, metadata } = req.body;

        const role = await Role.create({
            name,
            level: level || 0,
            description,
            active: active !== undefined ? active : true,
            metadata: metadata || {}
        });

        res.status(201).json({
            success: true,
            data: role
        });
    } catch (error) {
        if (error.name === 'SequelizeUniqueConstraintError') {
            return res.status(400).json({
                success: false,
                message: 'Role name already exists'
            });
        }
        next(error);
    }
};

/**
 * @desc    Update role
 * @route   PUT /api/permissions/roles/:id
 * @access  Private (Admin only)
 */
exports.updateRole = async (req, res, next) => {
    try {
        const { name, level, description, active, metadata } = req.body;

        const role = await Role.findByPk(req.params.id);
        if (!role) {
            return res.status(404).json({
                success: false,
                message: 'Role not found'
            });
        }

        await role.update({
            name: name || role.name,
            level: level !== undefined ? level : role.level,
            description: description !== undefined ? description : role.description,
            active: active !== undefined ? active : role.active,
            metadata: metadata !== undefined ? metadata : role.metadata
        });

        // Clear cache for all users with this role
        const users = await User.findAll({ where: { roleId: role.id } });
        for (const user of users) {
            clearPermissionCache(user.id);
        }

        res.json({
            success: true,
            data: role
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get all permissions
 * @route   GET /api/permissions/permissions
 * @access  Private (Admin only)
 */
exports.getPermissions = async (req, res, next) => {
    try {
        const permissions = await Permission.findAll({
            where: { active: true },
            order: [['resource', 'ASC'], ['action', 'ASC']]
        });

        // Group by resource, excluding 'import' and 'manage' actions
        const grouped = {};
        for (const perm of permissions) {
            // Skip 'import' and 'manage' permissions
            if (perm.action === 'import' || perm.action === 'manage') {
                continue;
            }
            
            if (!grouped[perm.resource]) {
                grouped[perm.resource] = [];
            }
            grouped[perm.resource].push({
                id: perm.id,
                action: perm.action,
                description: perm.description,
                metadata: typeof perm.metadata === 'string' ? JSON.parse(perm.metadata) : perm.metadata
            });
        }

        res.json({
            success: true,
            data: grouped
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create permission
 * @route   POST /api/permissions/permissions
 * @access  Private (Admin only)
 */
exports.createPermission = async (req, res, next) => {
    try {
        const { resource, action, description, metadata } = req.body;

        const permission = await Permission.create({
            resource,
            action,
            description,
            active: true,
            metadata: metadata || {}
        });

        res.status(201).json({
            success: true,
            data: permission
        });
    } catch (error) {
        if (error.name === 'SequelizeUniqueConstraintError') {
            return res.status(400).json({
                success: false,
                message: 'Permission already exists'
            });
        }
        next(error);
    }
};

/**
 * @desc    Get role permissions
 * @route   GET /api/permissions/roles/:roleId/permissions
 * @access  Private (Admin only)
 */
exports.getRolePermissions = async (req, res, next) => {
    try {
        const role = await Role.findByPk(req.params.roleId, {
            include: [{
                model: Permission,
                as: 'permissions',
                through: {
                    attributes: ['granted', 'active', 'metadata']
                },
                required: false // LEFT JOIN to include all permissions
            }]
        });

        if (!role) {
            return res.status(404).json({
                success: false,
                message: 'Role not found'
            });
        }

        // Get ALL permissions first (to show complete list)
        const allPermissions = await Permission.findAll({
            where: { active: true },
            order: [['resource', 'ASC'], ['action', 'ASC']]
        });

        // Get role permissions with their granted/active status
        // Query RolePermission directly using raw query for accuracy
        const { query } = require('../config/database');
        const rolePermsData = await query(
            `SELECT rp.permission_id, rp.granted, rp.active, p.resource, p.action
             FROM tbl_role_permission rp
             INNER JOIN tbl_permission p ON rp.permission_id = p.id
             WHERE rp.role_id = ? AND p.active = true`,
            [req.params.roleId]
        );

        const rolePermsMap = new Map();
        for (const rp of rolePermsData) {
            rolePermsMap.set(rp.permission_id, {
                granted: rp.granted === true || rp.granted === 1,
                active: rp.active === true || rp.active === 1
            });
        }

        // Format as resource:action map, excluding 'import' and 'manage' actions
        const permissions = {};
        for (const perm of allPermissions) {
            // Skip 'import' and 'manage' permissions
            if (perm.action === 'import' || perm.action === 'manage') {
                continue;
            }
            
            if (!permissions[perm.resource]) {
                permissions[perm.resource] = {};
            }
            
            // Check if permission is granted and active
            const rolePerm = rolePermsMap.get(perm.id);
            const isGranted = rolePerm?.granted === true && rolePerm?.active === true;
            permissions[perm.resource][perm.action] = isGranted;
        }

        console.log(`📖 Loaded permissions for role ${req.params.roleId}:`, 
            Object.keys(permissions).length, 'resources');

        res.json({
            success: true,
            data: permissions
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update role permissions
 * @route   PUT /api/permissions/roles/:roleId/permissions
 * @access  Private (Admin only)
 */
exports.updateRolePermissions = async (req, res, next) => {
    try {
        const { roleId } = req.params;
        const { permissions } = req.body; // { resource: { action: true/false } }

        console.log('🔐 Updating role permissions for role:', roleId);
        console.log('📋 Received permissions:', JSON.stringify(permissions, null, 2));

        // Convert roleId to integer
        const roleIdInt = parseInt(roleId, 10);
        if (isNaN(roleIdInt)) {
            return res.status(400).json({
                success: false,
                message: 'Invalid role ID'
            });
        }

        const role = await Role.findByPk(roleIdInt);
        if (!role) {
            return res.status(404).json({
                success: false,
                message: 'Role not found'
            });
        }

        // Get all permissions
        const allPermissions = await Permission.findAll({ where: { active: true } });

        // Process each permission, excluding 'import' and 'manage'
        let updatedCount = 0;
        let createdCount = 0;
        for (const perm of allPermissions) {
            // Skip 'import' and 'manage' permissions
            if (perm.action === 'import' || perm.action === 'manage') {
                continue;
            }
            
            const shouldHave = permissions[perm.resource]?.[perm.action] === true;

            // Use findOrCreate to ensure record exists
            // Use camelCase property names (roleId, permissionId) as defined in the model
            const [rolePermission, created] = await RolePermission.findOrCreate({
                where: {
                    roleId: roleIdInt,
                    permissionId: perm.id
                },
                defaults: {
                    roleId: roleIdInt,
                    permissionId: perm.id,
                    granted: shouldHave,
                    active: shouldHave
                }
            });

            // Update if it already existed and value changed
            if (!created) {
                const needsUpdate = rolePermission.granted !== shouldHave || rolePermission.active !== shouldHave;
                if (needsUpdate) {
                    await rolePermission.update({
                        granted: shouldHave,
                        active: shouldHave
                    });
                    updatedCount++;
                }
            } else {
                createdCount++;
            }
        }

        console.log(`✅ Updated ${updatedCount} permissions, created ${createdCount} new assignments`);

        // Clear cache for all users with this role
        const users = await User.findAll({ where: { roleId: roleIdInt } });
        for (const user of users) {
            clearPermissionCache(user.id);
        }

        res.json({
            success: true,
            message: 'Role permissions updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get users with roles
 * @route   GET /api/permissions/users
 * @access  Private (Admin only)
 */
exports.getUsers = async (req, res, next) => {
    try {
        const users = await User.findAll({
            include: [{
                model: Role,
                as: 'role',
                attributes: ['id', 'name', 'level']
            }],
            attributes: ['id', 'name', 'roleId', 'phNo', 'address'],
            order: [['name', 'ASC']]
        });

        res.json({
            success: true,
            data: users
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update user role
 * @route   PUT /api/permissions/users/:userId/role
 * @access  Private (Admin only)
 */
exports.updateUserRole = async (req, res, next) => {
    try {
        const { userId } = req.params;
        const { roleId } = req.body;

        const user = await User.findByPk(userId);
        if (!user) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        const role = await Role.findByPk(roleId);
        if (!role) {
            return res.status(404).json({
                success: false,
                message: 'Role not found'
            });
        }

        await user.update({ roleId });

        // Clear user's permission cache
        clearPermissionCache(userId);

        res.json({
            success: true,
            message: 'User role updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get user permissions
 * @route   GET /api/permissions/users/:userId/permissions
 * @access  Private (Admin only)
 */
exports.getUserPermissions = async (req, res, next) => {
    try {
        const { getUserPermissions: getUserPerms } = require('../utils/permissions');
        const permissions = await getUserPerms(req.params.userId);

        res.json({
            success: true,
            data: permissions
        });
    } catch (error) {
        next(error);
    }
};

