/**
 * Advanced Pharmacy Management Controller
 * Drug Formulary, Prescriptions, Medication Administration, Controlled Substances
 */

const { query } = require('../config/database');

// ============================================
// DRUG FORMULARY MANAGEMENT
// ============================================

/**
 * @desc    Get drug formulary
 * @route   GET /api/pharmacy/formulary
 * @access  Private
 */
exports.getDrugFormulary = async (req, res, next) => {
    try {
        const { drug_class, formulary_status, controlled, search } = req.query;
        let sql = 'SELECT * FROM tbl_drug_formulary WHERE is_active = 1';
        const params = [];

        if (drug_class) {
            sql += ' AND drug_class = ?';
            params.push(drug_class);
        }
        if (formulary_status) {
            sql += ' AND formulary_status = ?';
            params.push(formulary_status);
        }
        if (controlled) {
            sql += ' AND is_controlled_substance = 1';
        }
        if (search) {
            sql += ' AND (generic_name LIKE ? OR brand_name LIKE ? OR drug_code LIKE ?)';
            const searchTerm = `%${search}%`;
            params.push(searchTerm, searchTerm, searchTerm);
        }

        sql += ' ORDER BY generic_name';
        const drugs = await query(sql, params);

        res.json({
            success: true,
            count: drugs.length,
            data: drugs
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Add drug to formulary
 * @route   POST /api/pharmacy/formulary
 * @access  Private
 */
exports.addDrugToFormulary = async (req, res, next) => {
    try {
        const {
            drug_code,
            generic_name,
            brand_name,
            drug_class,
            therapeutic_category,
            form,
            strength,
            route,
            unit_of_measure,
            formulary_status = 'Preferred',
            is_controlled_substance = 0,
            dea_schedule = 'None',
            requires_authorization = 0,
            max_daily_dose,
            typical_dosage,
            contraindications,
            side_effects,
            storage_requirements,
            pregnancy_category,
            manufacturer
        } = req.body;

        // Basic validation
        if (!drug_code || !generic_name || !drug_class || !form || !strength || !route) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing: drug_code, generic_name, drug_class, form, strength, route'
            });
        }

        // Check if drug code already exists
        const existingDrug = await query('SELECT id FROM tbl_drug_formulary WHERE drug_code = ?', [drug_code]);
        if (existingDrug.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Drug code already exists in formulary'
            });
        }

        const sql = `INSERT INTO tbl_drug_formulary 
            (drug_code, generic_name, brand_name, drug_class, therapeutic_category, form, strength, route, 
            unit_of_measure, formulary_status, is_controlled_substance, dea_schedule, requires_authorization,
            max_daily_dose, typical_dosage, contraindications, side_effects, storage_requirements, 
            pregnancy_category, manufacturer, is_active)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)`;
        
        const result = await query(sql, [
            drug_code,
            generic_name,
            brand_name || null,
            drug_class,
            therapeutic_category || null,
            form,
            strength,
            route,
            unit_of_measure || null,
            formulary_status,
            is_controlled_substance,
            dea_schedule,
            requires_authorization,
            max_daily_dose || null,
            typical_dosage || null,
            contraindications || null,
            side_effects || null,
            storage_requirements || null,
            pregnancy_category || null,
            manufacturer || null
        ]);

        res.status(201).json({
            success: true,
            data: {
                id: result.insertId,
                drug_code: drug_code,
                generic_name: generic_name
            },
            message: 'Drug added to formulary successfully'
        });
    } catch (error) {
        if (error.code === 'ER_DUP_ENTRY') {
            return res.status(400).json({
                success: false,
                message: 'Drug code already exists in formulary'
            });
        }
        next(error);
    }
};

/**
 * @desc    Check drug interactions
 * @route   POST /api/pharmacy/check-interactions
 * @access  Private
 */
exports.checkDrugInteractions = async (req, res, next) => {
    try {
        const { drug_ids } = req.body; // Array of drug IDs

        if (!drug_ids || drug_ids.length < 2) {
            return res.json({
                success: true,
                interactions: []
            });
        }

        const sql = `SELECT i.*, d1.generic_name as drug1_name, d2.generic_name as drug2_name
            FROM tbl_drug_interactions i
            JOIN tbl_drug_formulary d1 ON i.drug1_id = d1.id
            JOIN tbl_drug_formulary d2 ON i.drug2_id = d2.id
            WHERE (i.drug1_id IN (${drug_ids.join(',')}) AND i.drug2_id IN (${drug_ids.join(',')}))
            ORDER BY FIELD(i.interaction_severity, 'Contraindicated', 'Major', 'Moderate', 'Minor')`;
        
        const interactions = await query(sql);

        res.json({
            success: true,
            interactions: interactions,
            has_critical: interactions.some(i => ['Contraindicated', 'Major'].includes(i.interaction_severity))
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// PRESCRIPTION MANAGEMENT
// ============================================

/**
 * @desc    Create prescription order
 * @route   POST /api/pharmacy/prescriptions
 * @access  Private
 */
exports.createPrescription = async (req, res, next) => {
    try {
        const {
            patient_id,
            doctor_id,
            encounter_type = 'Outpatient',
            diagnosis,
            medications // Array of medication details
        } = req.body;

        if (!patient_id || !doctor_id || !medications || medications.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing'
            });
        }

        // Generate prescription order number
        const year = new Date().getFullYear();
        await query('UPDATE tbl_prescription_sequence SET last_order_number = last_order_number + 1 WHERE year = ?', [year]);
        const seqResult = await query('SELECT last_order_number FROM tbl_prescription_sequence WHERE year = ?', [year]);
        const orderNum = seqResult[0].last_order_number;
        const order_number = `RX${year}${String(orderNum).padStart(6, '0')}`;

        // Check for drug interactions
        const drugIds = medications.filter(m => m.drug_id).map(m => m.drug_id);
        if (drugIds.length >= 2) {
            const interactionsSql = `SELECT * FROM tbl_drug_interactions 
                WHERE (drug1_id IN (${drugIds.join(',')}) AND drug2_id IN (${drugIds.join(',')}))
                AND interaction_severity IN ('Contraindicated', 'Major')`;
            const interactions = await query(interactionsSql);
            
            if (interactions.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Critical drug interactions detected',
                    interactions: interactions
                });
            }
        }

        // Calculate total amount
        const drugPrices = await Promise.all(
            medications.map(async (med) => {
                if (med.drug_id) {
                    const result = await query('SELECT price FROM tbl_stock WHERE id = ? LIMIT 1', [med.drug_id]);
                    return (result[0]?.price || 0) * (med.quantity || 1);
                }
                return 0;
            })
        );
        const total_amount = drugPrices.reduce((sum, price) => sum + price, 0);

        // Create prescription order
        const orderSql = `INSERT INTO tbl_prescription_orders 
            (order_number, patient_id, doctor_id, encounter_type, diagnosis, total_amount, created_by, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'Active')`;
        const orderResult = await query(orderSql, [
            order_number,
            patient_id,
            doctor_id,
            encounter_type ?? 'Outpatient',
            diagnosis ?? null,
            total_amount,
            req.user.id
        ]);

        // Insert prescription details
        for (const med of medications) {
            const detailSql = `INSERT INTO tbl_prescription_details 
                (prescription_id, drug_id, drug_name, dosage, frequency, route, duration, 
                quantity, refills, instructions, indication, start_date, end_date, is_prn, price, status)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Pending')`;
            
            await query(detailSql, [
                orderResult.insertId,
                med.drug_id || null,
                med.drug_name,
                med.dosage,
                med.frequency,
                med.route,
                med.duration || null,
                med.quantity,
                med.refills || 0,
                med.instructions || null,
                med.indication || null,
                med.start_date || null,
                med.end_date || null,
                med.is_prn || 0,
                med.price || 0
            ]);
        }

        res.status(201).json({
            success: true,
            data: {
                id: orderResult.insertId,
                order_number: order_number,
                total_amount: total_amount
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get prescriptions
 * @route   GET /api/pharmacy/prescriptions
 * @access  Private
 */
exports.getPrescriptions = async (req, res, next) => {
    try {
        const { patient_id, doctor_id, status, from, to } = req.query;
        let sql = `SELECT p.*, pat.name as patient_name, d.name as doctor_name
            FROM tbl_prescription_orders p
            LEFT JOIN tbl_patient pat ON p.patient_id = pat.patient_id
            LEFT JOIN tbl_doctor d ON p.doctor_id = d.id
            WHERE 1=1`;
        const params = [];

        if (patient_id) {
            sql += ' AND p.patient_id = ?';
            params.push(patient_id);
        }
        if (doctor_id) {
            sql += ' AND p.doctor_id = ?';
            params.push(doctor_id);
        }
        if (status) {
            sql += ' AND p.status = ?';
            params.push(status);
        }
        if (from) {
            sql += ' AND p.order_date >= ?';
            params.push(from);
        }
        if (to) {
            sql += ' AND p.order_date <= ?';
            params.push(to);
        }

        sql += ' ORDER BY p.order_date DESC LIMIT 100';
        const prescriptions = await query(sql, params);

        // Get prescription details for each
        for (let rx of prescriptions) {
            rx.medications = await query(
                'SELECT * FROM tbl_prescription_details WHERE prescription_id = ?',
                [rx.id]
            );
        }

        res.json({
            success: true,
            count: prescriptions.length,
            data: prescriptions
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Dispense medication
 * @route   PATCH /api/pharmacy/prescriptions/:detailId/dispense
 * @access  Private
 */
exports.dispenseMedication = async (req, res, next) => {
    try {
        const sql = `UPDATE tbl_prescription_details SET 
            status = 'Dispensed',
            dispensed_date = NOW(),
            dispensed_by = ?
            WHERE id = ?`;
        
        await query(sql, [req.user.id, req.params.detailId]);

        res.json({
            success: true,
            message: 'Medication dispensed successfully'
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// MEDICATION ADMINISTRATION (MAR)
// ============================================

/**
 * @desc    Get medication schedule for patient
 * @route   GET /api/pharmacy/administration/patient/:patientId
 * @access  Private
 */
exports.getMedicationSchedule = async (req, res, next) => {
    try {
        const { date } = req.query;
        const targetDate = date || new Date().toISOString().split('T')[0];

        const sql = `SELECT m.*, pd.drug_name, pd.dosage, pd.route, pd.frequency
            FROM tbl_medication_administration m
            JOIN tbl_prescription_details pd ON m.prescription_detail_id = pd.id
            WHERE m.patient_id = ? AND DATE(m.scheduled_time) = ?
            ORDER BY m.scheduled_time`;
        
        const schedule = await query(sql, [req.params.patientId, targetDate]);

        res.json({
            success: true,
            count: schedule.length,
            data: schedule
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Administer medication
 * @route   PATCH /api/pharmacy/administration/:id/administer
 * @access  Private
 */
exports.administerMedication = async (req, res, next) => {
    try {
        const { dose_given, route, site, patient_response, notes } = req.body;

        const sql = `UPDATE tbl_medication_administration SET 
            actual_time = NOW(),
            dose_given = ?,
            route = ?,
            site = ?,
            administered_by = ?,
            status = 'Given',
            patient_response = ?,
            notes = ?
            WHERE id = ?`;
        
        await query(sql, [
            dose_given ?? null,
            route ?? null,
            site ?? null,
            req.user.id,
            patient_response ?? null,
            notes ?? null,
            req.params.id
        ]);

        res.json({
            success: true,
            message: 'Medication administration recorded'
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// CONTROLLED SUBSTANCE TRACKING
// ============================================

/**
 * @desc    Log controlled substance transaction
 * @route   POST /api/pharmacy/controlled-substances/log
 * @access  Private
 */
exports.logControlledSubstance = async (req, res, next) => {
    try {
        const {
            drug_id,
            drug_name,
            dea_schedule,
            transaction_type,
            quantity,
            balance_before,
            patient_id,
            prescription_id,
            witnessed_by,
            reason,
            notes
        } = req.body;

        const balance_after = transaction_type === 'Dispensed' || transaction_type === 'Wasted' || transaction_type === 'Destroyed'
            ? balance_before - quantity
            : balance_before + quantity;

        const sql = `INSERT INTO tbl_controlled_substance_log 
            (drug_id, drug_name, dea_schedule, transaction_type, quantity, balance_before, balance_after,
            patient_id, prescription_id, witnessed_by, performed_by, reason, notes)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;
        
        await query(sql, [
            drug_id,
            drug_name,
            dea_schedule,
            transaction_type,
            quantity,
            balance_before,
            balance_after,
            patient_id ?? null,
            prescription_id ?? null,
            witnessed_by ?? null,
            req.user.id,
            reason ?? null,
            notes ?? null
        ]);

        res.status(201).json({
            success: true,
            data: { balance_after }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get controlled substance audit log
 * @route   GET /api/pharmacy/controlled-substances/audit
 * @access  Private
 */
exports.getControlledSubstanceAudit = async (req, res, next) => {
    try {
        const { drug_id, from, to } = req.query;
        let sql = `SELECT l.*, u1.name as performed_by_name, u2.name as witnessed_by_name
            FROM tbl_controlled_substance_log l
            LEFT JOIN tbl_user u1 ON l.performed_by = u1.id
            LEFT JOIN tbl_user u2 ON l.witnessed_by = u2.id
            WHERE 1=1`;
        const params = [];

        if (drug_id) {
            sql += ' AND l.drug_id = ?';
            params.push(drug_id);
        }
        if (from) {
            sql += ' AND l.transaction_date >= ?';
            params.push(from);
        }
        if (to) {
            sql += ' AND l.transaction_date <= ?';
            params.push(to);
        }

        sql += ' ORDER BY l.transaction_date DESC LIMIT 500';
        const logs = await query(sql, params);

        res.json({
            success: true,
            count: logs.length,
            data: logs
        });
    } catch (error) {
        next(error);
    }
};

// ============================================
// ADVERSE DRUG EVENTS
// ============================================

/**
 * @desc    Report adverse drug event
 * @route   POST /api/pharmacy/adverse-events
 * @access  Private
 */
exports.reportAdverseDrugEvent = async (req, res, next) => {
    try {
        const {
            patient_id,
            drug_id,
            drug_name,
            event_date,
            severity,
            event_description,
            action_taken
        } = req.body;

        if (!patient_id || !drug_name || !event_date || !severity || !event_description) {
            return res.status(400).json({
                success: false,
                message: 'Required fields missing'
            });
        }

        // Generate report number
        const year = new Date().getFullYear();
        const count = await query('SELECT COUNT(*) as count FROM tbl_adverse_drug_events WHERE YEAR(reported_date) = ?', [year]);
        const reportNum = count[0].count + 1;
        const report_number = `ADE${year}${String(reportNum).padStart(6, '0')}`;

        const sql = `INSERT INTO tbl_adverse_drug_events 
            (report_number, patient_id, drug_id, drug_name, event_date, severity, 
            event_description, action_taken, reported_by, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'Open')`;
        
        const result = await query(sql, [
            report_number,
            patient_id,
            drug_id ?? null,
            drug_name,
            event_date,
            severity,
            event_description,
            action_taken ?? null,
            req.user.id
        ]);

        res.status(201).json({
            success: true,
            data: {
                id: result.insertId,
                report_number: report_number
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get adverse drug events
 * @route   GET /api/pharmacy/adverse-events
 * @access  Private
 */
exports.getAdverseDrugEvents = async (req, res, next) => {
    try {
        const { patient_id, drug_id, severity, status } = req.query;
        let sql = `SELECT e.*, p.name as patient_name
            FROM tbl_adverse_drug_events e
            LEFT JOIN tbl_patient p ON e.patient_id = p.patient_id
            WHERE 1=1`;
        const params = [];

        if (patient_id) {
            sql += ' AND e.patient_id = ?';
            params.push(patient_id);
        }
        if (drug_id) {
            sql += ' AND e.drug_id = ?';
            params.push(drug_id);
        }
        if (severity) {
            sql += ' AND e.severity = ?';
            params.push(severity);
        }
        if (status) {
            sql += ' AND e.status = ?';
            params.push(status);
        }

        sql += ' ORDER BY e.reported_date DESC LIMIT 100';
        const events = await query(sql, params);

        res.json({
            success: true,
            count: events.length,
            data: events
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get pharmacy statistics
 * @route   GET /api/pharmacy/stats
 * @access  Private
 */
exports.getPharmacyStats = async (req, res, next) => {
    try {
        const today = new Date().toISOString().split('T')[0];

        const prescriptionStats = await query(`
            SELECT 
                COUNT(*) as total_prescriptions,
                SUM(CASE WHEN status = 'Active' THEN 1 ELSE 0 END) as active,
                SUM(total_amount) as total_revenue
            FROM tbl_prescription_orders
            WHERE DATE(order_date) = '${today}'
        `);

        const pendingDispense = await query(`
            SELECT COUNT(*) as count
            FROM tbl_prescription_details
            WHERE status = 'Pending'
        `);

        const controlledSubstances = await query(`
            SELECT COUNT(DISTINCT id) as count
            FROM tbl_drug_formulary
            WHERE is_controlled_substance = 1 AND is_active = 1
        `);

        const adverseEvents = await query(`
            SELECT COUNT(*) as count
            FROM tbl_adverse_drug_events
            WHERE status = 'Open'
        `);

        res.json({
            success: true,
            data: {
                ...prescriptionStats[0],
                pending_dispense: pendingDispense[0].count,
                controlled_drugs: controlledSubstances[0].count,
                open_adverse_events: adverseEvents[0].count
            }
        });
    } catch (error) {
        next(error);
    }
};

