/**
 * Point of Sale Controller
 * Handles POS operations including sales, vouchers, and transactions
 */

const { query, transaction } = require('../config/database');
const models = require('../models');

/**
 * @desc    Get products for POS with images
 * @route   GET /api/pos/products
 * @access  Private
 */
exports.getProducts = async (req, res, next) => {
    try {
        const { category, search } = req.query;
        
        // Always use main stock (tbl_stock.qty)
        let sql = `SELECT s.*, c.name as category_name 
                   FROM tbl_stock s 
                   LEFT JOIN tbl_catalog c ON s.cat_id = c.id
                   WHERE s.qty > 0`;
        
        const params = [];
        
        if (category) {
            sql += ' AND s.cat_id = ?';
            params.push(category);
        }
        
        if (search) {
            sql += ' AND (s.name LIKE ? OR s.barcode LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }
        
        sql += ' ORDER BY s.name LIMIT 100';
        
        const products = await query(sql, params);
        
        res.json({
            success: true,
            count: products.length,
            data: products
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get services for POS
 * @route   GET /api/pos/services
 * @access  Private
 */
exports.getServices = async (req, res, next) => {
    try {
        const { category, search } = req.query;
        
        let sql = `SELECT 
                       s.id,
                       s.service_name as name,
                       s.barcode,
                       s.cat_id,
                       s.sale_price as sale_price_latli,
                       s.buy_price as cost_price,
                       1 as qty,
                       'service' as unit_name,
                       NULL as image_path,
                       'service' as item_type,
                       sc.name as category_name
                   FROM tbl_service s 
                   LEFT JOIN tbl_service_catalog sc ON s.cat_id = sc.id
                   WHERE 1=1`;
        
        const params = [];
        
        if (category) {
            sql += ' AND s.cat_id = ?';
            params.push(category);
        }
        
        if (search) {
            sql += ' AND (s.service_name LIKE ? OR s.barcode LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }
        
        sql += ' ORDER BY s.service_name LIMIT 100';
        
        const services = await query(sql, params);
        
        res.json({
            success: true,
            count: services.length,
            data: services
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get products and services for POS (combined)
 * @route   GET /api/pos/items
 * @access  Private
 */
exports.getItems = async (req, res, next) => {
    try {
        const { category, search, type, warehouse_id } = req.query;
        
        // Products query - filter by warehouse if provided, otherwise use main stock
        let productSql;
        const params = [];
        
        if (warehouse_id && warehouse_id !== '' && warehouse_id !== 'null' && warehouse_id !== 'undefined') {
            // Get products from specific warehouse - only show products that have stock in that warehouse
            productSql = `SELECT 
                            s.id as id,
                            s.name as name,
                            s.barcode as barcode,
                            s.cat_id as cat_id,
                            s.sale_price_latli as sale_price_latli,
                            s.cost_price as cost_price,
                            ws.quantity as qty,
                            (ws.quantity - COALESCE(ws.reserved_quantity, 0)) as available_qty,
                            s.unit_name as unit_name,
                            s.image_path as image_path,
                            'product' as item_type,
                            c.name as category_name
                          FROM tbl_warehouse_stock ws
                          INNER JOIN tbl_stock s ON ws.stock_id = s.id
                          LEFT JOIN tbl_catalog c ON s.cat_id = c.id
                          WHERE ws.warehouse_id = ? AND ws.quantity > 0`;
            params.push(warehouse_id);
        } else {
            // Get products from main stock
            productSql = `SELECT 
                            s.id as id,
                            s.name as name,
                            s.barcode as barcode,
                            s.cat_id as cat_id,
                            s.sale_price_latli as sale_price_latli,
                            s.cost_price as cost_price,
                            s.qty as qty,
                            s.qty as available_qty,
                            s.unit_name as unit_name,
                            s.image_path as image_path,
                            'product' as item_type,
                            c.name as category_name
                          FROM tbl_stock s 
                          LEFT JOIN tbl_catalog c ON s.cat_id = c.id
                          WHERE s.qty > 0`;
        }
        
        // Services query (not affected by warehouse)
        let serviceSql = `SELECT 
                            s.id as id,
                            service_name as name,
                            barcode,
                            cat_id,
                            sale_price as sale_price_latli,
                            buy_price as cost_price,
                            1 as qty,
                            NULL as available_qty,
                            'service' as unit_name,
                            NULL as image_path,
                            'service' as item_type,
                            sc.name as category_name
                          FROM tbl_service s 
                          LEFT JOIN tbl_service_catalog sc ON s.cat_id = sc.id
                          WHERE 1=1`;
        
        // Laboratory Items query (not affected by warehouse)
        let labItemSql = `SELECT 
                            li.id as id,
                            item_name as name,
                            barcode,
                            cat_id,
                            sale_price as sale_price_latli,
                            buy_price as cost_price,
                            1 as qty,
                            NULL as available_qty,
                            'lab-item' as unit_name,
                            NULL as image_path,
                            'laboratory-item' as item_type,
                            sc.name as category_name
                          FROM tbl_laboratory_item li 
                          LEFT JOIN tbl_service_catalog sc ON li.cat_id = sc.id
                          WHERE 1=1`;
        
        // Laboratory Packages query
        let labPackageSql = `SELECT 
                            lp.id as id,
                            package_name as name,
                            package_code as barcode,
                            NULL as cat_id,
                            total_price as sale_price_latli,
                            0 as cost_price,
                            1 as qty,
                            NULL as available_qty,
                            'package' as unit_name,
                            NULL as image_path,
                            'laboratory-package' as item_type,
                            'Laboratory Packages' as category_name
                          FROM tbl_laboratory_package lp
                          WHERE lp.is_active = 1`;
        
        if (category) {
            productSql += ' AND s.cat_id = ?';
            serviceSql += ' AND s.cat_id = ?';
            labItemSql += ' AND li.cat_id = ?';
            params.push(category, category, category);
        }
        
        if (search) {
            productSql += ' AND (s.name LIKE ? OR s.barcode LIKE ?)';
            serviceSql += ' AND (s.service_name LIKE ? OR s.barcode LIKE ?)';
            labItemSql += ' AND (li.item_name LIKE ? OR li.barcode LIKE ?)';
            labPackageSql += ' AND (lp.package_name LIKE ? OR lp.package_code LIKE ?)';
            const searchParam = `%${search}%`;
            params.push(searchParam, searchParam, searchParam, searchParam, searchParam, searchParam, searchParam, searchParam);
        }
        
        let combinedSql;
        if (type === 'products') {
            combinedSql = productSql + ' ORDER BY s.name LIMIT 50';
        } else if (type === 'services') {
            combinedSql = serviceSql + ' ORDER BY s.service_name LIMIT 50';
        } else if (type === 'laboratory-items') {
            combinedSql = labItemSql + ' ORDER BY li.item_name LIMIT 50';
        } else if (type === 'laboratory-packages') {
            combinedSql = labPackageSql + ' ORDER BY lp.package_name LIMIT 50';
        } else {
            // Get products, services, laboratory items, and packages
            combinedSql = `(${productSql} ORDER BY s.name LIMIT 50) 
                          UNION ALL 
                          (${serviceSql} ORDER BY s.service_name LIMIT 50)
                          UNION ALL
                          (${labItemSql} ORDER BY li.item_name LIMIT 50)
                          UNION ALL
                          (${labPackageSql} ORDER BY lp.package_name LIMIT 50)
                          ORDER BY name LIMIT 200`;
        }
        
        const items = await query(combinedSql, params);
        
        res.json({
            success: true,
            count: items.length,
            data: items
        });
    } catch (error) {
        console.error('Error fetching POS items:', error);
        
        // If tbl_laboratory_item doesn't exist, try querying without it
        if (error.code === 'ER_NO_SUCH_TABLE' && error.message.includes('tbl_laboratory_item')) {
            try {
                // Retry without laboratory items
                let combinedSqlWithoutLab = '';
                if (type === 'products') {
                    combinedSqlWithoutLab = productSql + ' ORDER BY s.name LIMIT 50';
                } else if (type === 'services') {
                    combinedSqlWithoutLab = serviceSql + ' ORDER BY s.service_name LIMIT 50';
                } else if (type === 'laboratory-items') {
                    // Return empty if laboratory items table doesn't exist
                    return res.json({
                        success: true,
                        count: 0,
                        data: []
                    });
                } else {
                    // Get products and services only
                    const productParams = [];
                    const serviceParams = [];
                    let productQuery = productSql;
                    let serviceQuery = serviceSql;
                    
                    if (category) {
                        productQuery += ' AND s.cat_id = ?';
                        serviceQuery += ' AND s.cat_id = ?';
                        productParams.push(category);
                        serviceParams.push(category);
                    }
                    
                    if (search) {
                        const searchParam = `%${search}%`;
                        productQuery += ' AND (s.name LIKE ? OR s.barcode LIKE ?)';
                        serviceQuery += ' AND (s.service_name LIKE ? OR s.barcode LIKE ?)';
                        productParams.push(searchParam, searchParam);
                        serviceParams.push(searchParam, searchParam);
                    }
                    
                    combinedSqlWithoutLab = `(${productQuery} ORDER BY s.name LIMIT 50) 
                                          UNION ALL 
                                          (${serviceQuery} ORDER BY s.service_name LIMIT 50)
                                          ORDER BY name LIMIT 100`;
                    params = [...productParams, ...serviceParams];
                }
                
                const items = await query(combinedSqlWithoutLab, params);
                return res.json({
                    success: true,
                    count: items.length,
                    data: items
                });
            } catch (retryError) {
                console.error('Error fetching POS items (retry):', retryError);
            }
        }
        
        res.json({
            success: true,
            count: 0,
            data: []
        });
    }
};

/**
 * @desc    Get categories for POS
 * @route   GET /api/pos/categories
 * @access  Private
 */
exports.getCategories = async (req, res, next) => {
    try {
        const categories = await models.Catalog.findAll({ limit: 100 });
        
        res.json({
            success: true,
            data: categories.data
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get next voucher number
 * @route   GET /api/pos/next-voucher
 * @access  Private
 */
exports.getNextVoucher = async (req, res, next) => {
    try {
        // Get the next voucher number by finding the maximum VNo and adding 1
        const sql = 'SELECT COALESCE(MAX(VNo), 0) + 1 as voucher FROM tbl_sale';
        const result = await query(sql);
        
        const voucherNo = result[0].voucher;
        
        res.json({
            success: true,
            voucher: voucherNo
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Process sale transaction
 * @route   POST /api/pos/sale
 * @access  Private
 */
exports.processSale = async (req, res, next) => {
    try {
        const {
            items,
            doctor_id,
            doctor_name,
            customer_name,
            patient_id,
            sub_total,
            discount,
            tax,
            total,
            payment_method,
            cashier,
            referring_doctor_id,
            referring_doctor_name,
            warehouse_id
        } = req.body;

        // Validate items
        if (!items || items.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'No items in cart'
            });
        }

        // Use transaction to ensure data consistency
        const result = await transaction(async (conn) => {
            // Get next voucher number first
            const [voucherResult] = await conn.query('SELECT COALESCE(MAX(VNo), 0) + 1 as voucher FROM tbl_sale');
            const voucherNo = voucherResult[0].voucher;
            
            // Handle deposit payment validation and processing
            if (payment_method === 'Deposit' && patient_id) {
                // Get current patient deposit balance
                const [patientResult] = await conn.query('SELECT deposit_balance FROM tbl_patient WHERE id = ?', [patient_id]);
                
                if (patientResult.length === 0) {
                    throw new Error('Patient not found');
                }
                
                const currentBalance = parseFloat(patientResult[0].deposit_balance || 0);
                if (currentBalance < total) {
                    throw new Error(`Insufficient deposit balance. Available: ${currentBalance}, Required: ${total}`);
                }
                
                // Calculate new balance after payment
                const newBalance = currentBalance - total;
                
                // Update patient deposit balance
                await conn.query('UPDATE tbl_patient SET deposit_balance = ? WHERE id = ?', [newBalance, patient_id]);
                
                // Create deposit transaction record for the withdrawal
                const depositTransactionSql = `
                    INSERT INTO tbl_deposit_transaction 
                    (patient_id, transaction_type, amount, balance_before, balance_after, description, reference_no, created_by)
                    VALUES (?, 'withdrawal', ?, ?, ?, ?, ?, ?)
                `;
                
                await conn.query(depositTransactionSql, [
                    patient_id,
                    total,
                    currentBalance,
                    newBalance,
                    `POS Sale - Voucher ${voucherNo}`,
                    `POS-${Date.now()}`,
                    req.user.id
                ]);
            }

            // Insert voucher
            const voucherSql = `INSERT INTO tbl_voucher 
                               (id, user_name, sub_total, actual_cost, dis, tax, customer_name, Date, status, payment, doctor_id, doctor_name, patient_id)
                               VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), 1, ?, ?, ?, ?)`;
            
            await conn.query(voucherSql, [
                voucherNo,
                cashier || req.user.name,
                sub_total,
                total,
                discount || 0,
                tax || 0,
                customer_name || 'Walk-in',
                payment_method || 'Cash',
                doctor_id || null,
                doctor_name || null,
                patient_id || null
            ]);

            // Insert sale items and update stock
            for (const item of items) {
                // Prefer explicit item_type; only fall back to barcode prefix if missing
                const itemType = item.item_type || null;
                const isService = itemType === 'service' || (!itemType && item.barcode && item.barcode.startsWith('S'));
                const isLabItem = itemType === 'laboratory-item' || (!itemType && item.barcode && item.barcode.startsWith('L'));

                let catId = null;
                let name = item.name;
                let costPrice = 0;
                let currentQty = 0;
                let qtyAlert = 0;

                if (!isService && !isLabItem) {
                    // Product path
                    const itemStockId = item?.stock_id ? parseInt(item.stock_id) : null;
                    const [stockItems] = itemStockId
                        ? await conn.query('SELECT * FROM tbl_stock WHERE id = ?', [itemStockId])
                        : await conn.query('SELECT * FROM tbl_stock WHERE barcode = ?', [item.barcode]);
                    if (stockItems.length === 0) {
                        throw new Error(`Item not found: ${item.barcode || itemStockId}`);
                    }
                    const stock = stockItems[0];
                    catId = stock.cat_id;
                    name = stock.name;
                    costPrice = stock.cost_price;
                    qtyAlert = stock.qty_alert;

                    // Check stock availability and deduct from warehouse if warehouse_id is provided
                    if (warehouse_id) {
                        // Check warehouse stock availability
                        const [warehouseStock] = await conn.query(
                            'SELECT quantity FROM tbl_warehouse_stock WHERE warehouse_id = ? AND stock_id = ?',
                            [warehouse_id, stock.id]
                        );
                        
                        if (warehouseStock.length === 0 || warehouseStock[0].quantity < item.quantity) {
                            const availableQty = warehouseStock.length > 0 ? warehouseStock[0].quantity : 0;
                            throw new Error(`Insufficient stock in warehouse for ${name}. Available: ${availableQty}, Required: ${item.quantity}`);
                        }
                        
                        currentQty = warehouseStock[0].quantity;
                        
                        // Create stock movement record to deduct from warehouse
                        // The trigger will automatically update warehouse stock
                        await conn.query(
                            `INSERT INTO tbl_stock_movement
                             (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id,
                              movement_date, notes, created_by)
                             VALUES ('OUT', ?, ?, ?, 'SALE', ?, NOW(), ?, ?)`,
                            [
                                warehouse_id,
                                stock.id,
                                item.quantity,
                                voucherNo,
                                `POS Sale: ${name}`,
                                req.user?.id || null
                            ]
                        );
                        console.log(`✅ Warehouse stock deducted: ${name} (${item.quantity} units) from warehouse ${warehouse_id}`);
                        
                        // Also update main stock for backward compatibility
                        await conn.query(
                            'UPDATE tbl_stock SET qty = qty - ? WHERE id = ?',
                            [item.quantity, stock.id]
                        );
                    } else {
                        // No warehouse specified - use main stock only
                        currentQty = stock.qty;
                        if (currentQty < item.quantity) {
                            throw new Error(`Insufficient stock for ${name}`);
                        }
                        
                        // Reduce main stock (prefer id, fallback to barcode)
                        let [mainUpdate] = await conn.query(
                            'UPDATE tbl_stock SET qty = qty - ? WHERE id = ?',
                            [item.quantity, stock.id]
                        );
                        let mainAffected = mainUpdate?.affectedRows ?? 0;
                        if (mainAffected === 0) {
                            const [fallbackMainUpdate] = await conn.query(
                                'UPDATE tbl_stock SET qty = qty - ? WHERE barcode = ?',
                                [item.quantity, item.barcode]
                            );
                            mainAffected = fallbackMainUpdate?.affectedRows ?? 0;
                        }
                        if (mainAffected === 0) {
                            throw new Error(`Stock not found for ${name}`);
                        }
                        console.log(`✅ Main stock deducted: ${name} (${item.quantity} units) - new qty: ${currentQty - item.quantity}`);
                    }
                    
                    // Low stock alert (use main stock quantity for alerts)
                    const finalQty = warehouse_id ? stock.qty - item.quantity : currentQty - item.quantity;
                    if (finalQty <= qtyAlert) {
                        await conn.query(
                            `INSERT INTO tbl_alert (cat_name, barcode, name, qty, qty_alert, date, status)
                             VALUES (?, ?, ?, ?, ?, CURDATE(), 1)
                             ON DUPLICATE KEY UPDATE qty = ?, date = CURDATE()`,
                            [catId, item.barcode, name, finalQty, qtyAlert, finalQty]
                        );
                    }
                } else if (isService) {
                    // Service path: fetch optional category from service, but don't touch stock
                    const [services] = await conn.query(
                        'SELECT cat_id, service_name, buy_price FROM tbl_service WHERE barcode = ?',
                        [item.barcode]
                    );
                    if (services.length > 0) {
                        catId = services[0].cat_id;
                        name = services[0].service_name || name;
                        costPrice = services[0].buy_price || 0;
                    }
                } else if (isLabItem) {
                    // Laboratory item path: fetch optional category from laboratory_item, but don't touch stock
                    const [labItems] = await conn.query(
                        'SELECT id, cat_id, item_name, buy_price FROM tbl_laboratory_item WHERE barcode = ?',
                        [item.barcode]
                    );
                    if (labItems.length > 0) {
                        catId = labItems[0].cat_id;
                        name = labItems[0].item_name || name;
                        costPrice = labItems[0].buy_price || 0;
                        // Store lab item data for order creation
                        item.labItemData = labItems[0];
                    }
                }

                const revenue = item.price * item.quantity;
                const itemDiscount = item.discount || 0;
                const profit = revenue - (costPrice * item.quantity) - itemDiscount;

                const saleSql = `INSERT INTO tbl_sale 
                                (cat_id, Barcode, Name, SaleType, SalePrice, dis, Qty, Total, Profit, VNo, Date, Cashier, cost_price, referring_doctor_id, referring_doctor_name, warehouse_id)
                                VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, CURDATE(), ?, ?, ?, ?, ?)`;
                const saleResult = await conn.query(saleSql, [
                    catId,
                    item.barcode,
                    name,
                    isService ? 'Service' : (isLabItem ? 'Laboratory Item' : (item.sale_type || 'Retail')),
                    item.price,
                    itemDiscount,
                    item.quantity,
                    revenue - itemDiscount,
                    profit,
                    voucherNo,
                    cashier || req.user.name,
                    costPrice,
                    referring_doctor_id || null,
                    referring_doctor_name || null,
                    warehouse_id || null
                ]);
                
                // Create lab order if this is a laboratory item
                if (isLabItem) {
                    const saleId = saleResult.insertId;
                    const labItemId = item.labItemData ? item.labItemData.id : null;
                    
                    // Get patient info if patient_id is provided
                    let patientName = customer_name || 'Walk-in';
                    if (patient_id) {
                        const [patientInfo] = await conn.query(
                            'SELECT name FROM tbl_patient WHERE id = ? OR patient_id = ? LIMIT 1',
                            [patient_id, patient_id]
                        );
                        if (patientInfo.length > 0) {
                            patientName = patientInfo[0].name;
                        }
                    }
                    
                    // Generate order number
                    const [orderCount] = await conn.query(
                        'SELECT COALESCE(MAX(CAST(SUBSTRING(order_number, 4) AS UNSIGNED)), 0) + 1 as next_num FROM tbl_lab_order WHERE order_number LIKE "LAB%"'
                    );
                    const orderNumber = `LAB${String(orderCount[0].next_num).padStart(8, '0')}`;
                    
                    // Create lab order
                    await conn.query(
                        `INSERT INTO tbl_lab_order 
                         (order_number, sale_id, sale_vno, lab_item_id, lab_item_name, patient_id, patient_name, 
                          doctor_id, doctor_name, referring_doctor_id, referring_doctor_name, 
                          order_date, total_amount, status, created_by)
                         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), ?, 'Ordered', ?)`,
                        [
                            orderNumber,
                            saleId,
                            voucherNo,
                            labItemId,
                            name,
                            patient_id || customer_name || 'Walk-in',
                            patientName,
                            doctor_id || null,
                            doctor_name || null,
                            referring_doctor_id || null,
                            referring_doctor_name || null,
                            revenue - itemDiscount,
                            req.user.id
                        ]
                    );
                }
            }

            // Create commission if there's a referring doctor
            if (referring_doctor_id && referring_doctor_id !== '' && referring_doctor_id !== 'null' && referring_doctor_id !== 'undefined') {
                try {
                    console.log('Creating commission for referring doctor ID:', referring_doctor_id);
                    
                    // Get referring doctor's commission settings
                    const [referringDoctorRows] = await conn.query(
                        'SELECT commission_type, commission_value FROM tbl_referring_doctor WHERE id = ? AND active = 1',
                        [referring_doctor_id]
                    );
                    
                    if (referringDoctorRows && referringDoctorRows.length > 0) {
                        const doctor = referringDoctorRows[0];
                        let commission_amount = 0;
                        let commission_rate = null;
                        
                        if (doctor.commission_type === 'percentage') {
                            commission_amount = (total * doctor.commission_value) / 100;
                            commission_rate = doctor.commission_value;
                        } else if (doctor.commission_type === 'fixed') {
                            commission_amount = doctor.commission_value;
                        }
                        
                        console.log('Commission calculated:', { commission_amount, commission_rate, commission_type: doctor.commission_type });
                        
                        if (commission_amount > 0) {
                            // Create commission record
                            const commissionResult = await conn.query(
                                `INSERT INTO tbl_referral_commission 
                                (referring_doctor_id, transaction_type, sale_id, voucher_no, patient_id, patient_name,
                                 transaction_date, transaction_amount, commission_type, commission_rate, commission_amount,
                                 payment_status, created_by)
                                VALUES (?, 'sale', NULL, ?, ?, ?, CURDATE(), ?, ?, ?, ?, 'pending', ?)`,
                                [referring_doctor_id, voucherNo, patient_id || null, customer_name || 'Walk-in',
                                 total, doctor.commission_type, commission_rate, commission_amount, req.user?.id || null]
                            );
                            
                            console.log('Commission record created with ID:', commissionResult[0]?.insertId);
                            
                            // Update referring doctor totals
                            await conn.query(
                                `UPDATE tbl_referring_doctor 
                                SET total_referrals = COALESCE(total_referrals, 0) + 1,
                                    total_revenue = COALESCE(total_revenue, 0) + ?,
                                    total_commission_earned = COALESCE(total_commission_earned, 0) + ?
                                WHERE id = ?`,
                                [total, commission_amount, referring_doctor_id]
                            );
                            
                            console.log('Referring doctor totals updated');
                        } else {
                            console.log('Commission amount is 0, skipping commission creation');
                        }
                    } else {
                        console.log('Referring doctor not found or inactive:', referring_doctor_id);
                    }
                } catch (commissionError) {
                    console.error('Error creating commission:', commissionError);
                    console.error('Commission error stack:', commissionError.stack);
                    // Don't fail the sale if commission creation fails
                }
            } else {
                console.log('No referring doctor ID provided or invalid:', referring_doctor_id);
            }

            return { voucherNo };
        });

        res.status(201).json({
            success: true,
            message: 'Sale completed successfully',
            data: result
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Save draft sale
 * @route   POST /api/pos/draft
 * @access  Private
 */
exports.saveDraft = async (req, res, next) => {
    try {
        const { 
            items, 
            customer_name, 
            cashier,
            doctor_id,
            doctor_name,
            patient_id,
            sub_total,
            discount,
            tax,
            total,
            payment_method,
            referring_doctor_id,
            referring_doctor_name
        } = req.body;

        // Generate draft number
        const countResult = await query('SELECT COUNT(*) as count FROM tbl_sale_draft_no');
        const draftNo = 1000 + countResult[0].count + 1;

        // Calculate totals if not provided
        const calculatedSubTotal = sub_total || items.reduce((sum, item) => sum + (item.price * item.quantity), 0);
        const calculatedDiscount = discount || 0;
        const calculatedTax = tax || 0;
        const calculatedTotal = total || (calculatedSubTotal - calculatedDiscount + calculatedTax);

        // Insert draft voucher
        const voucherSql = `INSERT INTO tbl_sale_draft_no 
                           (id, user_name, customer_name, Date, status, sub_total, actual_cost, dis, tax, payment, doctor_id, doctor_name, patient_id, referring_doctor_id, referring_doctor_name)
                           VALUES (?, ?, ?, NOW(), 0, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;
        
        await query(voucherSql, [
            draftNo, 
            cashier || req.user.name, 
            customer_name || 'Walk-in',
            calculatedSubTotal,
            calculatedTotal,
            calculatedDiscount,
            calculatedTax,
            payment_method || 'Cash',
            doctor_id || null,
            doctor_name || null,
            patient_id || null,
            referring_doctor_id || null,
            referring_doctor_name || null
        ]);

        // Insert draft items
        for (const item of items) {
            const itemDiscount = item.discount || 0;
            const itemTotal = (item.price * item.quantity) - itemDiscount;
            
            const saleSql = `INSERT INTO tbl_sale_draft 
                            (Barcode, Name, SaleType, SalePrice, dis, Qty, Total, VNo, Date, Cashier, cat_id)
                            VALUES (?, ?, ?, ?, ?, ?, ?, ?, CURDATE(), ?, ?)`;
            
            const isService = item.item_type === 'service' || (item.barcode && item.barcode.startsWith('S'));
            const isLabItem = item.item_type === 'laboratory-item' || (item.barcode && item.barcode.startsWith('L'));
            
            await query(saleSql, [
                item.barcode,
                item.name,
                item.sale_type || (isService ? 'Service' : (isLabItem ? 'Laboratory Item' : 'Retail')),
                item.price,
                itemDiscount,
                item.quantity,
                itemTotal,
                draftNo,
                cashier || req.user.name,
                item.cat_id || null
            ]);
        }

        res.status(201).json({
            success: true,
            message: 'Draft saved successfully',
            data: { draftNo }
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get payment methods
 * @route   GET /api/pos/payment-methods
 * @access  Private
 */
exports.getPaymentMethods = async (req, res, next) => {
    try {
        const methods = await models.PaymentMethod.findAll({ limit: 20 });
        
        res.json({
            success: true,
            data: methods.data
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get all drafts
 * @route   GET /api/pos/drafts
 * @access  Private
 */
exports.getDrafts = async (req, res, next) => {
    try {
        const sql = `SELECT 
                        d.id,
                        d.user_name,
                        d.customer_name,
                        d.Date,
                        d.status,
                        d.sub_total,
                        d.actual_cost as total,
                        d.dis as discount,
                        d.tax,
                        d.payment,
                        d.doctor_id,
                        d.doctor_name,
                        d.patient_id,
                        d.referring_doctor_id,
                        d.referring_doctor_name,
                        COUNT(i.id) as item_count
                     FROM tbl_sale_draft_no d
                     LEFT JOIN tbl_sale_draft i ON d.id = i.VNo
                     WHERE d.status = 0
                     GROUP BY d.id
                     ORDER BY d.Date DESC
                     LIMIT 100`;
        
        const drafts = await query(sql);

        res.json({
            success: true,
            count: drafts.length,
            data: drafts
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get draft by ID
 * @route   GET /api/pos/draft/:id
 * @access  Private
 */
exports.getDraft = async (req, res, next) => {
    try {
        const { id } = req.params;

        const [draft, items] = await Promise.all([
            query('SELECT * FROM tbl_sale_draft_no WHERE id = ? AND status = 0', [id]),
            query('SELECT * FROM tbl_sale_draft WHERE VNo = ?', [id])
        ]);

        if (draft.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Draft not found'
            });
        }

        res.json({
            success: true,
            data: {
                draft: draft[0],
                items: items.map(item => ({
                    barcode: item.Barcode,
                    name: item.Name,
                    price: item.SalePrice,
                    quantity: item.Qty,
                    discount: item.dis || 0,
                    sale_type: item.SaleType,
                    item_type: item.SaleType === 'Service' ? 'service' : (item.SaleType === 'Laboratory Item' ? 'laboratory-item' : 'product'),
                    cat_id: item.cat_id
                }))
            }
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Delete draft
 * @route   DELETE /api/pos/draft/:id
 * @access  Private
 */
exports.deleteDraft = async (req, res, next) => {
    try {
        const { id } = req.params;

        // Check if draft exists
        const [draft] = await query('SELECT * FROM tbl_sale_draft_no WHERE id = ? AND status = 0', [id]);
        
        if (draft.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Draft not found'
            });
        }

        // Delete draft items first
        await query('DELETE FROM tbl_sale_draft WHERE VNo = ?', [id]);
        
        // Delete draft header
        await query('DELETE FROM tbl_sale_draft_no WHERE id = ?', [id]);

        res.json({
            success: true,
            message: 'Draft deleted successfully'
        });

    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get sale by voucher number
 * @route   GET /api/pos/voucher/:vno
 * @access  Private
 */
exports.getVoucherDetails = async (req, res, next) => {
    try {
        const { vno } = req.params;

        const [voucher, items] = await Promise.all([
            query('SELECT * FROM tbl_voucher WHERE id = ?', [vno]),
            query('SELECT * FROM tbl_sale WHERE VNo = ?', [vno])
        ]);

        if (voucher.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Voucher not found'
            });
        }

        res.json({
            success: true,
            data: {
                voucher: voucher[0],
                items
            }
        });

    } catch (error) {
        next(error);
    }
};

