/**
 * Product/Stock Management Controller
 * Handles product CRUD with image upload
 */

const { query } = require('../config/database');
const { recordDelete } = require('../utils/delete-history');
const path = require('path');
const fs = require('fs');

/**
 * @desc    Get all products
 * @route   GET /api/products
 * @access  Private
 */
exports.getProducts = async (req, res, next) => {
    try {
        const page = parseInt(req.query.page) || 1;
        const limit = parseInt(req.query.limit) || 20;
        const { category, search } = req.query;
        const offset = (page - 1) * limit;

        let sql = `SELECT s.*, c.name as category_name 
                   FROM tbl_stock s 
                   LEFT JOIN tbl_catalog c ON s.cat_id = c.id
                   WHERE 1=1`;
        
        const params = [];

        if (category) {
            sql += ' AND s.cat_id = ?';
            params.push(category);
        }

        if (search) {
            sql += ' AND (s.name LIKE ? OR s.barcode LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }

        sql += ' ORDER BY s.name LIMIT ? OFFSET ?';
        params.push(limit, offset);

        const products = await query(sql, params);

        // Get total count
        let countSql = `SELECT COUNT(*) as total FROM tbl_stock s WHERE 1=1`;
        const countParams = [];
        
        if (category) {
            countSql += ' AND s.cat_id = ?';
            countParams.push(category);
        }
        
        if (search) {
            countSql += ' AND (s.name LIKE ? OR s.barcode LIKE ?)';
            countParams.push(`%${search}%`, `%${search}%`);
        }

        const countResult = await query(countSql, countParams);

        res.json({
            success: true,
            count: products.length,
            total: countResult[0].total,
            page: page,
            pages: Math.ceil(countResult[0].total / limit),
            data: products
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get single product
 * @route   GET /api/products/:id
 * @access  Private
 */
exports.getProduct = async (req, res, next) => {
    try {
        const sql = `SELECT s.*, c.name as category_name, 
                            sup.name as supplier_name, sup.id as supplier_id
                     FROM tbl_stock s 
                     LEFT JOIN tbl_catalog c ON s.cat_id = c.id
                     LEFT JOIN tbl_supplier sup ON s.supplier_id = sup.id
                     WHERE s.id = ?`;
        
        const products = await query(sql, [req.params.id]);

        if (products.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Product not found'
            });
        }

        res.json({
            success: true,
            data: products[0]
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get warehouse stock for a product
 * @route   GET /api/products/:id/warehouse-stock
 * @access  Private
 */
exports.getProductWarehouseStock = async (req, res, next) => {
    try {
        const sql = `SELECT ws.*, w.name as warehouse_name, w.code as warehouse_code
                     FROM tbl_warehouse_stock ws
                     INNER JOIN tbl_warehouse w ON ws.warehouse_id = w.id
                     WHERE ws.stock_id = ?
                     ORDER BY w.name`;
        
        const warehouseStocks = await query(sql, [req.params.id]);

        res.json({
            success: true,
            data: warehouseStocks
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create new product
 * @route   POST /api/products
 * @access  Private
 */
exports.createProduct = async (req, res, next) => {
    let image_path = null;
    let stockId = null;
    
    try {
        const {
            barcode,
            name,
            cat_id,
            cat_id_2,
            supplier_id,
            qty,
            qty_alert,
            unit_name,
            cost_price,
            sale_price_latli,
            sale_price_latkar,
            remark,
            expire_date,
            alert_month,
            margin,
            warehouse_id,
            image_path: bodyImagePath
        } = req.body;

        // Validate required fields
        if (!name) {
            return res.status(400).json({
                success: false,
                message: 'Product name is required'
            });
        }

        if (!cost_price || parseFloat(cost_price) <= 0) {
            return res.status(400).json({
                success: false,
                message: 'Valid cost price is required'
            });
        }

        if (!sale_price_latli || parseFloat(sale_price_latli) <= 0) {
            return res.status(400).json({
                success: false,
                message: 'Valid sale price is required'
            });
        }

        // Validate warehouse_id is required
        if (!warehouse_id) {
            return res.status(400).json({
                success: false,
                message: 'Warehouse assignment is required. Please select a warehouse.'
            });
        }

        // Validate warehouse exists
        const warehouse = await query('SELECT id, name FROM tbl_warehouse WHERE id = ?', [warehouse_id]);
        if (warehouse.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'Selected warehouse does not exist'
            });
        }

        // Get image path if uploaded via file or provided in body
        image_path = bodyImagePath || null;
        if (req.file) {
            image_path = `/uploads/products/${req.file.filename}`;
        }

        // Check if barcode already exists (only if barcode is provided)
        if (barcode) {
            const existingProduct = await query('SELECT id FROM tbl_stock WHERE barcode = ?', [barcode]);
            if (existingProduct.length > 0) {
                // Delete uploaded file if validation fails
                if (req.file) {
                    const filePath = path.join(__dirname, '../uploads/products', req.file.filename);
                    if (fs.existsSync(filePath)) {
                        fs.unlinkSync(filePath);
                    }
                }
                return res.status(400).json({
                    success: false,
                    message: 'Product with this barcode already exists'
                });
            }
        }

        // Calculate alert date if expire date provided
        let alert_date = null;
        if (expire_date && alert_month) {
            const expireDateTime = new Date(expire_date);
            if (!isNaN(expireDateTime.getTime())) {
                expireDateTime.setMonth(expireDateTime.getMonth() - parseInt(alert_month));
                alert_date = expireDateTime.toISOString().split('T')[0];
            }
        }

        const sql = `INSERT INTO tbl_stock 
                     (barcode, name, cat_id, cat_id_2, supplier_id, qty, qty_alert, unit_name, 
                      cost_price, sale_price_latli, sale_price_latkar, remark, 
                      expire_date, alert_month, alert_date, margin, image_path)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;

        const result = await query(sql, [
            barcode || null,
            name,
            cat_id || null,
            cat_id_2 || null,
            supplier_id || null,
            parseInt(qty) || 0,
            parseInt(qty_alert) || 10,
            unit_name || null,
            parseFloat(cost_price),
            parseFloat(sale_price_latli),
            parseFloat(sale_price_latkar) || parseFloat(sale_price_latli),
            remark || '',
            expire_date || null,
            parseInt(alert_month) || null,
            alert_date,
            parseFloat(margin) || null,
            image_path
        ]);

        stockId = result.insertId;
        const productQty = parseInt(qty) || 0;

        // Assign stock to warehouse (required)
        if (stockId && warehouse_id && productQty > 0) {
            try {
                // Create stock movement record for warehouse
                // The trigger trg_update_warehouse_stock_on_movement will automatically
                // update tbl_warehouse_stock when this record is inserted
                await query(
                    `INSERT INTO tbl_stock_movement
                     (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id,
                      movement_date, notes, created_by)
                     VALUES ('IN', ?, ?, ?, 'ASSIGNMENT', ?, NOW(), ?, ?)`,
                    [
                        warehouse_id,
                        stockId,
                        productQty,
                        stockId, // Link to product record
                        `Product created: ${name}`,
                        req.user?.id || null
                    ]
                );
                
                console.log(`✅ Warehouse stock assigned for product: ${name} (${productQty} units) in warehouse ${warehouse_id}`);
            } catch (warehouseError) {
                console.error('Warehouse stock assignment error:', warehouseError.message);
                // Delete the product if warehouse assignment fails
                await query('DELETE FROM tbl_stock WHERE id = ?', [stockId]);
                // Delete uploaded file
                if (req.file) {
                    const filePath = path.join(__dirname, '../uploads/products', req.file.filename);
                    if (fs.existsSync(filePath)) {
                        fs.unlinkSync(filePath);
                    }
                }
                return res.status(500).json({
                    success: false,
                    message: 'Failed to assign stock to warehouse: ' + warehouseError.message
                });
            }
        } else if (productQty > 0) {
            // If quantity > 0 but warehouse assignment failed, rollback
            await query('DELETE FROM tbl_stock WHERE id = ?', [stockId]);
            if (req.file) {
                const filePath = path.join(__dirname, '../uploads/products', req.file.filename);
                if (fs.existsSync(filePath)) {
                    fs.unlinkSync(filePath);
                }
            }
            return res.status(400).json({
                success: false,
                message: 'Warehouse assignment is required when quantity is greater than 0'
            });
        }

        res.status(201).json({
            success: true,
            message: 'Product created and assigned to warehouse successfully',
            data: {
                id: stockId,
                barcode,
                name,
                image_path,
                warehouse_id
            }
        });
    } catch (error) {
        // Delete uploaded file if database insert fails
        if (req.file) {
            const filePath = path.join(__dirname, '../uploads/products', req.file.filename);
            if (fs.existsSync(filePath)) {
                fs.unlinkSync(filePath);
            }
        }
        // Delete product if it was created but warehouse assignment failed
        if (stockId) {
            try {
                await query('DELETE FROM tbl_stock WHERE id = ?', [stockId]);
            } catch (deleteError) {
                console.error('Error deleting product after failure:', deleteError);
            }
        }
        next(error);
    }
};

/**
 * @desc    Update product
 * @route   PUT /api/products/:id
 * @access  Private
 */
exports.updateProduct = async (req, res, next) => {
    try {
        console.log('Updating product:', req.params.id, req.body); // Debug log
        
        const {
            barcode,
            name,
            cat_id,
            cat_id_2,
            supplier_id,
            qty,
            qty_alert,
            unit_name,
            cost_price,
            sale_price_latli,
            sale_price_latkar,
            remark,
            expire_date,
            alert_month,
            margin,
            image_path: bodyImagePath
        } = req.body;

        // Validate required fields
        if (!name) {
            return res.status(400).json({
                success: false,
                message: 'Product name is required'
            });
        }

        // Get existing product
        const existingProducts = await query('SELECT * FROM tbl_stock WHERE id = ?', [req.params.id]);
        
        if (existingProducts.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Product not found'
            });
        }

        const existingProduct = existingProducts[0];
        let image_path = existingProduct.image_path;
        
        // Quantity should not be updated directly - it's calculated from warehouse stocks
        // Only allow qty update for backward compatibility if explicitly provided, but prefer warehouse stock
        let finalQty = existingProduct.qty; // Keep existing quantity by default
        if (qty !== undefined && qty !== null) {
            // If qty is provided, we'll calculate it from warehouse stocks instead
            // But for now, we'll ignore direct qty updates when editing
            console.log('Quantity update ignored - quantity is managed through warehouse operations');
        }

        // Handle new image upload or image path from body
        if (req.file) {
            // Delete old image if exists
            if (existingProduct.image_path) {
                const oldImagePath = path.join(__dirname, '..', existingProduct.image_path);
                if (fs.existsSync(oldImagePath)) {
                    fs.unlinkSync(oldImagePath);
                }
            }
            image_path = `/uploads/products/${req.file.filename}`;
        } else if (bodyImagePath) {
            // Use image path from request body (uploaded via separate endpoint)
            image_path = bodyImagePath;
        }

        // Calculate alert date
        let alert_date = null;
        if (expire_date && alert_month) {
            const expireDateTime = new Date(expire_date);
            expireDateTime.setMonth(expireDateTime.getMonth() - parseInt(alert_month));
            alert_date = expireDateTime.toISOString().split('T')[0];
        }

        // Update main stock quantity from warehouse stocks (calculate total)
        let calculatedQty = existingProduct.qty; // Default to existing
        try {
            const warehouseStocks = await query(
                'SELECT SUM(quantity) as total FROM tbl_warehouse_stock WHERE stock_id = ?',
                [req.params.id]
            );
            if (warehouseStocks.length > 0 && warehouseStocks[0].total !== null) {
                calculatedQty = parseInt(warehouseStocks[0].total) || 0;
            }
        } catch (error) {
            console.log('Could not calculate quantity from warehouse stocks, using existing:', error.message);
        }

        const sql = `UPDATE tbl_stock 
                     SET barcode = ?, name = ?, cat_id = ?, cat_id_2 = ?, supplier_id = ?, qty = ?, 
                         qty_alert = ?, unit_name = ?, cost_price = ?, 
                         sale_price_latli = ?, sale_price_latkar = ?, remark = ?,
                         expire_date = ?, alert_month = ?, alert_date = ?, margin = ?, image_path = ?
                     WHERE id = ?`;

        const result = await query(sql, [
            barcode || null,
            name,
            cat_id || null,
            cat_id_2 || null,
            supplier_id || null,
            calculatedQty, // Use calculated quantity from warehouse stocks
            parseInt(qty_alert) || 10,
            unit_name || null,
            parseFloat(cost_price) || 0,
            parseFloat(sale_price_latli) || 0,
            parseFloat(sale_price_latkar) || parseFloat(sale_price_latli) || 0,
            remark || null,
            expire_date || null,
            parseInt(alert_month) || null,
            alert_date,
            parseFloat(margin) || null,
            image_path,
            req.params.id
        ]);

        console.log('Update result:', result); // Debug log
        console.log('Image path saved:', image_path); // Debug log
        console.log('Quantity updated from warehouse stocks:', calculatedQty);

        res.json({
            success: true,
            message: 'Product updated successfully',
            data: {
                id: req.params.id,
                affectedRows: result.affectedRows,
                image_path: image_path // Return the image path
            }
        });
    } catch (error) {
        console.error('Update product error:', error); // Debug log
        
        // Delete uploaded file if update fails
        if (req.file) {
            const filePath = path.join(__dirname, '../uploads/products', req.file.filename);
            if (fs.existsSync(filePath)) {
                fs.unlinkSync(filePath);
            }
        }
        
        res.status(500).json({
            success: false,
            message: 'Error updating product: ' + error.message
        });
    }
};

/**
 * @desc    Delete product
 * @route   DELETE /api/products/:id
 * @access  Private
 */
exports.deleteProduct = async (req, res, next) => {
    try {
        const stockId = req.params.id;
        
        // Get product to delete image
        const products = await query('SELECT * FROM tbl_stock WHERE id = ?', [stockId]);
        
        if (products.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Product not found'
            });
        }

        const product = products[0];

        // Check for references before attempting deletion
        let references = {
            stock_movements: 0,
            warehouse_stock: 0,
            purchases: 0,
            sales: 0
        };

        // Check tbl_stock_movement
        try {
            const movements = await query('SELECT COUNT(*) as count FROM tbl_stock_movement WHERE stock_id = ?', [stockId]);
            references.stock_movements = parseInt(movements[0]?.count) || 0;
        } catch (err) {
            console.error('Error checking stock movements:', err);
        }

        // Check tbl_warehouse_stock
        try {
            const warehouseStock = await query('SELECT COUNT(*) as count FROM tbl_warehouse_stock WHERE stock_id = ?', [stockId]);
            references.warehouse_stock = parseInt(warehouseStock[0]?.count) || 0;
        } catch (err) {
            console.error('Error checking warehouse stock:', err);
        }

        // Check tbl_purchase (by barcode or name)
        try {
            const purchases = await query('SELECT COUNT(*) as count FROM tbl_purchase WHERE barcode = ? OR name = ?', [product.barcode || '', product.name || '']);
            references.purchases = parseInt(purchases[0]?.count) || 0;
        } catch (err) {
            console.error('Error checking purchases:', err);
        }

        // Check tbl_sale (by barcode or name)
        try {
            const sales = await query('SELECT COUNT(*) as count FROM tbl_sale WHERE Barcode = ? OR Name = ?', [product.barcode || '', product.name || '']);
            references.sales = parseInt(sales[0]?.count) || 0;
        } catch (err) {
            console.error('Error checking sales:', err);
        }

        // If there are references, check if force delete is requested
        const totalReferences = references.stock_movements + references.warehouse_stock + references.purchases + references.sales;
        const forceDelete = req.query.force === 'true' || req.body.force === true;
        
        if (totalReferences > 0 && !forceDelete) {
            const referenceDetails = [];
            if (references.stock_movements > 0) referenceDetails.push(`${references.stock_movements} stock movement(s)`);
            if (references.warehouse_stock > 0) referenceDetails.push(`${references.warehouse_stock} warehouse stock record(s)`);
            if (references.purchases > 0) referenceDetails.push(`${references.purchases} purchase(s)`);
            if (references.sales > 0) referenceDetails.push(`${references.sales} sale(s)`);

            return res.status(400).json({
                success: false,
                message: `Cannot delete product because it is referenced in: ${referenceDetails.join(', ')}. Please remove all references first or use force delete.`,
                references: references
            });
        }

        // Force delete: Clean up references first
        if (totalReferences > 0 && forceDelete) {
            console.log(`Force deleting product ${stockId} - cleaning up ${totalReferences} references...`);
            
            // Delete stock movements
            if (references.stock_movements > 0) {
                try {
                    await query('DELETE FROM tbl_stock_movement WHERE stock_id = ?', [stockId]);
                    console.log(`Deleted ${references.stock_movements} stock movement(s)`);
                } catch (err) {
                    console.error('Error deleting stock movements:', err);
                }
            }

            // Delete warehouse stock
            if (references.warehouse_stock > 0) {
                try {
                    await query('DELETE FROM tbl_warehouse_stock WHERE stock_id = ?', [stockId]);
                    console.log(`Deleted ${references.warehouse_stock} warehouse stock record(s)`);
                } catch (err) {
                    console.error('Error deleting warehouse stock:', err);
                }
            }

            // Note: We don't delete purchases and sales as they are important historical records
            // But we can set them to reference NULL or keep them as-is
            if (references.purchases > 0 || references.sales > 0) {
                console.log(`Warning: Product has ${references.purchases} purchase(s) and ${references.sales} sale(s) - these will remain in history`);
            }
        }

        // Delete product from database
        try {
            await query('DELETE FROM tbl_stock WHERE id = ?', [stockId]);
        } catch (error) {
            // Foreign key constraint (fallback check)
            const errorCode = error.code || error.errno || error.original?.code || error.original?.errno;
            const errorMessage = (error.message || error.original?.message || '').toLowerCase();
            
            console.error('Delete product error details:', {
                code: errorCode,
                errno: error.errno || error.original?.errno,
                message: error.message || error.original?.message,
                sqlState: error.sqlState || error.original?.sqlState
            });
            
            if (errorCode === 'ER_ROW_IS_REFERENCED_2' || 
                errorCode === 1451 || 
                errorCode === '23000' ||
                errorMessage.includes('foreign key constraint') ||
                errorMessage.includes('cannot delete or update a parent row') ||
                errorMessage.includes('a foreign key constraint fails')) {
                
                // Re-check references if we hit a foreign key error
                const refDetails = [];
                if (references.stock_movements > 0) refDetails.push(`${references.stock_movements} stock movement(s)`);
                if (references.warehouse_stock > 0) refDetails.push(`${references.warehouse_stock} warehouse stock record(s)`);
                if (references.purchases > 0) refDetails.push(`${references.purchases} purchase(s)`);
                if (references.sales > 0) refDetails.push(`${references.sales} sale(s)`);
                
                const refMessage = refDetails.length > 0 
                    ? `Cannot delete product because it is referenced in: ${refDetails.join(', ')}. Please remove all references first or use force delete.`
                    : 'Cannot delete product because it is referenced in other records. Please remove all references first or use force delete.';
                
                console.log('Returning error with references:', references);
                
                return res.status(400).json({
                    success: false,
                    message: refMessage,
                    references: references || {
                        stock_movements: 0,
                        warehouse_stock: 0,
                        purchases: 0,
                        sales: 0
                    }
                });
            }
            throw error;
        }

        // Record delete history (non-blocking - don't fail deletion if this fails)
        try {
            await recordDelete({
                tableName: 'tbl_stock',
                recordId: req.params.id,
                oldData: product,
                deletedBy: req.user?.id ?? null,
                username: req.user?.name || req.user?.username || null,
                reason: req.body?.reason || 'Product deleted',
                ip: req.ip || null,
                userAgent: req.get('user-agent') || null,
            });
        } catch (historyError) {
            console.error('Error recording delete history:', historyError);
            // Continue with deletion even if history recording fails
        }

        // Delete image file if exists
        if (product.image_path) {
            const imagePath = path.join(__dirname, '..', product.image_path);
            if (fs.existsSync(imagePath)) {
                fs.unlinkSync(imagePath);
            }
        }

        res.json({
            success: true,
            message: 'Product deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Generate barcode (auto-increment; start 100001 when empty)
 * @route   GET /api/products/generate-barcode
 * @access  Private
 */
exports.generateBarcode = async (req, res, next) => {
    try {
        // Fetch recent barcodes and derive numeric sequences in JS to avoid DB regex portability issues
        const rows = await query(
            `SELECT barcode FROM tbl_stock 
             WHERE barcode IS NOT NULL AND barcode <> '' 
             ORDER BY id DESC 
             LIMIT 5000`
        );

        const toDigits = (val) => String(val || '').replace(/\D/g, '');

        const numericEntries = rows
            .map(r => {
                const digits = toDigits(r.barcode);
                return digits.length ? { digits, len: digits.length, num: parseInt(digits, 10) } : null;
            })
            .filter(e => e && !isNaN(e.num));

        let barcode = '100001'; // default starting point when no stock

        if (numericEntries.length > 0) {
            const maxEntry = numericEntries.reduce((max, entry) => {
                if (entry.num > max.num) return entry;
                return max;
            }, { num: -1, len: 6, digits: '' });

            const nextNumber = maxEntry.num + 1;
            const targetLength = Math.max(maxEntry.len, String(nextNumber).length);
            barcode = String(nextNumber).padStart(targetLength, '0');
        }

        res.json({ success: true, barcode });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update stock quantity
 * @route   PUT /api/products/:id/stock
 * @access  Private
 */
exports.updateStock = async (req, res, next) => {
    try {
        const { qty, operation } = req.body; // operation: 'add' or 'subtract'

        const products = await query('SELECT qty FROM tbl_stock WHERE id = ?', [req.params.id]);
        
        if (products.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Product not found'
            });
        }

        let newQty = products[0].qty;
        
        if (operation === 'add') {
            newQty += parseInt(qty);
        } else if (operation === 'subtract') {
            newQty -= parseInt(qty);
            if (newQty < 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Insufficient stock'
                });
            }
        } else {
            newQty = parseInt(qty);
        }

        await query('UPDATE tbl_stock SET qty = ? WHERE id = ?', [newQty, req.params.id]);

        res.json({
            success: true,
            message: 'Stock updated successfully',
            data: { qty: newQty }
        });
    } catch (error) {
        next(error);
    }
};

