// Referring Doctor Controller
const { query } = require('../config/database');

// Get all referring doctors
exports.getAllReferringDoctors = async (req, res, next) => {
    try {
        const { active, search, limit = 100, offset = 0 } = req.query;
        
        let sql = 'SELECT * FROM tbl_referring_doctor WHERE 1=1';
        const params = [];
        
        if (active !== undefined) {
            sql += ' AND active = ?';
            params.push(active === 'true' || active === '1' ? 1 : 0);
        }
        
        if (search) {
            sql += ' AND (name LIKE ? OR clinic_name LIKE ? OR specialty LIKE ?)';
            const searchTerm = `%${search}%`;
            params.push(searchTerm, searchTerm, searchTerm);
        }
        
        sql += ' ORDER BY name ASC LIMIT ? OFFSET ?';
        params.push(parseInt(limit), parseInt(offset));
        
        const doctors = await query(sql, params);
        
        // Get total count
        let countSql = 'SELECT COUNT(*) as total FROM tbl_referring_doctor WHERE 1=1';
        const countParams = [];
        
        if (active !== undefined) {
            countSql += ' AND active = ?';
            countParams.push(active === 'true' || active === '1' ? 1 : 0);
        }
        
        if (search) {
            countSql += ' AND (name LIKE ? OR clinic_name LIKE ? OR specialty LIKE ?)';
            const searchTerm = `%${search}%`;
            countParams.push(searchTerm, searchTerm, searchTerm);
        }
        
        const [{ total }] = await query(countSql, countParams);
        
        res.json({
            success: true,
            data: doctors,
            pagination: {
                total,
                limit: parseInt(limit),
                offset: parseInt(offset),
                hasMore: (parseInt(offset) + doctors.length) < total
            }
        });
    } catch (error) {
        next(error);
    }
};

// Get single referring doctor by ID
exports.getReferringDoctorById = async (req, res, next) => {
    try {
        const { id } = req.params;
        
        const [doctor] = await query(
            'SELECT * FROM tbl_referring_doctor WHERE id = ?',
            [id]
        );
        
        if (!doctor) {
            return res.status(404).json({
                success: false,
                message: 'Referring doctor not found'
            });
        }
        
        res.json({
            success: true,
            data: doctor
        });
    } catch (error) {
        next(error);
    }
};

// Create new referring doctor
exports.createReferringDoctor = async (req, res, next) => {
    try {
        const {
            name,
            specialty,
            qualification,
            clinic_name,
            phone,
            email,
            address,
            city,
            state,
            commission_type = 'none',
            commission_value = 0,
            commission_applies_to = 'all',
            is_internal = 0,
            internal_doctor_id,
            registration_number,
            notes,
            photo_path,
            active = 1
        } = req.body;
        
        // Validation
        if (!name) {
            return res.status(400).json({
                success: false,
                message: 'Doctor name is required'
            });
        }
        
        const result = await query(
            `INSERT INTO tbl_referring_doctor 
            (name, specialty, qualification, clinic_name, phone, email, address, city, state,
             commission_type, commission_value, commission_applies_to, is_internal, internal_doctor_id,
             registration_number, notes, photo_path, active, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [name, specialty, qualification, clinic_name, phone, email, address, city, state,
             commission_type, commission_value, commission_applies_to, is_internal, internal_doctor_id,
             registration_number, notes, photo_path, active, req.user?.id]
        );
        
        const [newDoctor] = await query(
            'SELECT * FROM tbl_referring_doctor WHERE id = ?',
            [result.insertId]
        );
        
        res.status(201).json({
            success: true,
            message: 'Referring doctor created successfully',
            data: newDoctor
        });
    } catch (error) {
        if (error.code === 'ER_DUP_ENTRY') {
            return res.status(400).json({
                success: false,
                message: 'Email already exists'
            });
        }
        next(error);
    }
};

// Update referring doctor
exports.updateReferringDoctor = async (req, res, next) => {
    try {
        const { id } = req.params;
        const updateData = req.body;
        
        // Check if doctor exists
        const [existing] = await query(
            'SELECT id FROM tbl_referring_doctor WHERE id = ?',
            [id]
        );
        
        if (!existing) {
            return res.status(404).json({
                success: false,
                message: 'Referring doctor not found'
            });
        }
        
        // Build update query
        const allowedFields = [
            'name', 'specialty', 'qualification', 'clinic_name', 'phone', 'email', 
            'address', 'city', 'state', 'commission_type', 'commission_value',
            'commission_applies_to', 'is_internal', 'internal_doctor_id',
            'registration_number', 'notes', 'photo_path', 'active'
        ];
        
        const updates = [];
        const values = [];
        
        allowedFields.forEach(field => {
            if (updateData[field] !== undefined) {
                updates.push(`${field} = ?`);
                values.push(updateData[field]);
            }
        });
        
        if (updates.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'No valid fields to update'
            });
        }
        
        updates.push('updated_by = ?');
        values.push(req.user?.id);
        values.push(id);
        
        await query(
            `UPDATE tbl_referring_doctor SET ${updates.join(', ')} WHERE id = ?`,
            values
        );
        
        const [updated] = await query(
            'SELECT * FROM tbl_referring_doctor WHERE id = ?',
            [id]
        );
        
        res.json({
            success: true,
            message: 'Referring doctor updated successfully',
            data: updated
        });
    } catch (error) {
        next(error);
    }
};

// Delete referring doctor
exports.deleteReferringDoctor = async (req, res, next) => {
    try {
        const { id } = req.params;
        
        // Check if doctor exists
        const [existing] = await query(
            'SELECT id, name FROM tbl_referring_doctor WHERE id = ?',
            [id]
        );
        
        if (!existing) {
            return res.status(404).json({
                success: false,
                message: 'Referring doctor not found'
            });
        }
        
        // Check if doctor has any commissions
        const [commissions] = await query(
            'SELECT COUNT(*) as count FROM tbl_referral_commission WHERE referring_doctor_id = ?',
            [id]
        );
        
        if (commissions.count > 0) {
            return res.status(400).json({
                success: false,
                message: 'Cannot delete referring doctor with existing commission records. Consider deactivating instead.'
            });
        }
        
        await query('DELETE FROM tbl_referring_doctor WHERE id = ?', [id]);
        
        res.json({
            success: true,
            message: 'Referring doctor deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

// Get referring doctor statistics
exports.getReferringDoctorStats = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { start_date, end_date } = req.query;
        
        const [doctor] = await query(
            'SELECT * FROM tbl_referring_doctor WHERE id = ?',
            [id]
        );
        
        if (!doctor) {
            return res.status(404).json({
                success: false,
                message: 'Referring doctor not found'
            });
        }
        
        let dateSql = '';
        const params = [id];
        
        if (start_date && end_date) {
            dateSql = ' AND transaction_date BETWEEN ? AND ?';
            params.push(start_date, end_date);
        }
        
        // Get commission summary
        const [stats] = await query(
            `SELECT 
                COUNT(*) as total_transactions,
                SUM(transaction_amount) as total_revenue,
                SUM(commission_amount) as total_commission,
                SUM(CASE WHEN payment_status = 'paid' THEN commission_amount ELSE 0 END) as paid_commission,
                SUM(CASE WHEN payment_status = 'pending' THEN commission_amount ELSE 0 END) as pending_commission,
                SUM(CASE WHEN payment_status = 'approved' THEN commission_amount ELSE 0 END) as approved_commission
            FROM tbl_referral_commission 
            WHERE referring_doctor_id = ?${dateSql}`,
            params
        );
        
        res.json({
            success: true,
            data: {
                doctor,
                stats: stats || {
                    total_transactions: 0,
                    total_revenue: 0,
                    total_commission: 0,
                    paid_commission: 0,
                    pending_commission: 0,
                    approved_commission: 0
                }
            }
        });
    } catch (error) {
        next(error);
    }
};

// Get commission summary (overview of all referring doctors)
exports.getCommissionSummary = async (req, res, next) => {
    try {
        const { start_date, end_date, payment_status } = req.query;
        
        let sql = `
            SELECT 
                rd.id,
                rd.name,
                rd.specialty,
                rd.clinic_name,
                rd.commission_type,
                rd.commission_value,
                COUNT(rc.id) as total_referrals,
                COALESCE(SUM(rc.transaction_amount), 0) as total_revenue,
                COALESCE(SUM(rc.commission_amount), 0) as total_commission,
                COALESCE(SUM(CASE WHEN rc.payment_status = 'paid' THEN rc.commission_amount ELSE 0 END), 0) as paid_commission,
                COALESCE(SUM(CASE WHEN rc.payment_status = 'pending' THEN rc.commission_amount ELSE 0 END), 0) as pending_commission
            FROM tbl_referring_doctor rd
            LEFT JOIN tbl_referral_commission rc ON rd.id = rc.referring_doctor_id
            WHERE rd.active = 1
        `;
        
        const params = [];
        
        if (start_date && end_date) {
            sql += ' AND (rc.transaction_date IS NULL OR rc.transaction_date BETWEEN ? AND ?)';
            params.push(start_date, end_date);
        }
        
        if (payment_status) {
            sql += ' AND (rc.payment_status IS NULL OR rc.payment_status = ?)';
            params.push(payment_status);
        }
        
        sql += ' GROUP BY rd.id ORDER BY total_commission DESC';
        
        const summary = await query(sql, params);
        
        res.json({
            success: true,
            data: summary
        });
    } catch (error) {
        next(error);
    }
};

