/**
 * Settings Controller
 * Handles shop/hospital information and configuration
 */

const { query } = require('../config/database');
const path = require('path');
const fs = require('fs');

/**
 * @desc    Get shop/hospital information
 * @route   GET /api/settings/shop
 * @access  Private
 */
exports.getShopInfo = async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_shopinfo WHERE id = 1';
        const result = await query(sql);
        
        res.json({
            success: true,
            data: result[0] || {}
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update shop/hospital information
 * @route   PUT /api/settings/shop
 * @access  Private
 */
exports.updateShopInfo = async (req, res, next) => {
    try {
        const {
            name,
            type,
            ph_no,
            email,
            address,
            remark,
            thanks,
            printer,
            printer_voucher_count,
            user_code,
            product_code,
            software_code,
            currency,
            currency_symbol,
            currency_position
        } = req.body;

        // Check if shop info exists
        const checkSql = 'SELECT * FROM tbl_shopinfo WHERE id = 1';
        const existing = await query(checkSql);

        let logo_path = existing.length > 0 ? existing[0].photo_path : null;

        // Handle logo upload
        if (req.file) {
            // Delete old logo if exists
            if (existing.length > 0 && existing[0].photo_path) {
                const oldLogoPath = path.join(__dirname, '..', existing[0].photo_path);
                if (fs.existsSync(oldLogoPath)) {
                    fs.unlinkSync(oldLogoPath);
                }
            }
            logo_path = `/uploads/logo/${req.file.filename}`;
        }

        if (existing.length > 0) {
            // Update existing record
            const updateSql = `UPDATE tbl_shopinfo 
                              SET name = ?, type = ?, ph_no = ?, email = ?, address = ?, 
                                  remark = ?, thanks = ?, photo_path = ?, printer = ?, 
                                  printer_voucher_count = ?, user_code = ?, product_code = ?, 
                                  software_code = ?, currency = ?, currency_symbol = ?, currency_position = ?
                              WHERE id = 1`;
            
            await query(updateSql, [
                name,
                type,
                ph_no,
                email || null,
                address,
                remark || null,
                thanks || null,
                logo_path,
                printer || '80mm',
                printer_voucher_count || 1,
                user_code || null,
                product_code || null,
                software_code || null,
                currency || '$',
                currency_symbol || '$',
                currency_position || 'before'
            ]);
        } else {
            // Insert new record
            const insertSql = `INSERT INTO tbl_shopinfo 
                              (id, name, type, ph_no, email, address, remark, thanks, photo_path, 
                               printer, printer_voucher_count, user_code, product_code, software_code,
                               currency, currency_symbol, currency_position)
                              VALUES (1, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;
            
            await query(insertSql, [
                name,
                type,
                ph_no,
                email || null,
                address,
                remark || null,
                thanks || null,
                logo_path,
                printer || '80mm',
                printer_voucher_count || 1,
                user_code || null,
                product_code || null,
                software_code || null,
                currency || '$',
                currency_symbol || '$',
                currency_position || 'before'
            ]);
        }

        res.json({
            success: true,
            message: 'Settings updated successfully'
        });

    } catch (error) {
        // Delete uploaded file if database update fails
        if (req.file) {
            const filePath = path.join(__dirname, '../uploads/logo', req.file.filename);
            if (fs.existsSync(filePath)) {
                fs.unlinkSync(filePath);
            }
        }
        next(error);
    }
};

/**
 * @desc    Get system configuration
 * @route   GET /api/settings/config
 * @access  Private
 */
exports.getConfig = async (req, res, next) => {
    try {
        const config = {
            version: '3.0.0',
            database_tables: 46,
            features: [
                'POS',
                'EMR',
                'Patient Management',
                'Inventory',
                'Reports'
            ],
            languages: ['en', 'my']
        };

        res.json({
            success: true,
            data: config
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Backup database
 * @route   POST /api/settings/backup
 * @access  Private (Admin only)
 */
exports.backupDatabase = async (req, res, next) => {
    try {
        // This would typically trigger a database backup
        // Implementation depends on your backup strategy
        
        res.json({
            success: true,
            message: 'Backup initiated successfully',
            backup_file: `backup_${Date.now()}.sql`
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get categories
 * @route   GET /api/settings/categories
 * @access  Private
 */
exports.getCategories = async (req, res, next) => {
    try {
        const sql = 'SELECT * FROM tbl_catalog ORDER BY name';
        const categories = await query(sql);

        res.json({
            success: true,
            count: categories.length,
            data: categories
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Add category
 * @route   POST /api/settings/categories
 * @access  Private
 */
exports.addCategory = async (req, res, next) => {
    try {
        const { name } = req.body;

        const sql = 'INSERT INTO tbl_catalog (name) VALUES (?)';
        const result = await query(sql, [name]);

        res.status(201).json({
            success: true,
            message: 'Category added successfully',
            data: { id: result.insertId, name }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update category
 * @route   PUT /api/settings/categories/:id
 * @access  Private
 */
exports.updateCategory = async (req, res, next) => {
    try {
        const { name } = req.body;
        const { id } = req.params;

        const sql = 'UPDATE tbl_catalog SET name = ? WHERE id = ?';
        await query(sql, [name, id]);

        res.json({
            success: true,
            message: 'Category updated successfully',
            data: { id: parseInt(id), name }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Delete category
 * @route   DELETE /api/settings/categories/:id
 * @access  Private
 */
exports.deleteCategory = async (req, res, next) => {
    try {
        const sql = 'DELETE FROM tbl_catalog WHERE id = ?';
        await query(sql, [req.params.id]);

        res.json({
            success: true,
            message: 'Category deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Check and fix logo path
 * @route   GET /api/settings/logo-check
 * @access  Private
 */
exports.checkLogoPath = async (req, res, next) => {
    try {
        const logoFix = require('../utils/logo-fix');
        const result = await logoFix.checkAndFixLogoPath();
        
        res.json({
            success: result.success,
            message: result.message,
            data: result
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    List available logo files
 * @route   GET /api/settings/logo-list
 * @access  Private
 */
exports.listLogos = async (req, res, next) => {
    try {
        const logoFix = require('../utils/logo-fix');
        const logos = logoFix.listAvailableLogos();
        
        res.json({
            success: true,
            data: logos
        });
    } catch (error) {
        next(error);
    }
};

