/**
 * Stock Expense Controller
 * Handles stock expense operations with warehouse stock updates
 */

const { query } = require('../config/database');
const models = require('../models');

/**
 * Create stock expense and update warehouse stock
 */
exports.create = async (req, res, next) => {
    try {
        const { warehouse_id, barcode, qty } = req.body;

        console.log('Creating stock expense with data:', req.body);

        // Remove warehouse_id if it's null or empty to avoid column errors
        const expenseData = { ...req.body };
        if (!expenseData.warehouse_id || expenseData.warehouse_id === '') {
            delete expenseData.warehouse_id;
        }

        // Create the stock expense record
        let expense;
        try {
            expense = await models.StockExpense.create(expenseData);
            console.log('Stock expense created successfully:', expense);
        } catch (createError) {
            console.error('Error creating stock expense:', createError);
            console.error('Error stack:', createError.stack);
            console.error('Error code:', createError.code);
            
            // Check if it's a column not found error
            if (createError.code === 'ER_BAD_FIELD_ERROR' && createError.message.includes('warehouse_id')) {
                return res.status(400).json({
                    success: false,
                    message: 'Database migration required: warehouse_id column not found in tbl_stock_expense. Please run the migration: database/migrations/add_warehouse_id_to_stock_expense.sql'
                });
            }
            
            return res.status(400).json({
                success: false,
                message: 'Failed to create stock expense: ' + (createError.message || 'Unknown error'),
                error: process.env.NODE_ENV === 'development' ? createError.message : undefined,
                details: process.env.NODE_ENV === 'development' ? createError.stack : undefined
            });
        }

        // Update warehouse stock if warehouse_id is provided
        if (warehouse_id && barcode && qty) {
            try {
                // Find stock by barcode
                const stock = await query('SELECT id FROM tbl_stock WHERE barcode = ?', [barcode]);
                
                if (stock.length > 0) {
                    const stockId = stock[0].id;
                    const quantity = parseInt(qty);

                    // Check if warehouse stock exists
                    const warehouseStock = await query(
                        'SELECT id, quantity FROM tbl_warehouse_stock WHERE warehouse_id = ? AND stock_id = ?',
                        [warehouse_id, stockId]
                    );

                    if (warehouseStock.length > 0) {
                        // Update existing warehouse stock
                        const currentQty = parseInt(warehouseStock[0].quantity || 0);
                        const newQty = Math.max(0, currentQty - quantity);
                        
                        await query(
                            'UPDATE tbl_warehouse_stock SET quantity = ? WHERE id = ?',
                            [newQty, warehouseStock[0].id]
                        );
                    } else {
                        console.warn(`Warehouse stock not found for warehouse ${warehouse_id} and stock ${stockId}`);
                    }

                    // Also update main stock for backward compatibility
                    await query(
                        'UPDATE tbl_stock SET qty = GREATEST(0, qty - ?) WHERE id = ?',
                        [quantity, stockId]
                    );
                }
            } catch (stockError) {
                console.error('Error updating warehouse stock:', stockError);
                // Don't fail the expense creation if stock update fails
            }
        }

        res.status(201).json({
            success: true,
            message: 'Stock expense created successfully',
            data: expense
        });
    } catch (error) {
        console.error('Stock expense create error:', error);
        console.error('Error stack:', error.stack);
        console.error('Request body:', req.body);
        next(error);
    }
};

/**
 * Update stock expense and adjust warehouse stock
 */
exports.update = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { warehouse_id, barcode, qty } = req.body;

        // Get old expense to restore stock
        const oldExpense = await models.StockExpense.findById(id);
        if (!oldExpense) {
            return res.status(404).json({
                success: false,
                message: 'Stock expense not found'
            });
        }

        // Update the stock expense record
        const expense = await models.StockExpense.update(id, req.body);

        // Update warehouse stock if warehouse_id and barcode are provided
        if (warehouse_id && barcode && qty) {
            try {
                // Find stock by barcode
                const stock = await query('SELECT id FROM tbl_stock WHERE barcode = ?', [barcode]);
                
                if (stock.length > 0) {
                    const stockId = stock[0].id;
                    const oldQty = parseInt(oldExpense.qty || 0);
                    const newQty = parseInt(qty);
                    const qtyDiff = newQty - oldQty;

                    // Check if warehouse stock exists
                    const warehouseStock = await query(
                        'SELECT id, quantity FROM tbl_warehouse_stock WHERE warehouse_id = ? AND stock_id = ?',
                        [warehouse_id, stockId]
                    );

                    if (warehouseStock.length > 0) {
                        // Update warehouse stock (restore old, subtract new)
                        const currentQty = parseInt(warehouseStock[0].quantity || 0);
                        const adjustedQty = currentQty + oldQty - newQty;
                        const finalQty = Math.max(0, adjustedQty);
                        
                        await query(
                            'UPDATE tbl_warehouse_stock SET quantity = ? WHERE id = ?',
                            [finalQty, warehouseStock[0].id]
                        );
                    }

                    // Also update main stock for backward compatibility
                    await query(
                        'UPDATE tbl_stock SET qty = GREATEST(0, qty + ? - ?) WHERE id = ?',
                        [oldQty, newQty, stockId]
                    );
                }
            } catch (stockError) {
                console.error('Error updating warehouse stock:', stockError);
                // Don't fail the expense update if stock update fails
            }
        }

        res.json({
            success: true,
            message: 'Stock expense updated successfully',
            data: expense
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Delete stock expense and restore warehouse stock
 */
exports.delete = async (req, res, next) => {
    try {
        const { id } = req.params;

        // Get expense to restore stock
        const expense = await models.StockExpense.findById(id);
        if (!expense) {
            return res.status(404).json({
                success: false,
                message: 'Stock expense not found'
            });
        }

        // Delete the stock expense record
        const deleted = await models.StockExpense.delete(id);

        // Restore warehouse stock if warehouse_id is provided
        if (expense.warehouse_id && expense.barcode && expense.qty) {
            try {
                // Find stock by barcode
                const stock = await query('SELECT id FROM tbl_stock WHERE barcode = ?', [expense.barcode]);
                
                if (stock.length > 0) {
                    const stockId = stock[0].id;
                    const quantity = parseInt(expense.qty);

                    // Check if warehouse stock exists
                    const warehouseStock = await query(
                        'SELECT id, quantity FROM tbl_warehouse_stock WHERE warehouse_id = ? AND stock_id = ?',
                        [expense.warehouse_id, stockId]
                    );

                    if (warehouseStock.length > 0) {
                        // Restore warehouse stock
                        const currentQty = parseInt(warehouseStock[0].quantity || 0);
                        const newQty = currentQty + quantity;
                        
                        await query(
                            'UPDATE tbl_warehouse_stock SET quantity = ? WHERE id = ?',
                            [newQty, warehouseStock[0].id]
                        );
                    }

                    // Also restore main stock for backward compatibility
                    await query(
                        'UPDATE tbl_stock SET qty = qty + ? WHERE id = ?',
                        [quantity, stockId]
                    );
                }
            } catch (stockError) {
                console.error('Error restoring warehouse stock:', stockError);
                // Don't fail the deletion if stock update fails
            }
        }

        res.json({
            success: true,
            message: 'Stock expense deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Get all stock expenses
 */
exports.getAll = async (req, res, next) => {
    try {
        const result = await models.StockExpense.findAll(req.query);
        res.json({
            success: true,
            data: result.data,
            total: result.pagination?.total || result.data?.length || 0,
            count: result.data?.length || 0,
            page: result.pagination?.page || 1,
            pages: result.pagination?.pages || 1
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Get single stock expense
 */
exports.getOne = async (req, res, next) => {
    try {
        const data = await models.StockExpense.findById(req.params.id);
        if (!data) {
            return res.status(404).json({
                success: false,
                message: 'Record not found'
            });
        }
        res.json({ success: true, data });
    } catch (error) {
        next(error);
    }
};
