const { query } = require('../config/database');

/**
 * @desc    Get all suppliers
 * @route   GET /api/suppliers
 * @access  Private
 */
exports.getSuppliers = async (req, res, next) => {
    try {
        const sql = `
            SELECT id, name, ph_no as phone, email, shop, address, credit_balance, remark, date
            FROM tbl_supplier 
            ORDER BY name ASC
        `;
        const suppliers = await query(sql);

        res.status(200).json({
            success: true,
            count: suppliers.length,
            data: suppliers
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get single supplier
 * @route   GET /api/suppliers/:id
 * @access  Private
 */
exports.getSupplier = async (req, res, next) => {
    try {
        const sql = 'SELECT id, name, ph_no as phone, email, shop, address, credit_balance, remark, date FROM tbl_supplier WHERE id = ?';
        const suppliers = await query(sql, [req.params.id]);

        if (suppliers.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Supplier not found'
            });
        }

        res.status(200).json({
            success: true,
            data: suppliers[0]
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create new supplier
 * @route   POST /api/suppliers
 * @access  Private
 */
exports.createSupplier = async (req, res, next) => {
    try {
        const { name, phone, shop, address, credit_balance, email, remark } = req.body;

        // Validate required fields
        if (!name) {
            return res.status(400).json({
                success: false,
                message: 'Supplier name is required'
            });
        }

        const sql = `
            INSERT INTO tbl_supplier (name, ph_no, email, shop, address, credit_balance, remark, date)
            VALUES (?, ?, ?, ?, ?, ?, ?, CURDATE())
        `;

        const result = await query(sql, [
            name,
            phone || null,  // Maps to ph_no column
            email || null,
            shop || null,
            address || null,
            credit_balance || 0,
            remark || null
        ]);

        res.status(201).json({
            success: true,
            message: 'Supplier created successfully',
            data: {
                id: result.insertId,
                name: name
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update supplier
 * @route   PUT /api/suppliers/:id
 * @access  Private
 */
exports.updateSupplier = async (req, res, next) => {
    try {
        const { name, phone, shop, address, credit_balance, email, remark } = req.body;
        const supplierId = req.params.id;

        // Check if supplier exists
        const existingSupplier = await query('SELECT id FROM tbl_supplier WHERE id = ?', [supplierId]);
        if (existingSupplier.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Supplier not found'
            });
        }

        const sql = `
            UPDATE tbl_supplier 
            SET name = ?, ph_no = ?, email = ?, shop = ?, address = ?, credit_balance = ?, remark = ?
            WHERE id = ?
        `;

        await query(sql, [
            name,
            phone || null,  // Maps to ph_no column
            email || null,
            shop || null,
            address || null,
            credit_balance || 0,
            remark || null,
            supplierId
        ]);

        res.status(200).json({
            success: true,
            message: 'Supplier updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get stock/products for a supplier
 * @route   GET /api/suppliers/:id/stock
 * @access  Private
 */
exports.getSupplierStock = async (req, res, next) => {
    try {
        const supplierId = req.params.id;
        const { date_from, date_to } = req.query;

        // Check if supplier exists
        const existingSupplier = await query('SELECT id, name FROM tbl_supplier WHERE id = ?', [supplierId]);
        if (existingSupplier.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Supplier not found'
            });
        }

        let sql = `
            SELECT 
                s.id,
                s.barcode,
                s.name,
                s.qty,
                s.qty_alert,
                s.unit_name,
                s.cost_price,
                s.sale_price_latli,
                s.image_path,
                c.name as category_name,
                MAX(p.date) as last_purchase_date
            FROM tbl_stock s
            LEFT JOIN tbl_catalog c ON s.cat_id = c.id
            LEFT JOIN tbl_purchase p ON s.barcode = p.barcode
            LEFT JOIN tbl_purchase_voucher pv ON p.purchase_voucher_id = pv.id
            WHERE s.supplier_id = ?
              AND (pv.supplier_name = ? OR pv.supplier_name IS NULL)
        `;

        const params = [supplierId, existingSupplier[0].name];

        // Add date filter if provided
        if (date_from && date_to) {
            sql += ` AND p.date BETWEEN ? AND ?`;
            params.push(date_from, date_to);
        } else if (date_from) {
            sql += ` AND p.date >= ?`;
            params.push(date_from);
        } else if (date_to) {
            sql += ` AND p.date <= ?`;
            params.push(date_to);
        }

        sql += ` GROUP BY s.id, s.barcode, s.name, s.qty, s.qty_alert, s.unit_name, s.cost_price, s.sale_price_latli, s.image_path, c.name
                 ORDER BY s.name ASC`;

        const stock = await query(sql, params);

        // Calculate totals
        const totalQuantity = stock.reduce((sum, item) => sum + (parseInt(item.qty) || 0), 0);
        const totalAmount = stock.reduce((sum, item) => {
            const qty = parseInt(item.qty) || 0;
            const costPrice = parseFloat(item.cost_price) || 0;
            return sum + (qty * costPrice);
        }, 0);

        res.status(200).json({
            success: true,
            count: stock.length,
            data: stock,
            supplier: existingSupplier[0],
            totals: {
                total_quantity: totalQuantity,
                total_amount: totalAmount
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Delete supplier
 * @route   DELETE /api/suppliers/:id
 * @access  Private
 */
exports.deleteSupplier = async (req, res, next) => {
    try {
        const supplierId = req.params.id;

        // Check if supplier exists
        const existingSupplier = await query('SELECT id FROM tbl_supplier WHERE id = ?', [supplierId]);
        if (existingSupplier.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Supplier not found'
            });
        }

        const sql = 'DELETE FROM tbl_supplier WHERE id = ?';
        await query(sql, [supplierId]);

        res.status(200).json({
            success: true,
            message: 'Supplier deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};
