const { query } = require('../config/database');
const bcrypt = require('bcryptjs');

/**
 * Get all users
 */
exports.getUsers = async (req, res, next) => {
    try {
        const sql = `
            SELECT u.id, u.name, u.ph_no, u.address, u.date,
                   r.name as role_name, u.role_id
            FROM tbl_user u
            LEFT JOIN tbl_role r ON u.role_id = r.id
            ORDER BY u.date DESC
        `;
        const users = await query(sql);

        // Don't send passwords
        const safeUsers = users.map(user => {
            const { passwd, ...safeUser } = user;
            return safeUser;
        });

        res.status(200).json({
            success: true,
            count: safeUsers.length,
            data: safeUsers
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Get single user
 */
exports.getUser = async (req, res, next) => {
    try {
        const sql = `
            SELECT u.id, u.name, u.ph_no, u.address, u.date,
                   r.name as role_name, u.role_id
            FROM tbl_user u
            LEFT JOIN tbl_role r ON u.role_id = r.id
            WHERE u.id = ?
        `;
        const users = await query(sql, [req.params.id]);

        if (users.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        // Don't send password
        const { passwd, ...safeUser } = users[0];

        res.status(200).json({
            success: true,
            data: safeUser
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Create new user
 */
exports.createUser = async (req, res, next) => {
    try {
        const { name, password, role_id, ph_no, address } = req.body;

        // Validate required fields
        if (!name || !password || !role_id) {
            return res.status(400).json({
                success: false,
                message: 'Name, password, and role are required'
            });
        }

        // Check if username already exists
        const existingUser = await query('SELECT id FROM tbl_user WHERE name = ?', [name]);
        if (existingUser.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Username already exists'
            });
        }

        // Hash password
        const hashedPassword = await bcrypt.hash(password, 10);

        const sql = `
            INSERT INTO tbl_user (name, passwd, role_id, ph_no, address, date)
            VALUES (?, ?, ?, ?, ?, CURDATE())
        `;
        
        const result = await query(sql, [
            name,
            hashedPassword,
            role_id,
            ph_no || null,
            address || null
        ]);

        res.status(201).json({
            success: true,
            message: 'User created successfully',
            data: {
                id: result.insertId,
                name: name
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Update user
 */
exports.updateUser = async (req, res, next) => {
    try {
        const { name, password, role_id, ph_no, address } = req.body;
        const userId = req.params.id;

        // Check if user exists
        const existingUser = await query('SELECT id FROM tbl_user WHERE id = ?', [userId]);
        if (existingUser.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        // If password is provided, hash it
        let updateSql;
        let updateParams;

        if (password) {
            const hashedPassword = await bcrypt.hash(password, 10);
            updateSql = `
                UPDATE tbl_user 
                SET name = ?, passwd = ?, role_id = ?, ph_no = ?, address = ?
                WHERE id = ?
            `;
            updateParams = [name, hashedPassword, role_id, ph_no, address, userId];
        } else {
            updateSql = `
                UPDATE tbl_user 
                SET name = ?, role_id = ?, ph_no = ?, address = ?
                WHERE id = ?
            `;
            updateParams = [name, role_id, ph_no, address, userId];
        }

        await query(updateSql, updateParams);

        res.status(200).json({
            success: true,
            message: 'User updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Delete user
 */
exports.deleteUser = async (req, res, next) => {
    try {
        const userId = req.params.id;

        // Prevent deleting admin user (ID 1)
        if (userId == 1) {
            return res.status(403).json({
                success: false,
                message: 'Cannot delete admin user'
            });
        }

        // Check if user exists
        const existingUser = await query('SELECT id FROM tbl_user WHERE id = ?', [userId]);
        if (existingUser.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        const sql = 'DELETE FROM tbl_user WHERE id = ?';
        await query(sql, [userId]);

        res.status(200).json({
            success: true,
            message: 'User deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Change user password
 */
exports.changePassword = async (req, res, next) => {
    try {
        const { currentPassword, newPassword } = req.body;
        const userId = req.user.id; // From auth middleware

        if (!currentPassword || !newPassword) {
            return res.status(400).json({
                success: false,
                message: 'Current password and new password are required'
            });
        }

        // Get current user
        const users = await query('SELECT passwd FROM tbl_user WHERE id = ?', [userId]);
        if (users.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'User not found'
            });
        }

        // Verify current password
        const isMatch = await bcrypt.compare(currentPassword, users[0].passwd);
        if (!isMatch) {
            return res.status(401).json({
                success: false,
                message: 'Current password is incorrect'
            });
        }

        // Hash new password
        const hashedPassword = await bcrypt.hash(newPassword, 10);

        // Update password
        await query('UPDATE tbl_user SET passwd = ? WHERE id = ?', [hashedPassword, userId]);

        res.status(200).json({
            success: true,
            message: 'Password changed successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Get all roles
 */
exports.getRoles = async (req, res, next) => {
    try {
        const roles = await query('SELECT * FROM tbl_role ORDER BY id');

        res.status(200).json({
            success: true,
            count: roles.length,
            data: roles
        });
    } catch (error) {
        next(error);
    }
};

/**
 * Get user statistics
 */
exports.getUserStats = async (req, res, next) => {
    try {
        // Total users
        const totalUsers = await query('SELECT COUNT(*) as count FROM tbl_user');
        
        // Users by role
        const usersByRole = await query(`
            SELECT r.name as role_name, COUNT(u.id) as count
            FROM tbl_role r
            LEFT JOIN tbl_user u ON r.id = u.role_id
            GROUP BY r.id, r.name
            ORDER BY count DESC
        `);

        // Recent users (last 7 days)
        const recentUsers = await query(`
            SELECT COUNT(*) as count 
            FROM tbl_user 
            WHERE date >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
        `);

        res.status(200).json({
            success: true,
            data: {
                total_users: totalUsers[0].count,
                users_by_role: usersByRole,
                recent_users: recentUsers[0].count
            }
        });
    } catch (error) {
        next(error);
    }
};

