const { query } = require('../config/database');

/**
 * @desc    Get stock movement history
 * @route   GET /api/warehouses/movements
 * @access  Private
 */
exports.getMovements = async (req, res, next) => {
    try {
        const {
            warehouse_id,
            stock_id,
            movement_type,
            reference_type,
            date_from,
            date_to,
            assigned_person,
            page = 1,
            limit = 50
        } = req.query;

        const offset = (parseInt(page) - 1) * parseInt(limit);

        let sql = `SELECT m.*,
                   w.name as warehouse_name,
                   w.warehouse_code,
                   s.name as product_name,
                   s.barcode,
                   u.name as created_by_name
                   FROM tbl_stock_movement m
                   INNER JOIN tbl_warehouse w ON m.warehouse_id = w.id
                   INNER JOIN tbl_stock s ON m.stock_id = s.id
                   LEFT JOIN tbl_user u ON m.created_by = u.id
                   WHERE 1=1`;
        const params = [];

        if (warehouse_id) {
            sql += ' AND m.warehouse_id = ?';
            params.push(warehouse_id);
        }

        if (stock_id) {
            sql += ' AND m.stock_id = ?';
            params.push(stock_id);
        }

        if (movement_type) {
            sql += ' AND m.movement_type = ?';
            params.push(movement_type);
        }

        if (reference_type) {
            sql += ' AND m.reference_type = ?';
            params.push(reference_type);
        }

        if (date_from) {
            sql += ' AND DATE(m.movement_date) >= ?';
            params.push(date_from);
        }

        if (date_to) {
            sql += ' AND DATE(m.movement_date) <= ?';
            params.push(date_to);
        }

        if (assigned_person) {
            sql += ' AND (m.assigned_person LIKE ? OR m.assigned_person_id = ?)';
            params.push(`%${assigned_person}%`, assigned_person);
        }

        // Get total count - build count query separately using the same WHERE conditions
        // Extract WHERE conditions from the main query
        const whereConditions = sql.split('WHERE 1=1')[1] || '';
        const countSql = `SELECT COUNT(*) as total
                   FROM tbl_stock_movement m
                   INNER JOIN tbl_warehouse w ON m.warehouse_id = w.id
                   INNER JOIN tbl_stock s ON m.stock_id = s.id
                   WHERE 1=1${whereConditions}`;
        
        const countResult = await query(countSql, params);
        const total = countResult && countResult[0] ? (countResult[0].total || 0) : 0;

        // Add ordering and pagination
        sql += ' ORDER BY m.movement_date DESC, m.created_at DESC LIMIT ? OFFSET ?';
        params.push(parseInt(limit), offset);

        const movements = await query(sql, params);

        res.status(200).json({
            success: true,
            count: movements.length,
            total,
            page: parseInt(page),
            pages: Math.ceil(total / parseInt(limit)),
            data: movements
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get stock movement summary by warehouse
 * @route   GET /api/warehouses/movements/summary
 * @access  Private
 */
exports.getMovementSummary = async (req, res, next) => {
    try {
        const { warehouse_id, date_from, date_to } = req.query;

        let sql = `SELECT 
                   w.id as warehouse_id,
                   w.name as warehouse_name,
                   w.warehouse_code,
                   m.movement_type,
                   COUNT(*) as movement_count,
                   SUM(m.quantity) as total_quantity
                   FROM tbl_stock_movement m
                   INNER JOIN tbl_warehouse w ON m.warehouse_id = w.id
                   WHERE 1=1`;
        const params = [];

        if (warehouse_id) {
            sql += ' AND m.warehouse_id = ?';
            params.push(warehouse_id);
        }

        if (date_from) {
            sql += ' AND DATE(m.movement_date) >= ?';
            params.push(date_from);
        }

        if (date_to) {
            sql += ' AND DATE(m.movement_date) <= ?';
            params.push(date_to);
        }

        sql += ' GROUP BY w.id, w.name, w.warehouse_code, m.movement_type ORDER BY w.name, m.movement_type';

        const summary = await query(sql, params);

        res.status(200).json({
            success: true,
            data: summary
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get movements by assigned person
 * @route   GET /api/warehouses/movements/by-person
 * @access  Private
 */
exports.getMovementsByPerson = async (req, res, next) => {
    try {
        const { assigned_person, date_from, date_to } = req.query;

        let sql = `SELECT 
                   m.assigned_person,
                   m.assigned_person_id,
                   u.name as username,
                   COUNT(*) as movement_count,
                   SUM(CASE WHEN m.movement_type IN ('IN', 'TRANSFER_IN') THEN m.quantity ELSE 0 END) as total_in,
                   SUM(CASE WHEN m.movement_type IN ('OUT', 'TRANSFER_OUT') THEN m.quantity ELSE 0 END) as total_out,
                   w.name as warehouse_name
                   FROM tbl_stock_movement m
                   INNER JOIN tbl_warehouse w ON m.warehouse_id = w.id
                   LEFT JOIN tbl_user u ON m.assigned_person_id = u.id
                   WHERE 1=1`;
        const params = [];

        if (assigned_person) {
            sql += ' AND (m.assigned_person LIKE ? OR m.assigned_person_id = ?)';
            params.push(`%${assigned_person}%`, assigned_person);
        }

        if (date_from) {
            sql += ' AND DATE(m.movement_date) >= ?';
            params.push(date_from);
        }

        if (date_to) {
            sql += ' AND DATE(m.movement_date) <= ?';
            params.push(date_to);
        }

        sql += ' GROUP BY m.assigned_person, m.assigned_person_id, u.name, w.name ORDER BY movement_count DESC';

        const movements = await query(sql, params);

        res.status(200).json({
            success: true,
            count: movements.length,
            data: movements
        });
    } catch (error) {
        next(error);
    }
};

