const { query } = require('../config/database');

/**
 * @desc    Get all stock transfers
 * @route   GET /api/warehouses/transfers
 * @access  Private
 */
exports.getTransfers = async (req, res, next) => {
    try {
        const { status, from_warehouse, to_warehouse, warehouse, date_from, date_to } = req.query;

        let sql = `SELECT t.*,
                   w1.name as from_warehouse_name,
                   w1.warehouse_code as from_warehouse_code,
                   w2.name as to_warehouse_name,
                   w2.warehouse_code as to_warehouse_code,
                   u1.name as created_by_name,
                   u2.name as approved_by_name,
                   u3.name as received_by_name,
                   (SELECT COUNT(*) FROM tbl_stock_transfer_item WHERE transfer_id = t.id) as items_count
                   FROM tbl_stock_transfer t
                   LEFT JOIN tbl_warehouse w1 ON t.from_warehouse_id = w1.id
                   LEFT JOIN tbl_warehouse w2 ON t.to_warehouse_id = w2.id
                   LEFT JOIN tbl_user u1 ON t.created_by = u1.id
                   LEFT JOIN tbl_user u2 ON t.approved_by = u2.id
                   LEFT JOIN tbl_user u3 ON t.received_by = u3.id
                   WHERE 1=1`;
        const params = [];

        if (status) {
            sql += ' AND t.status = ?';
            params.push(status);
        }

        // Filter by warehouse (either sender or receiver)
        if (warehouse) {
            sql += ' AND (t.from_warehouse_id = ? OR t.to_warehouse_id = ?)';
            params.push(warehouse, warehouse);
        } else {
            // If warehouse is not specified, use from_warehouse and to_warehouse filters
            if (from_warehouse) {
                sql += ' AND t.from_warehouse_id = ?';
                params.push(from_warehouse);
            }

            if (to_warehouse) {
                sql += ' AND t.to_warehouse_id = ?';
                params.push(to_warehouse);
            }
        }

        if (date_from) {
            sql += ' AND t.transfer_date >= ?';
            params.push(date_from);
        }

        if (date_to) {
            sql += ' AND t.transfer_date <= ?';
            params.push(date_to);
        }

        sql += ' ORDER BY t.transfer_date DESC, t.created_at DESC';

        const transfers = await query(sql, params);

        res.status(200).json({
            success: true,
            count: transfers.length,
            data: transfers
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get single transfer with items
 * @route   GET /api/warehouses/transfers/:id
 * @access  Private
 */
exports.getTransfer = async (req, res, next) => {
    try {
        const transferSql = `SELECT t.*,
                            w1.name as from_warehouse_name,
                            w1.warehouse_code as from_warehouse_code,
                            w2.name as to_warehouse_name,
                            w2.warehouse_code as to_warehouse_code,
                            u1.name as created_by_name,
                            u2.name as approved_by_name,
                            u3.name as received_by_name
                            FROM tbl_stock_transfer t
                            LEFT JOIN tbl_warehouse w1 ON t.from_warehouse_id = w1.id
                            LEFT JOIN tbl_warehouse w2 ON t.to_warehouse_id = w2.id
                            LEFT JOIN tbl_user u1 ON t.created_by = u1.id
                            LEFT JOIN tbl_user u2 ON t.approved_by = u2.id
                            LEFT JOIN tbl_user u3 ON t.received_by = u3.id
                            WHERE t.id = ?`;

        const transfers = await query(transferSql, [req.params.id]);

        if (transfers.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Transfer not found'
            });
        }

        // Get transfer items
        const itemsSql = `SELECT ti.*,
                         s.name as product_name,
                         s.barcode,
                         s.unit_name
                         FROM tbl_stock_transfer_item ti
                         INNER JOIN tbl_stock s ON ti.stock_id = s.id
                         WHERE ti.transfer_id = ?`;

        const items = await query(itemsSql, [req.params.id]);

        res.status(200).json({
            success: true,
            data: {
                ...transfers[0],
                items
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create stock transfer
 * @route   POST /api/warehouses/transfers
 * @access  Private
 */
exports.createTransfer = async (req, res, next) => {
    try {
        const { from_warehouse_id, to_warehouse_id, transfer_date, items, notes } = req.body;

        if (!from_warehouse_id || !to_warehouse_id || !items || items.length === 0) {
            return res.status(400).json({
                success: false,
                message: 'From warehouse, to warehouse, and items are required'
            });
        }

        if (from_warehouse_id === to_warehouse_id) {
            return res.status(400).json({
                success: false,
                message: 'From and to warehouse cannot be the same'
            });
        }

        // Generate transfer number
        const countResult = await query('SELECT COUNT(*) as count FROM tbl_stock_transfer');
        const transferNumber = `TRF-${String(countResult[0].count + 1).padStart(6, '0')}`;

        // Validate stock availability
        for (const item of items) {
            const stockCheck = await query(
                `SELECT quantity, reserved_quantity, (quantity - reserved_quantity) as available 
                 FROM tbl_warehouse_stock 
                 WHERE warehouse_id = ? AND stock_id = ?`,
                [from_warehouse_id, item.stock_id]
            );

            const available = stockCheck.length > 0 ? (stockCheck[0].available || 0) : 0;
            if (stockCheck.length === 0 || available < item.quantity) {
                const stockName = await query('SELECT name FROM tbl_stock WHERE id = ?', [item.stock_id]);
                return res.status(400).json({
                    success: false,
                    message: `Insufficient stock for ${stockName[0]?.name || 'item'}. Available: ${available}`
                });
            }
        }

        // Create transfer
        const transferSql = `INSERT INTO tbl_stock_transfer
                            (transfer_number, from_warehouse_id, to_warehouse_id, transfer_date, status, notes, created_by)
                            VALUES (?, ?, ?, ?, 'Pending', ?, ?)`;

        const transferResult = await query(transferSql, [
            transferNumber,
            from_warehouse_id,
            to_warehouse_id,
            transfer_date || new Date().toISOString().split('T')[0],
            notes || null,
            req.user?.id || null
        ]);

        const transferId = transferResult.insertId;
        const assignedPerson = req.body.assigned_person || req.user?.name || 'System User';

        // Insert transfer items and immediately transfer stock
        for (const item of items) {
            await query(
                `INSERT INTO tbl_stock_transfer_item (transfer_id, stock_id, quantity, notes)
                 VALUES (?, ?, ?, ?)`,
                [transferId, item.stock_id, item.quantity, item.notes || null]
            );

            // Create OUT movement for source warehouse (trigger will automatically deduct stock)
            await query(
                `INSERT INTO tbl_stock_movement
                 (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id, 
                  movement_date, assigned_person, assigned_person_id, notes, created_by)
                 VALUES ('TRANSFER_OUT', ?, ?, ?, 'TRANSFER', ?, NOW(), ?, ?, ?, ?)`,
                [
                    from_warehouse_id,
                    item.stock_id,
                    item.quantity,
                    transferId,
                    assignedPerson,
                    req.user?.id || null,
                    `Transferred to warehouse ${to_warehouse_id}`,
                    req.user?.id || null
                ]
            );

            // Create IN movement for destination warehouse
            await query(
                `INSERT INTO tbl_stock_movement
                 (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id, 
                  movement_date, assigned_person, assigned_person_id, notes, created_by)
                 VALUES ('TRANSFER_IN', ?, ?, ?, 'TRANSFER', ?, NOW(), ?, ?, ?, ?)`,
                [
                    to_warehouse_id,
                    item.stock_id,
                    item.quantity,
                    transferId,
                    assignedPerson,
                    req.user?.id || null,
                    `Received from transfer ${transferNumber}`,
                    req.user?.id || null
                ]
            );
        }

        // Update transfer status to Completed immediately
        await query(
            `UPDATE tbl_stock_transfer 
             SET status = 'Completed',
                 received_by = ?,
                 completed_at = NOW()
             WHERE id = ?`,
            [req.user?.id || null, transferId]
        );

        res.status(201).json({
            success: true,
            message: 'Stock transfer created successfully',
            data: {
                id: transferId,
                transfer_number: transferNumber
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Complete transfer (receive stock)
 * @route   PUT /api/warehouses/transfers/:id/complete
 * @access  Private
 */
exports.completeTransfer = async (req, res, next) => {
    try {
        const { id } = req.params;
        const { items, received_by, notes } = req.body;

        // Get transfer
        const transfer = await query('SELECT * FROM tbl_stock_transfer WHERE id = ?', [id]);
        if (transfer.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Transfer not found'
            });
        }

        if (transfer[0].status === 'Completed') {
            return res.status(400).json({
                success: false,
                message: 'Transfer already completed'
            });
        }

        // Get transfer items
        const transferItems = await query(
            'SELECT * FROM tbl_stock_transfer_item WHERE transfer_id = ?',
            [id]
        );

        // Process received items
        for (const item of transferItems) {
            const receivedQty = items?.find(i => i.item_id === item.id)?.received_quantity || item.quantity;

            // Update received quantity
            await query(
                'UPDATE tbl_stock_transfer_item SET received_quantity = ? WHERE id = ?',
                [receivedQty, item.id]
            );

            // Release reserved stock (reduce reserved_quantity)
            await query(
                `UPDATE tbl_warehouse_stock 
                 SET reserved_quantity = reserved_quantity - ?
                 WHERE warehouse_id = ? AND stock_id = ?`,
                [item.quantity, transfer[0].from_warehouse_id, item.stock_id]
            );

            // Create OUT movement for source warehouse (trigger will automatically reduce stock)
            await query(
                `INSERT INTO tbl_stock_movement
                 (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id, 
                  movement_date, assigned_person, assigned_person_id, notes, created_by)
                 VALUES ('TRANSFER_OUT', ?, ?, ?, 'TRANSFER', ?, NOW(), ?, ?, ?, ?)`,
                [
                    transfer[0].from_warehouse_id,
                    item.stock_id,
                    receivedQty,
                    id,
                    received_by || req.user?.name || null,
                    req.user?.id || null,
                    `Transferred to warehouse ${transfer[0].to_warehouse_id}`,
                    req.user?.id || null
                ]
            );

            // Create IN movement for destination warehouse (trigger will automatically add stock)
            await query(
                `INSERT INTO tbl_stock_movement
                 (movement_type, warehouse_id, stock_id, quantity, reference_type, reference_id, 
                  movement_date, assigned_person, assigned_person_id, notes, created_by)
                 VALUES ('TRANSFER_IN', ?, ?, ?, 'TRANSFER', ?, NOW(), ?, ?, ?, ?)`,
                [
                    transfer[0].to_warehouse_id,
                    item.stock_id,
                    receivedQty,
                    id,
                    received_by || req.user?.name || null,
                    req.user?.id || null,
                    `Received from transfer ${transfer[0].transfer_number}`,
                    req.user?.id || null
                ]
            );
        }

        // Update transfer status
        await query(
            `UPDATE tbl_stock_transfer 
             SET status = 'Completed', 
                 received_by = ?,
                 completed_at = NOW()
             WHERE id = ?`,
            [req.user?.id || null, id]
        );

        res.status(200).json({
            success: true,
            message: 'Transfer completed successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Cancel transfer
 * @route   PUT /api/warehouses/transfers/:id/cancel
 * @access  Private
 */
exports.cancelTransfer = async (req, res, next) => {
    try {
        const { id } = req.params;

        const transfer = await query('SELECT * FROM tbl_stock_transfer WHERE id = ?', [id]);
        if (transfer.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Transfer not found'
            });
        }

        if (transfer[0].status === 'Completed') {
            return res.status(400).json({
                success: false,
                message: 'Cannot cancel completed transfer'
            });
        }

        // Get transfer items
        const transferItems = await query(
            'SELECT * FROM tbl_stock_transfer_item WHERE transfer_id = ?',
            [id]
        );

        // Release reserved stock
        for (const item of transferItems) {
            await query(
                `UPDATE tbl_warehouse_stock 
                 SET reserved_quantity = reserved_quantity - ?
                 WHERE warehouse_id = ? AND stock_id = ?`,
                [item.quantity, transfer[0].from_warehouse_id, item.stock_id]
            );
        }

        // Update transfer status
        await query(
            'UPDATE tbl_stock_transfer SET status = "Cancelled" WHERE id = ?',
            [id]
        );

        res.status(200).json({
            success: true,
            message: 'Transfer cancelled successfully'
        });
    } catch (error) {
        next(error);
    }
};

