const { query } = require('../config/database');

/**
 * @desc    Get all warehouses
 * @route   GET /api/warehouses
 * @access  Private
 */
exports.getWarehouses = async (req, res, next) => {
    try {
        const { is_active, search } = req.query;
        
        console.log('📦 getWarehouses called with query params:', { is_active, search });
        
        let sql = `SELECT w.*, 
                   u.name as created_by_name,
                   COUNT(DISTINCT ws.stock_id) as total_products,
                   COUNT(DISTINCT ws.id) as stock_entries
                   FROM tbl_warehouse w
                   LEFT JOIN tbl_user u ON w.created_by = u.id
                   LEFT JOIN tbl_warehouse_stock ws ON w.id = ws.warehouse_id
                   WHERE 1=1`;
        const params = [];

        if (is_active !== undefined) {
            sql += ' AND w.is_active = ?';
            // Handle both '1'/'0' and 'true'/'false' string values
            const activeValue = is_active === 'true' || is_active === '1' || is_active === 1 || is_active === true ? 1 : 0;
            params.push(activeValue);
            console.log('🔍 Filtering by is_active:', activeValue, '(original value:', is_active, ')');
        }

        if (search) {
            sql += ' AND (w.name LIKE ? OR w.warehouse_code LIKE ? OR w.description LIKE ?)';
            params.push(`%${search}%`, `%${search}%`, `%${search}%`);
        }

        sql += ' GROUP BY w.id ORDER BY w.name';

        console.log('📝 Executing SQL:', sql);
        console.log('📝 With params:', params);

        const warehouses = await query(sql, params);
        
        console.log('✅ Query returned', warehouses.length, 'warehouses');
        if (warehouses.length > 0) {
            console.log('📦 Sample warehouse:', {
                id: warehouses[0].id,
                code: warehouses[0].warehouse_code,
                name: warehouses[0].name,
                is_active: warehouses[0].is_active
            });
        } else {
            // Let's also check if there are ANY warehouses in the database
            const allWarehouses = await query('SELECT id, warehouse_code, name, is_active FROM tbl_warehouse');
            console.log('⚠️ No warehouses returned by query, but total warehouses in DB:', allWarehouses.length);
            if (allWarehouses.length > 0) {
                console.log('📦 All warehouses in DB:', allWarehouses);
            }
        }

        res.status(200).json({
            success: true,
            count: warehouses.length,
            data: warehouses
        });
    } catch (error) {
        console.error('❌ Error in getWarehouses:', error);
        next(error);
    }
};

/**
 * @desc    Get single warehouse
 * @route   GET /api/warehouses/:id
 * @access  Private
 */
exports.getWarehouse = async (req, res, next) => {
    try {
        const sql = `SELECT w.*, u.name as created_by_name
                     FROM tbl_warehouse w
                     LEFT JOIN tbl_user u ON w.created_by = u.id
                     WHERE w.id = ?`;
        
        const warehouses = await query(sql, [req.params.id]);

        if (warehouses.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Warehouse not found'
            });
        }

        res.status(200).json({
            success: true,
            data: warehouses[0]
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Create warehouse
 * @route   POST /api/warehouses
 * @access  Private
 */
exports.createWarehouse = async (req, res, next) => {
    try {
        const {
            warehouse_code,
            name,
            description,
            address,
            phone,
            email,
            manager_name,
            manager_phone,
            is_active
        } = req.body;

        // Check if warehouse code already exists
        const existingCheck = await query(
            'SELECT id FROM tbl_warehouse WHERE warehouse_code = ?',
            [warehouse_code]
        );

        if (existingCheck.length > 0) {
            return res.status(400).json({
                success: false,
                message: 'Warehouse code already exists'
            });
        }

        const sql = `INSERT INTO tbl_warehouse 
                     (warehouse_code, name, description, address, phone, email, 
                      manager_name, manager_phone, is_active, created_by)
                     VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`;

        const result = await query(sql, [
            warehouse_code,
            name,
            description || null,
            address || null,
            phone || null,
            email || null,
            manager_name || null,
            manager_phone || null,
            is_active !== undefined ? is_active : 1,
            req.user?.id || null
        ]);

        res.status(201).json({
            success: true,
            message: 'Warehouse created successfully',
            data: {
                id: result.insertId,
                warehouse_code,
                name
            }
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Update warehouse
 * @route   PUT /api/warehouses/:id
 * @access  Private
 */
exports.updateWarehouse = async (req, res, next) => {
    try {
        const {
            warehouse_code,
            name,
            description,
            address,
            phone,
            email,
            manager_name,
            manager_phone,
            is_active
        } = req.body;

        // Check if warehouse code already exists (excluding current warehouse)
        if (warehouse_code) {
            const existingCheck = await query(
                'SELECT id FROM tbl_warehouse WHERE warehouse_code = ? AND id != ?',
                [warehouse_code, req.params.id]
            );

            if (existingCheck.length > 0) {
                return res.status(400).json({
                    success: false,
                    message: 'Warehouse code already exists'
                });
            }
        }

        const sql = `UPDATE tbl_warehouse 
                     SET warehouse_code = COALESCE(?, warehouse_code),
                         name = COALESCE(?, name),
                         description = ?,
                         address = ?,
                         phone = ?,
                         email = ?,
                         manager_name = ?,
                         manager_phone = ?,
                         is_active = COALESCE(?, is_active)
                     WHERE id = ?`;

        await query(sql, [
            warehouse_code,
            name,
            description,
            address,
            phone,
            email,
            manager_name,
            manager_phone,
            is_active,
            req.params.id
        ]);

        res.status(200).json({
            success: true,
            message: 'Warehouse updated successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Delete warehouse
 * @route   DELETE /api/warehouses/:id
 * @access  Private
 */
exports.deleteWarehouse = async (req, res, next) => {
    try {
        // Check if warehouse has stock
        const stockCheck = await query(
            'SELECT COUNT(*) as count FROM tbl_warehouse_stock WHERE warehouse_id = ?',
            [req.params.id]
        );

        if (stockCheck[0].count > 0) {
            return res.status(400).json({
                success: false,
                message: 'Cannot delete warehouse with existing stock. Please transfer or remove all stock first.'
            });
        }

        // Check if warehouse has pending transfers
        const transferCheck = await query(
            'SELECT COUNT(*) as count FROM tbl_stock_transfer WHERE (from_warehouse_id = ? OR to_warehouse_id = ?) AND status != "Completed" AND status != "Cancelled"',
            [req.params.id, req.params.id]
        );

        if (transferCheck[0].count > 0) {
            return res.status(400).json({
                success: false,
                message: 'Cannot delete warehouse with pending transfers. Please complete or cancel all transfers first.'
            });
        }

        const sql = 'DELETE FROM tbl_warehouse WHERE id = ?';
        await query(sql, [req.params.id]);

        res.status(200).json({
            success: true,
            message: 'Warehouse deleted successfully'
        });
    } catch (error) {
        next(error);
    }
};

/**
 * @desc    Get warehouse stock
 * @route   GET /api/warehouses/:id/stock
 * @access  Private
 */
exports.getWarehouseStock = async (req, res, next) => {
    try {
        const { id } = req.params; // Use 'id' from route parameter
        const { search, low_stock } = req.query;

        console.log('📦 getWarehouseStock called for warehouse_id:', id);
        console.log('📦 Query params:', { search, low_stock });

        let sql = `SELECT ws.*, 
                   s.name as product_name,
                   s.barcode,
                   s.cost_price,
                   s.sale_price_latli,
                   s.unit_name,
                   (ws.quantity - COALESCE(ws.reserved_quantity, 0)) as available_quantity
                   FROM tbl_warehouse_stock ws
                   INNER JOIN tbl_stock s ON ws.stock_id = s.id
                   WHERE ws.warehouse_id = ?`;
        const params = [id];

        if (search) {
            sql += ' AND (s.name LIKE ? OR s.barcode LIKE ?)';
            params.push(`%${search}%`, `%${search}%`);
        }

        if (low_stock === 'true') {
            sql += ' AND (ws.quantity - COALESCE(ws.reserved_quantity, 0)) <= 10';
        }

        sql += ' ORDER BY s.name';

        console.log('📝 Executing SQL:', sql);
        console.log('📝 With params:', params);

        const stock = await query(sql, params);
        
        console.log('✅ Query returned', stock.length, 'stock items');
        if (stock.length > 0) {
            console.log('📦 Sample stock item:', {
                stock_id: stock[0].stock_id,
                product_name: stock[0].product_name,
                quantity: stock[0].quantity,
                available: stock[0].available_quantity
            });
        } else {
            // Check if there's any stock in the warehouse at all
            const totalStock = await query(
                'SELECT COUNT(*) as count FROM tbl_warehouse_stock WHERE warehouse_id = ?',
                [id]
            );
            console.log('⚠️ No stock returned by query, but total stock entries in DB:', totalStock[0]?.count || 0);
        }

        res.status(200).json({
            success: true,
            count: stock.length,
            data: stock
        });
    } catch (error) {
        console.error('❌ Error in getWarehouseStock:', error);
        next(error);
    }
};

/**
 * @desc    Assign stock to warehouse
 * @route   POST /api/warehouses/:id/assign-stock
 * @access  Private
 */
exports.assignStock = async (req, res, next) => {
    try {
        const { id } = req.params; // Use 'id' from route parameter
        const { stock_id, quantity, assigned_person, notes } = req.body;

        console.log('📦 assignStock called with:', {
            warehouse_id: id,
            stock_id,
            quantity,
            user_id: req.user?.id
        });

        if (!stock_id || quantity === undefined || quantity === null) {
            console.error('❌ Missing required fields: stock_id or quantity');
            return res.status(400).json({
                success: false,
                message: 'Stock ID and quantity are required'
            });
        }

        if (quantity <= 0) {
            console.error('❌ Invalid quantity:', quantity);
            return res.status(400).json({
                success: false,
                message: 'Quantity must be greater than 0'
            });
        }

        // Check if warehouse exists
        const warehouse = await query('SELECT id, name FROM tbl_warehouse WHERE id = ?', [id]);
        if (warehouse.length === 0) {
            console.error('❌ Warehouse not found:', id);
            return res.status(404).json({
                success: false,
                message: 'Warehouse not found'
            });
        }
        console.log('✅ Warehouse found:', warehouse[0].name);

        // Check if stock exists
        const stock = await query('SELECT id, name FROM tbl_stock WHERE id = ?', [stock_id]);
        if (stock.length === 0) {
            console.error('❌ Stock item not found:', stock_id);
            return res.status(404).json({
                success: false,
                message: 'Stock item not found'
            });
        }
        console.log('✅ Stock item found:', stock[0].name);

        // Create stock movement record
        // The database trigger will automatically update warehouse stock
        // DO NOT manually update warehouse stock here to avoid double counting
        const movementSql = `INSERT INTO tbl_stock_movement
                             (movement_type, warehouse_id, stock_id, quantity, reference_type, 
                              movement_date, assigned_person, assigned_person_id, notes, created_by)
                             VALUES ('IN', ?, ?, ?, 'ASSIGNMENT', NOW(), ?, ?, ?, ?)`;

        console.log('📝 Creating stock movement record (trigger will update warehouse stock)...');
        await query(movementSql, [
            id,
            stock_id,
            quantity,
            assigned_person || null,
            req.user?.id || null,
            notes || null,
            req.user?.id || null
        ]);
        console.log('✅ Stock movement record created - warehouse stock updated by trigger');

        // Verify the assignment by querying the warehouse stock
        const verifyStock = await query(
            'SELECT quantity, reserved_quantity FROM tbl_warehouse_stock WHERE warehouse_id = ? AND stock_id = ?',
            [id, stock_id]
        );
        console.log('✅ Verification - Warehouse stock:', verifyStock[0]);

        res.status(201).json({
            success: true,
            message: 'Stock assigned to warehouse successfully',
            data: {
                warehouse_id: id,
                stock_id,
                quantity: verifyStock[0]?.quantity || quantity
            }
        });
    } catch (error) {
        console.error('❌ Error in assignStock:', error);
        next(error);
    }
};

/**
 * @desc    Update warehouse stock quantity
 * @route   PUT /api/warehouses/:warehouseId/stock/:stockId/quantity
 * @access  Private
 */
exports.updateWarehouseStockQuantity = async (req, res, next) => {
    try {
        const { warehouseId, warehouseStockId } = req.params;
        const stockId = warehouseStockId; // tbl_warehouse_stock.id used in queries
        const { operation = 'set', quantity, notes } = req.body;

        if (!['set', 'add', 'subtract'].includes(operation)) {
            return res.status(400).json({
                success: false,
                message: "Invalid operation. Use 'set', 'add', or 'subtract'."
            });
        }

        if (quantity === undefined || quantity === null || isNaN(parseInt(quantity)) || parseInt(quantity) < 0) {
            return res.status(400).json({
                success: false,
                message: 'Valid quantity is required'
            });
        }

        // Get current warehouse stock (lookup by id; verify warehouse_id)
        const currentStock = await query(
            'SELECT id, warehouse_id, quantity, reserved_quantity, stock_id FROM tbl_warehouse_stock WHERE id = ?',
            [stockId]
        );

        if (currentStock.length === 0) {
            return res.status(404).json({
                success: false,
                message: 'Warehouse stock record not found'
            });
        }

        if (String(currentStock[0].warehouse_id) !== String(warehouseId)) {
            return res.status(404).json({
                success: false,
                message: 'Warehouse stock does not belong to the selected warehouse'
            });
        }

        const oldQuantity = parseInt(currentStock[0].quantity) || 0;
        const reservedQty = parseInt(currentStock[0].reserved_quantity) || 0;
        const qtyInput = parseInt(quantity) || 0;

        let newQuantity = oldQuantity;
        if (operation === 'set') {
            newQuantity = qtyInput;
        } else if (operation === 'add') {
            newQuantity = oldQuantity + qtyInput;
        } else if (operation === 'subtract') {
            newQuantity = oldQuantity - qtyInput;
        }

        if (newQuantity < 0) {
            return res.status(400).json({
                success: false,
                message: 'Quantity cannot be negative'
            });
        }

        // Don't allow setting below reserved quantity (if reservations are used)
        if (newQuantity < reservedQty) {
            return res.status(400).json({
                success: false,
                message: `Quantity cannot be less than reserved quantity (${reservedQty})`
            });
        }

        const quantityDifference = newQuantity - oldQuantity;

        // Don't manually update warehouse stock - let the trigger handle it
        // Create stock movement record for audit trail - trigger will update warehouse stock
        if (quantityDifference !== 0) {
            const movementType = quantityDifference > 0 ? 'IN' : 'OUT';
            const movementSql = `INSERT INTO tbl_stock_movement
                                 (movement_type, warehouse_id, stock_id, quantity, reference_type, 
                                  movement_date, notes, created_by)
                                 VALUES (?, ?, ?, ?, 'ADJUSTMENT', NOW(), ?, ?)`;

            await query(movementSql, [
                movementType,
                warehouseId,
                currentStock[0].stock_id,
                Math.abs(quantityDifference),
                notes || `Manual stock ${operation}: ${oldQuantity} → ${newQuantity}`,
                req.user?.id || null
            ]);
            
            // Trigger will automatically update warehouse stock
            // Wait a moment for trigger to complete, then verify
            await new Promise(resolve => setTimeout(resolve, 100));
        }

        // Verify the updated warehouse stock quantity (after trigger has run)
        const updatedStock = await query(
            'SELECT quantity FROM tbl_warehouse_stock WHERE id = ? AND warehouse_id = ?',
            [stockId, warehouseId]
        );
        
        const actualNewQuantity = updatedStock.length > 0 ? parseInt(updatedStock[0].quantity) || 0 : newQuantity;

        // Update main stock quantity (sum of all warehouse stocks)
        const totalWarehouseQty = await query(
            'SELECT SUM(quantity) as total FROM tbl_warehouse_stock WHERE stock_id = ?',
            [currentStock[0].stock_id]
        );

        const totalQty = parseInt(totalWarehouseQty[0]?.total) || 0;
        await query(
            'UPDATE tbl_stock SET qty = ? WHERE id = ?',
            [totalQty, currentStock[0].stock_id]
        );

        res.json({
            success: true,
            message: 'Warehouse stock quantity updated successfully',
            data: {
                warehouse_stock_id: stockId,
                old_quantity: oldQuantity,
                new_quantity: actualNewQuantity,
                main_stock_quantity: totalQty
            }
        });
    } catch (error) {
        console.error('❌ Error updating warehouse stock quantity:', error);
        next(error);
    }
};

