# Permission System Quick Start Guide

## Installation

1. **Install dependencies:**
```bash
cd backend
npm install
```

2. **Run setup script:**
```bash
npm run permissions:setup
```

Or manually:
```bash
# Run migration
node migrations/001-create-permission-tables.js

# Run seeders
node seeders/001-seed-default-roles.js
node seeders/002-seed-default-permissions.js
node seeders/003-seed-role-permissions.js
```

3. **Verify installation:**
```bash
node scripts/diagnose-permissions.js
```

## Basic Usage

### Protect a Route

```javascript
const { requirePermission } = require('./middleware/permissions');
const { protect } = require('./middleware/auth');

router.post('/users', 
    protect, 
    requirePermission('users', 'create'), 
    createUserHandler
);
```

### Check Permission in Controller

```javascript
const { hasPermission } = require('../utils/permissions');

async function myController(req, res) {
    const canCreate = await hasPermission(
        req.user.id, 
        'patients', 
        'create',
        req.session
    );
    
    if (!canCreate) {
        return res.status(403).json({ error: 'Permission denied' });
    }
    
    // Your logic here
}
```

### Frontend (EJS)

```ejs
<% if (user.permissions?.patients?.create) { %>
    <button>Create Patient</button>
<% } %>
```

## Default Roles

- **Admin** (Level 100): Full access, bypasses all checks
- **Manager** (Level 50): Most permissions except user/settings management
- **Staff** (Level 10): Read/create/update for most resources

## Common Commands

```bash
# Diagnose user permissions
node scripts/diagnose-permissions.js [userId]

# List all permissions
node scripts/list-permissions.js

# List permissions for a role
node scripts/list-permissions.js --role=Manager

# Repair permission issues
node scripts/repair-permissions.js
```

## Next Steps

- Read full documentation: `backend/docs/PERMISSIONS_README.md`
- See examples: `backend/examples/permission-usage-examples.js`
- Test in your routes

