# Role-Based Permission System

A comprehensive role-based access control (RBAC) system for Clinic Pro V3, built with Sequelize and MySQL.

## Overview

This permission system provides:
- **Role hierarchy** with configurable levels
- **Resource/action-based permissions** (e.g., `users:create`, `patients:read`)
- **Session-based caching** for fast permission checks
- **Express middleware** for route protection
- **Admin bypass** for full system access

## Architecture

### Database Schema

- **`tbl_role`**: User roles (Admin, Manager, Staff) with hierarchy levels
- **`tbl_permission`**: Individual permissions as resource/action pairs
- **`tbl_role_permission`**: Junction table linking roles to permissions
- **`tbl_user`**: Users with `role_id` foreign key

### Key Components

1. **Models** (`backend/models/sequelize/`): Sequelize ORM models
2. **Migrations** (`backend/migrations/`): Database schema migrations
3. **Seeders** (`backend/seeders/`): Default roles and permissions
4. **Utils** (`backend/utils/permissions.js`): Core permission checking logic
5. **Middleware** (`backend/middleware/permissions.js`): Express route protection

## Installation & Setup

### 1. Install Dependencies

```bash
cd backend
npm install sequelize express-session sequelize-cli
```

### 2. Run Migrations

```bash
# Using Sequelize CLI
npm run sequelize:migrate

# Or manually run the migration
node migrations/001-create-permission-tables.js
```

### 3. Seed Default Data

```bash
# Using Sequelize CLI
npm run sequelize:seed

# Or use the setup script
npm run permissions:setup
```

### 4. Verify Setup

```bash
node scripts/diagnose-permissions.js
```

## Permission Resources

The system uses the following resource/action pattern:

### Resources
- `users` - User management
- `patients` - Patient records
- `doctors` - Doctor management
- `appointments` - Appointment scheduling
- `sales` - Sales transactions
- `purchases` - Purchase orders
- `stock` - Inventory management
- `reports` - Reports and analytics
- `settings` - System settings
- `dashboard` - Dashboard access
- `emr` - Electronic medical records
- `inpatients` - Inpatient management
- `laboratory` - Lab tests
- `pharmacy` - Pharmacy operations
- `warehouse` - Warehouse management
- `expenses` - Expense tracking
- `suppliers` - Supplier management

### Actions
- `create` - Create new records
- `read` - View records
- `update` - Modify records
- `delete` - Remove records
- `manage` - Full management (all CRUD)
- `export` - Export data
- `import` - Import data

## Default Roles

### Admin (Level 100)
- **Bypass**: All permission checks
- **Access**: Full system access
- **Metadata**: `{ bypassChecks: true }`

### Manager (Level 50)
- **Permissions**: Most resources except user management and system settings
- **Access**: Read-only for `users` and `settings`

### Staff (Level 10)
- **Permissions**: Read/create/update for most resources
- **Restrictions**: No delete for `users`, `settings`, `reports`

## Usage Examples

### Route Protection

```javascript
const { requirePermission, requireCrudPermission } = require('./middleware/permissions');
const { protect } = require('./middleware/auth');

// Single permission check
router.post('/users', 
    protect, 
    requirePermission('users', 'create'), 
    createUser
);

// CRUD permission (auto-maps HTTP methods)
router.use('/patients', protect, requireCrudPermission('patients'));

// Multiple permissions (all required)
router.get('/reports', 
    protect,
    requireAllPermissions([
        { resource: 'reports', action: 'read' },
        { resource: 'dashboard', action: 'read' }
    ]),
    getReports
);

// Any permission (at least one required)
router.post('/export', 
    protect,
    requireAnyPermission([
        { resource: 'reports', action: 'export' },
        { resource: 'sales', action: 'export' }
    ]),
    exportData
);
```

### In Controllers

```javascript
const { hasPermission, getUserPermissions } = require('../utils/permissions');

// Check permission in controller
async function createPatient(req, res) {
    const canCreate = await hasPermission(
        req.user.id, 
        'patients', 
        'create',
        req.session
    );
    
    if (!canCreate) {
        return res.status(403).json({ error: 'Permission denied' });
    }
    
    // Create patient...
}

// Get all user permissions
async function getDashboard(req, res) {
    const permissions = await getUserPermissions(req.user.id, req.session);
    
    res.json({
        user: req.user,
        permissions // { patients: { create: true, read: true, ... } }
    });
}
```

### Frontend (EJS/React)

#### EJS Example

```ejs
<% const userPerms = user.permissions || {}; %>

<% if (userPerms.users?.create) { %>
    <button onclick="createUser()">Create User</button>
<% } %>

<% if (userPerms.patients?.read) { %>
    <a href="/patients">View Patients</a>
<% } %>

<% if (userPerms.reports?.export) { %>
    <button onclick="exportReport()">Export</button>
<% } %>
```

#### React Example

```jsx
function UserManagement({ userPermissions }) {
    const canCreate = userPermissions?.users?.create;
    const canDelete = userPermissions?.users?.delete;
    
    return (
        <div>
            {canCreate && (
                <button onClick={handleCreate}>Create User</button>
            )}
            {canDelete && (
                <button onClick={handleDelete}>Delete User</button>
            )}
        </div>
    );
}
```

## Session Integration

Permissions are automatically loaded into the session on login:

```javascript
// On login (auth.controller.js)
const permissions = await getUserPermissions(user.id);
req.session.permissions = permissions;

// Response includes permissions
res.json({
    success: true,
    token,
    user: { ... },
    permissions // Frontend can use this
});
```

## Troubleshooting

### Permission Check Fails

1. **Check user role**: Verify user has a valid `role_id`
   ```sql
   SELECT u.id, u.name, r.name as role_name, r.level 
   FROM tbl_user u 
   JOIN tbl_role r ON u.role_id = r.id 
   WHERE u.id = ?;
   ```

2. **Verify permission exists**: Check if permission is in database
   ```sql
   SELECT * FROM tbl_permission 
   WHERE resource = 'users' AND action = 'create';
   ```

3. **Check role-permission assignment**:
   ```sql
   SELECT rp.*, r.name as role_name, p.resource, p.action
   FROM tbl_role_permission rp
   JOIN tbl_role r ON rp.role_id = r.id
   JOIN tbl_permission p ON rp.permission_id = p.id
   WHERE rp.role_id = ? AND rp.granted = 1 AND rp.active = 1;
   ```

4. **Clear cache**: Permission cache may be stale
   ```javascript
   const { clearPermissionCache } = require('./utils/permissions');
   clearPermissionCache(userId); // Clear for specific user
   clearPermissionCache(); // Clear all
   ```

### Admin Bypass Not Working

- Verify role metadata has `bypassChecks: true`
- Check role level is >= 100
- Ensure role is active

### Session Permissions Not Loading

- Check session middleware is configured in `server.js`
- Verify `SESSION_SECRET` is set in `.env`
- Check session cookie settings

## CLI Utilities

### Diagnose Permissions

```bash
node scripts/diagnose-permissions.js [userId]
```

Shows:
- User role and level
- All assigned permissions
- Permission cache status
- Missing permissions

### Repair Permissions

```bash
node scripts/repair-permissions.js
```

- Fixes orphaned role-permission assignments
- Reassigns default permissions to roles
- Clears stale cache entries

### List All Permissions

```bash
node scripts/list-permissions.js
```

Displays all resources, actions, and role assignments.

## Future Enhancements

- [ ] Permission groups for easier management
- [ ] Audit logging for permission changes
- [ ] Time-based permissions (expiration)
- [ ] Conditional permissions (based on data)
- [ ] Redis cache for distributed systems
- [ ] Permission inheritance from parent roles
- [ ] API endpoint for permission management UI

## API Reference

### `hasPermission(userId, resource, action, sessionData)`
Check if user has specific permission.

### `getUserPermissions(userId, sessionData)`
Get all permissions for user as structured object.

### `clearPermissionCache(userId)`
Clear permission cache (optional userId).

### `requirePermission(resource, action)`
Express middleware for single permission.

### `requireCrudPermission(resource)`
Express middleware that maps HTTP methods to CRUD actions.

### `requireAllPermissions(permissionList)`
Express middleware requiring all listed permissions.

### `requireAnyPermission(permissionList)`
Express middleware requiring at least one permission.

## Support

For issues or questions:
1. Check this documentation
2. Run diagnostic scripts
3. Review migration/seed logs
4. Check database schema integrity

