/**
 * Permission System Usage Examples
 * 
 * This file demonstrates how to use the permission system
 * in various scenarios (routes, controllers, frontend)
 */

// ============================================
// 1. ROUTE PROTECTION EXAMPLES
// ============================================

const express = require('express');
const router = express.Router();
const { protect } = require('../middleware/auth');
const {
    requirePermission,
    requireCrudPermission,
    requireAllPermissions,
    requireAnyPermission
} = require('../middleware/permissions');

// Example 1: Single permission check
router.post('/users',
    protect, // First authenticate
    requirePermission('users', 'create'), // Then check permission
    async (req, res) => {
        // User has 'users:create' permission
        res.json({ message: 'User created' });
    }
);

// Example 2: CRUD permission (auto-maps HTTP methods)
// GET -> read, POST -> create, PUT/PATCH -> update, DELETE -> delete
router.use('/patients',
    protect,
    requireCrudPermission('patients')
);
// Now all routes under /patients are protected:
// GET /patients -> requires 'patients:read'
// POST /patients -> requires 'patients:create'
// PUT /patients/:id -> requires 'patients:update'
// DELETE /patients/:id -> requires 'patients:delete'

// Example 3: Multiple permissions (all required)
router.get('/reports/advanced',
    protect,
    requireAllPermissions([
        { resource: 'reports', action: 'read' },
        { resource: 'dashboard', action: 'read' }
    ]),
    async (req, res) => {
        // User must have BOTH permissions
        res.json({ reports: [] });
    }
);

// Example 4: Any permission (at least one required)
router.post('/export',
    protect,
    requireAnyPermission([
        { resource: 'reports', action: 'export' },
        { resource: 'sales', action: 'export' },
        { resource: 'patients', action: 'export' }
    ]),
    async (req, res) => {
        // User needs at least ONE of these permissions
        res.json({ exported: true });
    }
);

// ============================================
// 2. CONTROLLER EXAMPLES
// ============================================

const { hasPermission, getUserPermissions } = require('../utils/permissions');

// Example 5: Check permission in controller logic
async function createPatientController(req, res) {
    try {
        // Check permission manually
        const canCreate = await hasPermission(
            req.user.id,
            'patients',
            'create',
            req.session // Pass session for cache lookup
        );

        if (!canCreate) {
            return res.status(403).json({
                success: false,
                message: 'You do not have permission to create patients'
            });
        }

        // Proceed with creation
        const patient = await createPatient(req.body);
        res.json({ success: true, data: patient });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
}

// Example 6: Get all permissions for frontend
async function getDashboardData(req, res) {
    try {
        // Get all user permissions
        const permissions = await getUserPermissions(
            req.user.id,
            req.session
        );

        // permissions structure:
        // {
        //   patients: { create: true, read: true, update: true, delete: false },
        //   users: { create: false, read: true, update: false, delete: false },
        //   ...
        // }

        res.json({
            success: true,
            data: {
                user: req.user,
                permissions, // Send to frontend
                dashboard: await getDashboardStats()
            }
        });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
}

// Example 7: Conditional logic based on permissions
async function getPatientList(req, res) {
    try {
        const canViewAll = await hasPermission(
            req.user.id,
            'patients',
            'manage',
            req.session
        );

        // If user can manage, show all patients
        // Otherwise, show only their assigned patients
        const patients = canViewAll
            ? await getAllPatients()
            : await getAssignedPatients(req.user.id);

        res.json({ success: true, data: patients });
    } catch (error) {
        res.status(500).json({ success: false, message: error.message });
    }
}

// ============================================
// 3. FRONTEND EXAMPLES (EJS)
// ============================================

/*
<!-- Example 8: EJS Template with Permission Checks -->
<% const perms = user.permissions || {}; %>

<!-- Show button only if user can create -->
<% if (perms.users?.create) { %>
    <button onclick="createUser()" class="btn btn-primary">
        Create User
    </button>
<% } %>

<!-- Show menu item only if user can read -->
<% if (perms.patients?.read) { %>
    <li>
        <a href="/patients">Patients</a>
    </li>
<% } %>

<!-- Conditional table actions -->
<table>
    <tr>
        <th>Name</th>
        <th>Actions</th>
    </tr>
    <% patients.forEach(patient => { %>
        <tr>
            <td><%= patient.name %></td>
            <td>
                <% if (perms.patients?.update) { %>
                    <button onclick="editPatient(<%= patient.id %>)">Edit</button>
                <% } %>
                <% if (perms.patients?.delete) { %>
                    <button onclick="deletePatient(<%= patient.id %>)">Delete</button>
                <% } %>
            </td>
        </tr>
    <% }); %>
</table>
*/

// ============================================
// 4. FRONTEND EXAMPLES (React)
// ============================================

/*
// Example 9: React Component with Permissions
import React from 'react';

function UserManagement({ userPermissions }) {
    const canCreate = userPermissions?.users?.create;
    const canUpdate = userPermissions?.users?.update;
    const canDelete = userPermissions?.users?.delete;

    return (
        <div>
            <h1>User Management</h1>
            
            {canCreate && (
                <button onClick={handleCreate} className="btn-primary">
                    Create User
                </button>
            )}

            <table>
                <thead>
                    <tr>
                        <th>Name</th>
                        <th>Role</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    {users.map(user => (
                        <tr key={user.id}>
                            <td>{user.name}</td>
                            <td>{user.role}</td>
                            <td>
                                {canUpdate && (
                                    <button onClick={() => handleEdit(user.id)}>
                                        Edit
                                    </button>
                                )}
                                {canDelete && (
                                    <button onClick={() => handleDelete(user.id)}>
                                        Delete
                                    </button>
                                )}
                            </td>
                        </tr>
                    ))}
                </tbody>
            </table>
        </div>
    );
}

// Example 10: React Hook for Permissions
function usePermissions() {
    const [permissions, setPermissions] = React.useState({});

    React.useEffect(() => {
        // Fetch permissions from API
        fetch('/api/auth/me')
            .then(res => res.json())
            .then(data => {
                if (data.success && data.data.permissions) {
                    setPermissions(data.data.permissions);
                }
            });
    }, []);

    const hasPermission = (resource, action) => {
        return permissions[resource]?.[action] === true;
    };

    return { permissions, hasPermission };
}

// Usage in component
function MyComponent() {
    const { hasPermission } = usePermissions();

    return (
        <div>
            {hasPermission('patients', 'create') && (
                <button>Create Patient</button>
            )}
        </div>
    );
}
*/

// ============================================
// 5. ADVANCED EXAMPLES
// ============================================

// Example 11: Dynamic route protection
function createProtectedRoute(resource, action) {
    return [
        protect,
        requirePermission(resource, action)
    ];
}

// Usage:
router.post('/custom-endpoint',
    ...createProtectedRoute('custom', 'create'),
    handler
);

// Example 12: Permission-based data filtering
async function getFilteredData(req, res) {
    const permissions = await getUserPermissions(req.user.id, req.session);
    
    let query = {};
    
    // Only show sensitive data if user has manage permission
    if (!permissions.patients?.manage) {
        query.hideSensitive = true;
    }
    
    const data = await fetchData(query);
    res.json({ success: true, data });
}

module.exports = {
    createPatientController,
    getDashboardData,
    getPatientList
};

