/**
 * Authentication Middleware
 */

const jwt = require('jsonwebtoken');
const { query } = require('../config/database');

/**
 * Protect routes - requires valid JWT token
 */
exports.protect = async (req, res, next) => {
    let token;

    // Check for token in headers
    if (req.headers.authorization && req.headers.authorization.startsWith('Bearer')) {
        token = req.headers.authorization.split(' ')[1];
    }

    // Make sure token exists
    if (!token) {
        return res.status(401).json({
            success: false,
            message: 'Not authorized to access this route'
        });
    }

    try {
        // Verify token
        const decoded = jwt.verify(token, process.env.JWT_SECRET);

        // Get user from database
        const sql = 'SELECT id, name, role_id FROM tbl_user WHERE id = ?';
        const users = await query(sql, [decoded.id]);

        if (users.length === 0) {
            return res.status(401).json({
                success: false,
                message: 'User not found'
            });
        }

        req.user = users[0];
        next();
    } catch (error) {
        return res.status(401).json({
            success: false,
            message: 'Not authorized to access this route'
        });
    }
};

/**
 * Grant access to specific roles
 */
exports.authorize = (...roles) => {
    return (req, res, next) => {
        if (!roles.includes(req.user.role_id)) {
            return res.status(403).json({
                success: false,
                message: `User role ${req.user.role_id} is not authorized to access this route`
            });
        }
        next();
    };
};

