/**
 * Permission Middleware
 * Express middleware for route protection based on permissions
 */

const { hasPermission, hasAnyPermission, hasAllPermissions } = require('../utils/permissions');
const { protect } = require('./auth');

/**
 * Require a specific permission
 * @param {string} resource - Resource name
 * @param {string} action - Action name
 * @returns {Function} Express middleware
 */
function requirePermission(resource, action) {
    return async (req, res, next) => {
        // First ensure user is authenticated
        if (!req.user || !req.user.id) {
            return res.status(401).json({
                success: false,
                message: 'Authentication required'
            });
        }

        // Quick admin check before full permission check
        // This allows admins to access permission management pages
        try {
            const { query } = require('../config/database');
            
            // Use direct SQL query for reliability
            const sql = `
                SELECT r.level, r.metadata, r.active, r.id as role_id, r.name as role_name
                FROM tbl_role r
                INNER JOIN tbl_user u ON u.role_id = r.id
                WHERE u.id = ? AND r.active = 1
                LIMIT 1
            `;
            const roleResult = await query(sql, [req.user.id]);
            
            if (roleResult && roleResult.length > 0) {
                const role = roleResult[0];
                
                console.log(`[Permission Check] User ${req.user.id} - Role: ${role.role_name}, Level: ${role.level}, Active: ${role.active}`);
                
                // Check if role is admin (level >= 100 AND name is admin)
                if (role.level >= 100 && String(role.role_name || '').toLowerCase() === 'admin') {
                    console.log(`[Permission Check] Admin bypass granted for user ${req.user.id} (role ${role.role_name}, level ${role.level})`);
                    return next(); // Admin bypass
                }
                
                // Check metadata for bypassChecks flag
                if (role.metadata) {
                    let metadata = role.metadata;
                    if (typeof metadata === 'string') {
                        try {
                            metadata = JSON.parse(metadata);
                        } catch (e) {
                            // If parsing fails, continue
                        }
                    }
                    if (metadata && metadata.bypassChecks === true) {
                        console.log(`[Permission Check] Admin bypass granted for user ${req.user.id} (bypassChecks flag)`);
                        return next(); // Admin bypass
                    }
                }
            } else {
                console.log(`[Permission Check] No active role found for user ${req.user.id}`);
            }
        } catch (error) {
            console.error('[Permission Check] Error checking admin status:', error);
            // Continue to permission check if admin check fails
        }

        // Check permission
        const hasPerm = await hasPermission(
            req.user.id,
            resource,
            action,
            req.session
        );

        if (!hasPerm) {
            return res.status(403).json({
                success: false,
                message: `Permission denied: ${action} ${resource}`
            });
        }

        next();
    };
}

/**
 * Require CRUD permissions (create, read, update, delete) for a resource
 * Automatically maps HTTP methods to actions
 * @param {string} resource - Resource name
 * @returns {Function} Express middleware
 */
function requireCrudPermission(resource) {
    return async (req, res, next) => {
        if (!req.user || !req.user.id) {
            return res.status(401).json({
                success: false,
                message: 'Authentication required'
            });
        }

        // Map HTTP methods to actions
        const methodActionMap = {
            'GET': 'read',
            'POST': 'create',
            'PUT': 'update',
            'PATCH': 'update',
            'DELETE': 'delete'
        };

        const action = methodActionMap[req.method] || 'read';

        const hasPerm = await hasPermission(
            req.user.id,
            resource,
            action,
            req.session
        );

        if (!hasPerm) {
            return res.status(403).json({
                success: false,
                message: `Permission denied: ${action} ${resource}`
            });
        }

        next();
    };
}

/**
 * Require all specified permissions
 * @param {Array<{resource: string, action: string}>} permissionList - Array of permission objects
 * @returns {Function} Express middleware
 */
function requireAllPermissions(permissionList) {
    return async (req, res, next) => {
        if (!req.user || !req.user.id) {
            return res.status(401).json({
                success: false,
                message: 'Authentication required'
            });
        }

        const hasAll = await hasAllPermissions(
            req.user.id,
            permissionList,
            req.session
        );

        if (!hasAll) {
            return res.status(403).json({
                success: false,
                message: 'Permission denied: missing required permissions'
            });
        }

        next();
    };
}

/**
 * Require any of the specified permissions
 * @param {Array<{resource: string, action: string}>} permissionList - Array of permission objects
 * @returns {Function} Express middleware
 */
function requireAnyPermission(permissionList) {
    return async (req, res, next) => {
        if (!req.user || !req.user.id) {
            return res.status(401).json({
                success: false,
                message: 'Authentication required'
            });
        }

        const hasAny = await hasAnyPermission(
            req.user.id,
            permissionList,
            req.session
        );

        if (!hasAny) {
            return res.status(403).json({
                success: false,
                message: 'Permission denied: missing required permissions'
            });
        }

        next();
    };
}

/**
 * Combined middleware: authentication + permission check
 * @param {string} resource - Resource name
 * @param {string} action - Action name
 * @returns {Array} Array of Express middlewares
 */
function requireAuthAndPermission(resource, action) {
    return [
        protect,
        requirePermission(resource, action)
    ];
}

/**
 * Combined middleware: authentication + CRUD permission check
 * @param {string} resource - Resource name
 * @returns {Array} Array of Express middlewares
 */
function requireAuthAndCrud(resource) {
    return [
        protect,
        requireCrudPermission(resource)
    ];
}

module.exports = {
    requirePermission,
    requireCrudPermission,
    requireAllPermissions,
    requireAnyPermission,
    requireAuthAndPermission,
    requireAuthAndCrud
};

