/**
 * File Upload Middleware using Multer
 * Handles product images and medical images
 */

const multer = require('multer');
const path = require('path');
const fs = require('fs');

// Create uploads directories if they don't exist
// Paths are relative to backend/middleware/ so '../uploads' goes to backend/uploads
const productUploadDir = path.join(__dirname, '../uploads/products');
const medicalUploadDir = path.join(__dirname, '../uploads/medical');
const generalUploadDir = path.join(__dirname, '../uploads');
const backupsTempDir = path.join(__dirname, '../uploads');

[productUploadDir, medicalUploadDir, generalUploadDir, backupsTempDir].forEach(dir => {
    if (!fs.existsSync(dir)) {
        fs.mkdirSync(dir, { recursive: true });
    }
});

// Storage configuration for products
const productStorage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, productUploadDir);
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = path.extname(file.originalname);
        const nameWithoutExt = path.basename(file.originalname, ext);
        cb(null, `${nameWithoutExt}-${uniqueSuffix}${ext}`);
    }
});

// Storage configuration for medical images
const medicalStorage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, medicalUploadDir);
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = path.extname(file.originalname);
        cb(null, `medical-${uniqueSuffix}${ext}`);
    }
});

// General storage configuration (for doctor photos, logos, etc.)
const generalStorage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, generalUploadDir);
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = path.extname(file.originalname);
        const nameWithoutExt = path.basename(file.originalname, ext);
        // Sanitize filename: replace spaces and special characters with hyphens
        const sanitizedName = nameWithoutExt
            .replace(/\s+/g, '-')           // Replace spaces with hyphens
            .replace(/[^\w\-]/g, '')        // Remove special characters except hyphens
            .replace(/\-+/g, '-')           // Replace multiple hyphens with single hyphen
            .toLowerCase()                   // Convert to lowercase
            .substring(0, 50);              // Limit length to 50 chars
        cb(null, `${sanitizedName}-${uniqueSuffix}${ext}`);
    }
});

// Storage configuration for temporary backup uploads (.sql files)
const backupStorage = multer.diskStorage({
    destination: (req, file, cb) => {
        cb(null, backupsTempDir);
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = path.extname(file.originalname) || '.sql';
        const nameWithoutExt = path.basename(file.originalname, ext);
        const sanitizedName = nameWithoutExt
            .replace(/\s+/g, '-')
            .replace(/[^\w\-]/g, '')
            .replace(/\-+/g, '-')
            .toLowerCase()
            .substring(0, 80);
        cb(null, `${sanitizedName}-${uniqueSuffix}${ext}`);
    }
});

// File filter - only images
const fileFilter = (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif|webp|bmp|tiff/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
        return cb(null, true);
    } else {
        cb(new Error('Only image files are allowed (jpeg, jpg, png, gif, webp, bmp, tiff)'));
    }
};

// File filter for backups - only .sql
const backupFileFilter = (req, file, cb) => {
    const ext = path.extname(file.originalname).toLowerCase();
    const isSql = ext === '.sql';
    const mime = (file.mimetype || '').toLowerCase();
    const allowedMimes = [
        'application/sql',
        'application/x-sql',
        'text/plain',
        'application/octet-stream'
    ];
    if (isSql || allowedMimes.includes(mime)) {
        return cb(null, true);
    }
    cb(new Error('Only .sql backup files are allowed'));
};

// Multer upload configurations
const productUpload = multer({
    storage: productStorage,
    limits: {
        fileSize: parseInt(process.env.MAX_FILE_SIZE) || 10 * 1024 * 1024 // 10MB default
    },
    fileFilter: fileFilter
});

const medicalUpload = multer({
    storage: medicalStorage,
    limits: {
        fileSize: parseInt(process.env.MAX_FILE_SIZE) || 10 * 1024 * 1024 // 10MB default
    },
    fileFilter: fileFilter
});

const generalUpload = multer({
    storage: generalStorage,
    limits: {
        fileSize: parseInt(process.env.MAX_FILE_SIZE) || 10 * 1024 * 1024 // 10MB default
    },
    fileFilter: fileFilter
});

// Multer upload configuration for backups
const backupUpload = multer({
    storage: backupStorage,
    limits: {
        fileSize: parseInt(process.env.MAX_BACKUP_FILE_SIZE) || 200 * 1024 * 1024 // 200MB default
    },
    fileFilter: backupFileFilter
});

// Middleware for single image upload (products)
const uploadSingle = productUpload.single('image');

// Middleware for multiple images (medical records - up to 10 images)
const uploadMultiple = medicalUpload.array('images', 10);

// Error handling wrapper
const handleUploadError = (uploadMiddleware) => {
    return (req, res, next) => {
        uploadMiddleware(req, res, (err) => {
            if (err instanceof multer.MulterError) {
                if (err.code === 'LIMIT_FILE_SIZE') {
                    return res.status(400).json({
                        success: false,
                        message: 'File size too large. Maximum size is 10MB per file'
                    });
                }
                if (err.code === 'LIMIT_FILE_COUNT') {
                    return res.status(400).json({
                        success: false,
                        message: 'Too many files. Maximum is 10 images'
                    });
                }
                return res.status(400).json({
                    success: false,
                    message: err.message
                });
            } else if (err) {
                return res.status(400).json({
                    success: false,
                    message: err.message
                });
            }
            next();
        });
    };
};

module.exports = {
    uploadSingle: handleUploadError(uploadSingle),
    uploadMultiple: handleUploadError(uploadMultiple),
    upload: generalUpload,  // General upload object for doctor photos, etc.
    backupUpload  // For .sql backup files
};
