/**
 * Migration: Create Permission System Tables
 * Creates tbl_permission, updates tbl_role, creates tbl_role_permission
 */

'use strict';

module.exports = {
    async up(queryInterface, Sequelize) {
        const { DataTypes } = Sequelize;

        // Check if tbl_role exists and update it
        const [tables] = await queryInterface.sequelize.query(
            "SHOW TABLES LIKE 'tbl_role'"
        );

        if (tables.length > 0) {
            // Update existing tbl_role table
            const [columns] = await queryInterface.sequelize.query(
                "SHOW COLUMNS FROM tbl_role LIKE 'level'"
            );

            if (columns.length === 0) {
                await queryInterface.addColumn('tbl_role', 'level', {
                    type: DataTypes.INTEGER,
                    allowNull: false,
                    defaultValue: 0,
                    after: 'name'
                });
            }

            const [descColumns] = await queryInterface.sequelize.query(
                "SHOW COLUMNS FROM tbl_role LIKE 'description'"
            );
            if (descColumns.length === 0) {
                await queryInterface.addColumn('tbl_role', 'description', {
                    type: DataTypes.TEXT,
                    allowNull: true,
                    after: 'level'
                });
            }

            const [activeColumns] = await queryInterface.sequelize.query(
                "SHOW COLUMNS FROM tbl_role LIKE 'active'"
            );
            if (activeColumns.length === 0) {
                await queryInterface.addColumn('tbl_role', 'active', {
                    type: DataTypes.BOOLEAN,
                    allowNull: false,
                    defaultValue: true,
                    after: 'description'
                });
            }

            const [metaColumns] = await queryInterface.sequelize.query(
                "SHOW COLUMNS FROM tbl_role LIKE 'metadata'"
            );
            if (metaColumns.length === 0) {
                await queryInterface.addColumn('tbl_role', 'metadata', {
                    type: DataTypes.JSON,
                    allowNull: true,
                    after: 'active'
                });
            }

            const [createdColumns] = await queryInterface.sequelize.query(
                "SHOW COLUMNS FROM tbl_role LIKE 'created_at'"
            );
            if (createdColumns.length === 0) {
                await queryInterface.addColumn('tbl_role', 'created_at', {
                    type: DataTypes.DATE,
                    allowNull: false,
                    defaultValue: Sequelize.literal('CURRENT_TIMESTAMP')
                });
            }

            const [updatedColumns] = await queryInterface.sequelize.query(
                "SHOW COLUMNS FROM tbl_role LIKE 'updated_at'"
            );
            if (updatedColumns.length === 0) {
                await queryInterface.addColumn('tbl_role', 'updated_at', {
                    type: DataTypes.DATE,
                    allowNull: false,
                    defaultValue: Sequelize.literal('CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP')
                });
            }
        } else {
            // Create tbl_role if it doesn't exist
            await queryInterface.createTable('tbl_role', {
                id: {
                    type: DataTypes.INTEGER,
                    primaryKey: true,
                    autoIncrement: true
                },
                name: {
                    type: DataTypes.STRING(100),
                    allowNull: false,
                    unique: true
                },
                level: {
                    type: DataTypes.INTEGER,
                    allowNull: false,
                    defaultValue: 0
                },
                description: {
                    type: DataTypes.TEXT,
                    allowNull: true
                },
                active: {
                    type: DataTypes.BOOLEAN,
                    allowNull: false,
                    defaultValue: true
                },
                metadata: {
                    type: DataTypes.JSON,
                    allowNull: true
                },
                created_at: {
                    type: DataTypes.DATE,
                    allowNull: false,
                    defaultValue: Sequelize.literal('CURRENT_TIMESTAMP')
                },
                updated_at: {
                    type: DataTypes.DATE,
                    allowNull: false,
                    defaultValue: Sequelize.literal('CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP')
                }
            });
        }

        // Create tbl_permission table
        await queryInterface.createTable('tbl_permission', {
            id: {
                type: DataTypes.INTEGER,
                primaryKey: true,
                autoIncrement: true
            },
            resource: {
                type: DataTypes.STRING(100),
                allowNull: false
            },
            action: {
                type: DataTypes.STRING(50),
                allowNull: false
            },
            description: {
                type: DataTypes.TEXT,
                allowNull: true
            },
            active: {
                type: DataTypes.BOOLEAN,
                allowNull: false,
                defaultValue: true
            },
            metadata: {
                type: DataTypes.JSON,
                allowNull: true
            },
            created_at: {
                type: DataTypes.DATE,
                allowNull: false,
                defaultValue: Sequelize.literal('CURRENT_TIMESTAMP')
            },
            updated_at: {
                type: DataTypes.DATE,
                allowNull: false,
                defaultValue: Sequelize.literal('CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP')
            }
        });

        // Create unique index on resource + action
        await queryInterface.addIndex('tbl_permission', ['resource', 'action'], {
            unique: true,
            name: 'unique_resource_action'
        });

        // Create tbl_role_permission junction table
        await queryInterface.createTable('tbl_role_permission', {
            id: {
                type: DataTypes.INTEGER,
                primaryKey: true,
                autoIncrement: true
            },
            role_id: {
                type: DataTypes.INTEGER,
                allowNull: false,
                references: {
                    model: 'tbl_role',
                    key: 'id'
                },
                onDelete: 'CASCADE'
            },
            permission_id: {
                type: DataTypes.INTEGER,
                allowNull: false,
                references: {
                    model: 'tbl_permission',
                    key: 'id'
                },
                onDelete: 'CASCADE'
            },
            granted: {
                type: DataTypes.BOOLEAN,
                allowNull: false,
                defaultValue: true
            },
            active: {
                type: DataTypes.BOOLEAN,
                allowNull: false,
                defaultValue: true
            },
            metadata: {
                type: DataTypes.JSON,
                allowNull: true
            },
            created_at: {
                type: DataTypes.DATE,
                allowNull: false,
                defaultValue: Sequelize.literal('CURRENT_TIMESTAMP')
            },
            updated_at: {
                type: DataTypes.DATE,
                allowNull: false,
                defaultValue: Sequelize.literal('CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP')
            }
        });

        // Create unique index on role_id + permission_id
        await queryInterface.addIndex('tbl_role_permission', ['role_id', 'permission_id'], {
            unique: true,
            name: 'unique_role_permission'
        });

        // Update tbl_user to ensure role_id is an integer (if it's currently VARCHAR)
        const [userColumns] = await queryInterface.sequelize.query(
            "SHOW COLUMNS FROM tbl_user WHERE Field = 'role_id'"
        );
        if (userColumns.length > 0 && userColumns[0].Type.includes('varchar')) {
            // Convert VARCHAR role_id to INT
            await queryInterface.sequelize.query(`
                ALTER TABLE tbl_user 
                MODIFY COLUMN role_id INT NOT NULL,
                ADD CONSTRAINT fk_user_role 
                FOREIGN KEY (role_id) REFERENCES tbl_role(id) ON DELETE RESTRICT
            `);
        } else if (userColumns.length > 0) {
            // Add foreign key if it doesn't exist
            const [fks] = await queryInterface.sequelize.query(`
                SELECT CONSTRAINT_NAME 
                FROM information_schema.KEY_COLUMN_USAGE 
                WHERE TABLE_NAME = 'tbl_user' 
                AND COLUMN_NAME = 'role_id' 
                AND CONSTRAINT_NAME != 'PRIMARY'
            `);
            if (fks.length === 0) {
                await queryInterface.addConstraint('tbl_user', {
                    fields: ['role_id'],
                    type: 'foreign key',
                    name: 'fk_user_role',
                    references: {
                        table: 'tbl_role',
                        field: 'id'
                    },
                    onDelete: 'RESTRICT'
                });
            }
        }
    },

    async down(queryInterface, Sequelize) {
        // Drop tables in reverse order
        await queryInterface.dropTable('tbl_role_permission');
        await queryInterface.dropTable('tbl_permission');
        
        // Note: We don't drop tbl_role as it may have existing data
        // Just remove the added columns
        const [tables] = await queryInterface.sequelize.query(
            "SHOW TABLES LIKE 'tbl_role'"
        );
        if (tables.length > 0) {
            const [columns] = await queryInterface.sequelize.query(
                "SHOW COLUMNS FROM tbl_role"
            );
            const columnNames = columns.map(c => c.Field);
            
            if (columnNames.includes('metadata')) {
                await queryInterface.removeColumn('tbl_role', 'metadata');
            }
            if (columnNames.includes('active')) {
                await queryInterface.removeColumn('tbl_role', 'active');
            }
            if (columnNames.includes('description')) {
                await queryInterface.removeColumn('tbl_role', 'description');
            }
            if (columnNames.includes('level')) {
                await queryInterface.removeColumn('tbl_role', 'level');
            }
            if (columnNames.includes('created_at')) {
                await queryInterface.removeColumn('tbl_role', 'created_at');
            }
            if (columnNames.includes('updated_at')) {
                await queryInterface.removeColumn('tbl_role', 'updated_at');
            }
        }
    }
};

