/**
 * Migration: Create All 97 Tables and Trigger
 * Creates all database tables and the warehouse stock movement trigger
 * from the clinic_pro_db.sql dump file
 */

'use strict';

const fs = require('fs');
const path = require('path');

module.exports = {
    async up(queryInterface, Sequelize) {
        const sqlFilePath = path.join(__dirname, '../../clinic_pro_db.sql');
        
        // Check if SQL file exists
        if (!fs.existsSync(sqlFilePath)) {
            throw new Error(`SQL file not found: ${sqlFilePath}`);
        }

        console.log('📖 Reading SQL file...');
        const sqlContent = fs.readFileSync(sqlFilePath, 'utf8');

        // Extract only CREATE TABLE and CREATE TRIGGER statements
        // Remove data insertion statements and other non-structure SQL
        const lines = sqlContent.split('\n');
        const createStatements = [];
        let inCreateStatement = false;
        let currentStatement = '';
        let inTrigger = false;

        for (let i = 0; i < lines.length; i++) {
            const line = lines[i].trim();
            
            // Skip comments and empty lines
            if (line.startsWith('--') || line === '' || line.startsWith('/*!')) {
                continue;
            }

            // Detect CREATE TABLE statements
            if (line.match(/^CREATE TABLE/i)) {
                inCreateStatement = true;
                currentStatement = line;
                continue;
            }

            // Detect CREATE TRIGGER statements
            if (line.match(/CREATE.*TRIGGER/i)) {
                inTrigger = true;
                currentStatement = line;
                continue;
            }

            // Detect DELIMITER for triggers
            if (line.match(/^DELIMITER/i)) {
                if (line.includes(';;')) {
                    inTrigger = true;
                } else if (line.includes(';')) {
                    inTrigger = false;
                }
                continue;
            }

            // Continue building statement
            if (inCreateStatement || inTrigger) {
                currentStatement += '\n' + line;
                
                // Check if statement ends
                if (line.endsWith(';') || line.endsWith(';;')) {
                    // Clean up the statement
                    let cleanStatement = currentStatement
                        .replace(/\/\*!50003.*?\*\//g, '') // Remove MySQL version comments
                        .replace(/\/\*!40101.*?\*\//g, '') // Remove MySQL version comments
                        .replace(/\/\*!50017.*?\*\//g, '') // Remove DEFINER comments
                        .trim();

                    // Only add if it's a valid CREATE statement
                    if (cleanStatement.match(/CREATE (TABLE|TRIGGER)/i)) {
                        createStatements.push(cleanStatement);
                    }
                    
                    currentStatement = '';
                    inCreateStatement = false;
                    if (line.endsWith(';')) {
                        inTrigger = false;
                    }
                }
            }
        }

        console.log(`📊 Found ${createStatements.length} CREATE statements to execute`);

        // Disable foreign key checks temporarily
        await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 0');
        await queryInterface.sequelize.query('SET UNIQUE_CHECKS = 0');

        try {
            // Execute each CREATE statement
            for (let i = 0; i < createStatements.length; i++) {
                const statement = createStatements[i];
                
                // Skip if table already exists (for idempotency)
                const tableMatch = statement.match(/CREATE TABLE[^`]*`([^`]+)`/i);
                const triggerMatch = statement.match(/CREATE.*TRIGGER[^`]*`([^`]+)`/i);
                
                if (tableMatch) {
                    const tableName = tableMatch[1];
                    try {
                        // Check if table exists
                        const [tables] = await queryInterface.sequelize.query(
                            `SHOW TABLES LIKE '${tableName}'`
                        );
                        
                        if (tables.length > 0) {
                            console.log(`⏭️  Table ${tableName} already exists, skipping...`);
                            continue;
                        }
                    } catch (err) {
                        // Continue if check fails
                    }
                }

                if (triggerMatch) {
                    const triggerName = triggerMatch[1];
                    try {
                        // Check if trigger exists
                        const [triggers] = await queryInterface.sequelize.query(
                            `SHOW TRIGGERS WHERE Trigger = '${triggerName}'`
                        );
                        
                        if (triggers.length > 0) {
                            console.log(`⏭️  Trigger ${triggerName} already exists, skipping...`);
                            continue;
                        }
                    } catch (err) {
                        // Continue if check fails
                    }
                }

                // Execute the statement
                try {
                    // Replace DELIMITER syntax for triggers
                    let execStatement = statement.replace(/DELIMITER\s+;;/gi, '');
                    execStatement = execStatement.replace(/DELIMITER\s+;/gi, '');
                    execStatement = execStatement.replace(/;;/g, ';');
                    
                    await queryInterface.sequelize.query(execStatement);
                    
                    if (tableMatch) {
                        console.log(`✅ Created table: ${tableMatch[1]}`);
                    } else if (triggerMatch) {
                        console.log(`✅ Created trigger: ${triggerMatch[1]}`);
                    }
                } catch (error) {
                    // Log error but continue with other statements
                    console.error(`⚠️  Error executing statement ${i + 1}:`, error.message);
                    if (tableMatch) {
                        console.error(`   Table: ${tableMatch[1]}`);
                    } else if (triggerMatch) {
                        console.error(`   Trigger: ${triggerMatch[1]}`);
                    }
                    // Don't throw - continue with other tables
                }
            }
        } finally {
            // Re-enable foreign key checks
            await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 1');
            await queryInterface.sequelize.query('SET UNIQUE_CHECKS = 1');
        }

        console.log('✅ Migration completed!');
    },

    async down(queryInterface, Sequelize) {
        console.log('⚠️  Rolling back: Dropping all tables and triggers...');
        
        // Get list of all tables
        const [tables] = await queryInterface.sequelize.query(
            "SELECT TABLE_NAME FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME LIKE 'tbl_%'"
        );

        // Get list of all triggers
        const [triggers] = await queryInterface.sequelize.query(
            "SHOW TRIGGERS"
        );

        // Disable foreign key checks
        await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 0');

        try {
            // Drop triggers first
            for (const trigger of triggers) {
                try {
                    await queryInterface.sequelize.query(
                        `DROP TRIGGER IF EXISTS \`${trigger.Trigger}\``
                    );
                    console.log(`🗑️  Dropped trigger: ${trigger.Trigger}`);
                } catch (error) {
                    console.error(`⚠️  Error dropping trigger ${trigger.Trigger}:`, error.message);
                }
            }

            // Drop tables in reverse order (to handle foreign keys)
            // Sort tables to drop dependent tables first
            const sortedTables = tables.map(t => t.TABLE_NAME).reverse();
            
            for (const tableName of sortedTables) {
                try {
                    await queryInterface.sequelize.query(`DROP TABLE IF EXISTS \`${tableName}\``);
                    console.log(`🗑️  Dropped table: ${tableName}`);
                } catch (error) {
                    console.error(`⚠️  Error dropping table ${tableName}:`, error.message);
                }
            }
        } finally {
            // Re-enable foreign key checks
            await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 1');
        }

        console.log('✅ Rollback completed!');
    }
};

