/**
 * Migration: Create All 97 Tables, Trigger, and Stored Procedures
 * Executes the clinic_pro_db.sql file directly and creates stored procedures
 * 
 * Creates:
 * - 97 Tables from clinic_pro_db.sql
 * - 1 Trigger (trg_update_warehouse_stock_on_movement)
 * - 3 Stored Procedures:
 *   - auto_purchase_id
 *   - AutoGeneratevoucherNo_purchase
 *   - AutoGeneratevoucherNo_Sale
 */

'use strict';

const fs = require('fs');
const path = require('path');

module.exports = {
    async up(queryInterface, Sequelize) {
        const sqlFilePath = path.join(__dirname, '../../clinic_pro_db.sql');
        
        // Check if SQL file exists
        if (!fs.existsSync(sqlFilePath)) {
            throw new Error(`SQL file not found: ${sqlFilePath}\nPlease ensure clinic_pro_db.sql exists in the project root.`);
        }

        console.log('📖 Reading SQL file...');
        let sqlContent = fs.readFileSync(sqlFilePath, 'utf8');

        // Remove destructive/drop statements and data sections to keep existing data intact
        sqlContent = sqlContent
            // Strip any DROP statements to avoid wiping existing data
            .replace(/DROP\s+TABLE\s+IF\s+EXISTS[^;]*;/gi, '')
            .replace(/DROP\s+TRIGGER\s+IF\s+EXISTS[^;]*;/gi, '')
            .replace(/DROP\s+PROCEDURE\s+IF\s+EXISTS[^;]*;/gi, '')
            // Remove entire data dumping sections (from "-- Dumping data" to "UNLOCK TABLES")
            .replace(/--\s*Dumping data for table[^]*?UNLOCK TABLES;/gi, '')
            // Remove LOCK TABLES statements
            .replace(/LOCK TABLES[^;]*;/gi, '')
            // Remove INSERT statements
            .replace(/INSERT INTO[^;]*;/gi, '')
            // Remove MySQL version-specific SET statements (keep the structure)
            .replace(/\/\*!40101 SET[^;]*;\*\//gi, '')
            .replace(/\/\*!40000 ALTER TABLE[^;]*;\*\//gi, '');

        const statements = [];
        
        // Extract CREATE TABLE statements using regex
        // Match from CREATE TABLE to the closing ); with ENGINE=InnoDB
        const createTableRegex = /CREATE TABLE[^`]*`([^`]+)`[^;]*?ENGINE=InnoDB[^;]*;/gis;
        let match;
        while ((match = createTableRegex.exec(sqlContent)) !== null) {
            let statement = match[0];
            // Clean up MySQL version comments
            statement = statement
                .replace(/\/\*!40101[^*]*\*\//g, '')
                .replace(/\/\*!50003[^*]*\*\//g, '')
                .replace(/\/\*!50017[^*]*\*\//g, '')
                .trim();
            if (statement) {
                statements.push(statement);
            }
        }

        // Extract CREATE TRIGGER statements (handle DELIMITER blocks)
        // Find the trigger section
        const triggerStart = sqlContent.indexOf('DELIMITER ;;');
        if (triggerStart !== -1) {
            const triggerSection = sqlContent.substring(triggerStart);
            // Extract trigger between DELIMITER ;; and DELIMITER ;
            const triggerMatch = triggerSection.match(/CREATE.*?TRIGGER[^]*?END[^]*?;;/is);
            if (triggerMatch) {
                let triggerStatement = triggerMatch[0]
                    .replace(/DELIMITER\s+;;/gi, '')
                    .replace(/DELIMITER\s+;/gi, '')
                    .replace(/;;/g, ';')
                    .replace(/\/\*!50003[^*]*\*\//g, '')
                    .replace(/\/\*!50017[^*]*\*\//g, '')
                    .trim();
                if (triggerStatement) {
                    statements.push(triggerStatement);
                }
            }
        }

        // Add stored procedures manually (since they're not in clinic_pro_db.sql)
        const procedures = [
            {
                name: 'auto_purchase_id',
                sql: `CREATE PROCEDURE \`auto_purchase_id\`()
BEGIN
    DECLARE v_row_count INT;
    DECLARE v_id INT;
    DECLARE v_temp_new_number INT;
    
    SET v_row_count = (SELECT COUNT(id) FROM tbl_purchase);
    
    IF (v_row_count = 0) THEN
        SET v_id = 1;
    ELSE
        SET v_temp_new_number = (SELECT MAX(id) FROM tbl_purchase);
        SET v_id = v_temp_new_number + 1;
    END IF;
    
    SELECT v_id AS 'id';
END`
            },
            {
                name: 'AutoGeneratevoucherNo_purchase',
                sql: `CREATE PROCEDURE \`AutoGeneratevoucherNo_purchase\`()
BEGIN
    DECLARE v_row_count INT;
    DECLARE v_voucher INT;
    DECLARE v_temp_new_number INT;
    
    SET v_row_count = (SELECT COUNT(id) FROM tbl_purchase);
    
    IF (v_row_count = 0) THEN
        SET v_voucher = 1000;
    ELSE
        SET v_temp_new_number = (SELECT MAX(purchase_voucher_id) FROM tbl_purchase);
        SET v_voucher = v_temp_new_number + 1;
    END IF;
    
    SELECT v_voucher AS 'voucher';
END`
            },
            {
                name: 'AutoGeneratevoucherNo_Sale',
                sql: `CREATE PROCEDURE \`AutoGeneratevoucherNo_Sale\`()
BEGIN
    DECLARE v_row_count INT;
    DECLARE v_voucher INT;
    DECLARE v_temp_new_number INT;
    
    SET v_row_count = (SELECT COUNT(id) FROM tbl_sale);
    
    IF (v_row_count = 0) THEN
        SET v_voucher = 1000;
    ELSE
        SET v_temp_new_number = (SELECT MAX(VNo) FROM tbl_sale);
        SET v_voucher = v_temp_new_number + 1;
    END IF;
    
    SELECT v_voucher AS 'voucher';
END`
            }
        ];

        // Add procedure creation statements (will be executed separately with DELIMITER handling)
        // Store procedures separately for special handling

        console.log(`📊 Found ${statements.length} SQL statements to execute`);

        // Disable foreign key checks temporarily
        await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 0');
        await queryInterface.sequelize.query('SET UNIQUE_CHECKS = 0');

        let tablesCreated = 0;
        let triggersCreated = 0;
        let proceduresCreated = 0;
        let errors = 0;

        try {
            // Execute each statement
            for (let i = 0; i < statements.length; i++) {
                const statement = statements[i];
                
                // Skip empty or comment-only statements
                if (!statement || statement.match(/^--/) || statement.match(/^\/\*/)) {
                    continue;
                }

                try {
                    // Clean up MySQL version-specific comments
                    let cleanStatement = statement
                        .replace(/\/\*!50003[^*]*\*\/|\/\*!50017[^*]*\*\/|\/\*!40101[^*]*\*\//g, '')
                        .trim();

                    // Make table creation idempotent
                    if (cleanStatement.match(/^CREATE TABLE\s+`/i)) {
                        cleanStatement = cleanStatement.replace(/^CREATE TABLE\s+`/i, 'CREATE TABLE IF NOT EXISTS `');
                    }

                    if (!cleanStatement) continue;

                    // Execute statement
                    await queryInterface.sequelize.query(cleanStatement);

                    // Count what was created
                    if (cleanStatement.match(/CREATE TABLE/i)) {
                        const tableMatch = cleanStatement.match(/CREATE TABLE[^`]*`([^`]+)`/i);
                        if (tableMatch) {
                            tablesCreated++;
                            if (tablesCreated % 10 === 0) {
                                console.log(`   ✅ Created ${tablesCreated} tables...`);
                            }
                        }
                    } else if (cleanStatement.match(/CREATE.*TRIGGER/i)) {
                        const triggerMatch = cleanStatement.match(/CREATE.*TRIGGER[^`]*`([^`]+)`/i);
                        if (triggerMatch) {
                            triggersCreated++;
                            console.log(`   ✅ Created trigger: ${triggerMatch[1]}`);
                        }
                    } else if (cleanStatement.match(/CREATE.*PROCEDURE/i)) {
                        const procedureMatch = cleanStatement.match(/CREATE.*PROCEDURE[^`]*`([^`]+)`/i);
                        if (procedureMatch) {
                            proceduresCreated++;
                            console.log(`   ✅ Created procedure: ${procedureMatch[1]}`);
                        }
                    } else if (cleanStatement.match(/DROP PROCEDURE/i)) {
                        // Skip counting DROP statements
                    }
                } catch (error) {
                    // Check if it's a "table already exists" error
                    if (error.message && error.message.includes('already exists')) {
                        // Skip silently for idempotency
                        continue;
                    }
                    
                    // Log other errors but continue
                    errors++;
                    if (errors <= 10) { // Only log first 10 errors
                        console.error(`   ⚠️  Error in statement ${i + 1}: ${error.message.substring(0, 100)}`);
                    }
                }
            }
        } finally {
            // Re-enable foreign key checks
            await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 1');
            await queryInterface.sequelize.query('SET UNIQUE_CHECKS = 1');
        }

        // Create stored procedures
        console.log('\n📦 Creating stored procedures...');
        for (const proc of procedures) {
            try {
                // Drop procedure if exists
                await queryInterface.sequelize.query(`DROP PROCEDURE IF EXISTS \`${proc.name}\``);
                
                // Create procedure - Sequelize/mysql2 handles procedures without DELIMITER
                // The SQL is already correct (no DELIMITER needed in programmatic execution)
                await queryInterface.sequelize.query(proc.sql);
                
                proceduresCreated++;
                console.log(`   ✅ Created procedure: ${proc.name}`);
            } catch (error) {
                if (error.message && (error.message.includes('already exists') || error.message.includes('Duplicate'))) {
                    console.log(`   ⏭️  Procedure ${proc.name} already exists, skipping...`);
                } else {
                    errors++;
                    console.error(`   ⚠️  Error creating procedure ${proc.name}:`, error.message.substring(0, 150));
                }
            }
        }

        console.log(`\n✅ Migration completed!`);
        console.log(`   • Tables created: ${tablesCreated}`);
        console.log(`   • Triggers created: ${triggersCreated}`);
        console.log(`   • Procedures created: ${proceduresCreated}`);
        if (errors > 0) {
            console.log(`   • Errors encountered: ${errors}`);
        }
    },

    async down(queryInterface, Sequelize) {
        console.log('⚠️  Rolling back: Dropping all tables and triggers...');
        
        // Get list of all tables
        const [tables] = await queryInterface.sequelize.query(
            "SELECT TABLE_NAME FROM information_schema.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME LIKE 'tbl_%'"
        );

        // Get list of all triggers
        const [triggers] = await queryInterface.sequelize.query("SHOW TRIGGERS");

        // Get list of all procedures
        const [procedures] = await queryInterface.sequelize.query(
            "SHOW PROCEDURE STATUS WHERE Db = DATABASE()"
        );

        // Disable foreign key checks
        await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 0');

        try {
            // Drop procedures first
            for (const procedure of procedures) {
                try {
                    await queryInterface.sequelize.query(`DROP PROCEDURE IF EXISTS \`${procedure.Name}\``);
                    console.log(`   🗑️  Dropped procedure: ${procedure.Name}`);
                } catch (error) {
                    console.error(`   ⚠️  Error dropping procedure ${procedure.Name}:`, error.message);
                }
            }

            // Drop triggers
            for (const trigger of triggers) {
                try {
                    await queryInterface.sequelize.query(`DROP TRIGGER IF EXISTS \`${trigger.Trigger}\``);
                    console.log(`   🗑️  Dropped trigger: ${trigger.Trigger}`);
                } catch (error) {
                    console.error(`   ⚠️  Error dropping trigger ${trigger.Trigger}:`, error.message);
                }
            }

            // Drop tables
            for (const table of tables) {
                try {
                    await queryInterface.sequelize.query(`DROP TABLE IF EXISTS \`${table.TABLE_NAME}\``);
                } catch (error) {
                    console.error(`   ⚠️  Error dropping table ${table.TABLE_NAME}:`, error.message);
                }
            }
        } finally {
            // Re-enable foreign key checks
            await queryInterface.sequelize.query('SET FOREIGN_KEY_CHECKS = 1');
        }

        console.log(`✅ Rollback completed! Dropped ${tables.length} tables, ${triggers.length} triggers, and ${procedures.length} procedures.`);
    }
};

