# Migration Guide - Create All 97 Tables and Trigger

This guide explains how to use the migration files to create all 97 database tables and the warehouse stock movement trigger from the `clinic_pro_db.sql` file.

## Migration Files

### 1. `002-create-all-tables-simple.js` (Recommended)
- **Purpose:** Creates all 97 tables and 1 trigger from the SQL dump file
- **Method:** Reads `clinic_pro_db.sql` and executes CREATE TABLE and CREATE TRIGGER statements
- **Idempotent:** Skips tables/triggers that already exist
- **Usage:** Can be run via Sequelize CLI or standalone script

### 2. `002-create-all-tables-and-trigger.js` (Alternative)
- **Purpose:** Same as above but with more complex parsing
- **Method:** More detailed statement extraction and validation
- **Use Case:** If the simple version has issues with SQL parsing

## Prerequisites

1. **SQL File:** Ensure `clinic_pro_db.sql` exists in the project root directory
2. **Database:** MySQL/MariaDB database must be created and configured
3. **Environment:** `.env` file must have correct database credentials

## Running the Migration

### Option 1: Using Sequelize CLI (Recommended)

```bash
# From backend directory
npx sequelize-cli db:migrate
```

### Option 2: Using Standalone Script

```bash
# From project root
node backend/migrations/run-create-all-tables.js
```

### Option 3: Manual Execution

```javascript
const createAllTables = require('./backend/migrations/002-create-all-tables-simple');
const { Sequelize } = require('sequelize');

// Setup Sequelize connection
const sequelize = new Sequelize(/* your config */);
const queryInterface = sequelize.getQueryInterface();

// Run migration
await createAllTables.up(queryInterface, Sequelize);
```

## What the Migration Does

1. **Reads SQL File:** Loads `clinic_pro_db.sql` from project root
2. **Extracts Structure:** Filters out data insertion statements, keeps only:
   - CREATE TABLE statements
   - CREATE TRIGGER statements
   - DROP TABLE statements (for cleanup)
3. **Executes Statements:** Runs each CREATE statement in order
4. **Handles Errors:** Continues even if some tables already exist (idempotent)
5. **Creates Trigger:** Creates the `trg_update_warehouse_stock_on_movement` trigger

## Expected Output

```
📖 Reading SQL file...
📊 Found 98 SQL statements to execute
   ✅ Created 10 tables...
   ✅ Created 20 tables...
   ...
   ✅ Created trigger: trg_update_warehouse_stock_on_movement

✅ Migration completed!
   • Tables created: 97
   • Triggers created: 1
```

## Rollback

To rollback (drop all tables and triggers):

```bash
# Using Sequelize CLI
npx sequelize-cli db:migrate:undo

# Or manually
const createAllTables = require('./backend/migrations/002-create-all-tables-simple');
await createAllTables.down(queryInterface, Sequelize);
```

## Troubleshooting

### Error: SQL file not found
- **Solution:** Ensure `clinic_pro_db.sql` exists in the project root directory
- **Check:** `ls clinic_pro_db.sql` or `dir clinic_pro_db.sql`

### Error: Table already exists
- **Status:** This is normal - migration is idempotent
- **Action:** Migration will skip existing tables and continue

### Error: Foreign key constraint fails
- **Solution:** Migration temporarily disables foreign key checks
- **If persists:** Check table creation order in SQL file

### Error: Trigger creation fails
- **Check:** Ensure `tbl_stock_movement` table exists first
- **Check:** Ensure `tbl_warehouse_stock` table exists

## Migration Details

### Tables Created (97 total)
- All tables prefixed with `tbl_`
- Includes: patients, doctors, inventory, sales, EMR, laboratory, etc.
- Full list available in `clinic_pro_db.sql`

### Trigger Created (1 total)
- **Name:** `trg_update_warehouse_stock_on_movement`
- **Type:** AFTER INSERT trigger
- **Table:** `tbl_stock_movement`
- **Purpose:** Automatically updates `tbl_warehouse_stock` when stock movements occur

## Notes

- Migration is **idempotent** - safe to run multiple times
- Only creates structure, **does not insert data**
- Preserves existing data if tables already exist
- Uses transaction-like behavior (disables FK checks temporarily)

## Next Steps

After running the migration:

1. **Verify Tables:** Check that all 97 tables exist
   ```sql
   SELECT COUNT(*) FROM information_schema.TABLES 
   WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME LIKE 'tbl_%';
   ```

2. **Verify Trigger:** Check that trigger exists
   ```sql
   SHOW TRIGGERS WHERE Trigger = 'trg_update_warehouse_stock_on_movement';
   ```

3. **Seed Data:** Run seed scripts if needed
   ```bash
   node backend/utils/seed.js
   ```

4. **Test Application:** Start the server and verify functionality

## Support

For issues or questions:
- Check migration logs for specific errors
- Review `clinic_pro_db.sql` file structure
- Verify database connection settings in `.env`

