# Database Migrations

This directory contains database migration scripts for the Clinic Pro system.

## Available Migrations

### 1. Add Cleaning Status to Beds
**File:** `add-cleaning-status-to-beds.js`

**Purpose:** Adds the 'Cleaning' status to the bed status ENUM to support automatic bed management after patient discharge.

**Features:**
- Automatic bed cleaning after patient discharge
- Beds automatically become Available after 2 minutes of Cleaning
- Enhanced bed status management with better warnings

**When to Run:**
- When upgrading from an older version that doesn't have the 'Cleaning' status
- Before using the new inpatient management features

**How to Run:**
```bash
cd backend/migrations
node add-cleaning-status-to-beds.js
```

## Migration Best Practices

1. **Backup First:** Always backup your database before running migrations
2. **Test Environment:** Run migrations in a test environment first
3. **Check Prerequisites:** Ensure all required tables exist before running migrations
4. **Verify Results:** Check the console output to confirm successful migration

## Full Setup (New Installation)

If you're setting up a new database from scratch, run these in order:

```bash
# 1. Create all tables
cd backend/utils
node create-all-missing-tables.js

# 2. Seed hospital data
cd ..
node seed-hospital-fixed.js

# 3. Run any additional migrations (if needed)
cd migrations
node add-cleaning-status-to-beds.js
```

## Existing Database Update

If you're updating an existing database:

```bash
# 1. Backup your database first!

# 2. Run the specific migration
cd backend/migrations
node add-cleaning-status-to-beds.js
```

## Rollback

To rollback the Cleaning status migration (if needed):

```sql
ALTER TABLE tbl_beds 
MODIFY COLUMN status ENUM('Available', 'Occupied', 'Maintenance', 'Reserved') 
DEFAULT 'Available';
```

**Note:** Make sure no beds are currently in 'Cleaning' status before rolling back!

## Verification

After running migrations, verify the changes:

```sql
-- Check bed status ENUM values
SELECT COLUMN_TYPE 
FROM INFORMATION_SCHEMA.COLUMNS 
WHERE TABLE_NAME = 'tbl_beds' 
AND COLUMN_NAME = 'status';

-- Check current bed statuses
SELECT status, COUNT(*) as count 
FROM tbl_beds 
GROUP BY status;
```

## New Features Documentation

### Automatic Bed Cleaning Flow

1. **Patient Discharge:**
   - When a patient is discharged via `DELETE /api/inpatients/:id`
   - Bed status automatically changes to 'Cleaning'
   - Patient ID is cleared from the bed

2. **Auto-Cleaning Timer:**
   - After 2 minutes, bed status automatically changes to 'Available'
   - No manual intervention needed
   - Cleaning can be completed manually via `PATCH /api/inpatients/beds/:bedId/clean`

3. **Bed Assignment Validation:**
   - Enhanced error messages for occupied beds
   - Shows current patient details if bed is occupied
   - Prevents duplicate patient admissions by phone number
   - Different messages for different bed statuses (Cleaning, Maintenance, etc.)

### API Endpoints

**Get Available Beds:**
```
GET /api/inpatients/available-beds?department_id=1
```

**Clean Bed Manually:**
```
PATCH /api/inpatients/beds/:bedId/clean
```

**Discharge Patient (auto-cleans bed):**
```
DELETE /api/inpatients/:id
Body: {
  "discharge_date": "2024-01-15",
  "discharge_notes": "Patient recovered well"
}
```

## Troubleshooting

### Error: "Unknown column 'room_number'"
**Solution:** The column is named `room_no`, not `room_number` in `tbl_inpatient`

### Error: "Unknown enum value: 'Cleaning'"
**Solution:** Run the `add-cleaning-status-to-beds.js` migration

### Beds Not Auto-Cleaning
**Solution:** 
- Check server logs for setTimeout errors
- Ensure the server stays running for at least 2 minutes after discharge
- For production, consider using a job queue (Bull, Bee-Queue) instead of setTimeout

## Support

For issues or questions, refer to:
- Main project documentation: `/backend/README.md`
- Database schema: `/backend/docs/database-schema.md`

