/**
 * Migration: Add 'Cleaning' status to tbl_beds
 * This migration adds the 'Cleaning' status to the bed status ENUM
 * for automatic bed management after patient discharge
 */

require('dotenv').config();
const { pool } = require('../config/database');

const addCleaningStatusToBeds = async () => {
    try {
        console.log('🔄 Migrating tbl_beds to add Cleaning status...');
        
        // Check current ENUM values
        const [columns] = await pool.execute(`
            SELECT COLUMN_TYPE 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_beds' 
            AND COLUMN_NAME = 'status'
        `);

        if (columns.length === 0) {
            console.log('❌ tbl_beds table or status column not found!');
            return;
        }

        const currentEnum = columns[0].COLUMN_TYPE;
        console.log(`📊 Current status ENUM: ${currentEnum}`);

        // Check if 'Cleaning' already exists
        if (currentEnum.includes('Cleaning')) {
            console.log('✅ Cleaning status already exists in the ENUM. No migration needed.');
            return;
        }

        // Add 'Cleaning' to the ENUM
        console.log('Adding Cleaning status to tbl_beds...');
        await pool.execute(`
            ALTER TABLE tbl_beds 
            MODIFY COLUMN status ENUM('Available', 'Occupied', 'Cleaning', 'Maintenance', 'Reserved') 
            DEFAULT 'Available'
        `);

        console.log('✅ Successfully added Cleaning status to tbl_beds');
        
        // Verify the change
        const [newColumns] = await pool.execute(`
            SELECT COLUMN_TYPE 
            FROM INFORMATION_SCHEMA.COLUMNS 
            WHERE TABLE_SCHEMA = DATABASE() 
            AND TABLE_NAME = 'tbl_beds' 
            AND COLUMN_NAME = 'status'
        `);

        console.log(`📊 New status ENUM: ${newColumns[0].COLUMN_TYPE}`);
        
        console.log('\n🎉 Migration completed successfully!');
        console.log('\n✨ NEW FEATURES ENABLED:');
        console.log('   • Automatic bed cleaning after patient discharge');
        console.log('   • Beds automatically become Available after 2 minutes of Cleaning');
        console.log('   • Enhanced bed status management');
        
    } catch (error) {
        console.error('❌ Migration failed:', error.message);
        throw error;
    } finally {
        await pool.end();
    }
};

// Run if called directly
if (require.main === module) {
    addCleaningStatusToBeds().catch(console.error);
}

module.exports = addCleaningStatusToBeds;

