/**
 * Quick script to create missing tables
 * Run this if migration isn't complete: node backend/migrations/create-missing-tables.js
 */

require('dotenv').config({ path: require('path').join(__dirname, '../.env') });
const { query } = require('../config/database');

async function createMissingTables() {
    try {
        console.log('🔍 Checking for missing tables...\n');

        // Check and create tbl_laboratory_item
        try {
            await query('SELECT 1 FROM tbl_laboratory_item LIMIT 1');
            console.log('✅ tbl_laboratory_item exists');
        } catch (error) {
            if (error.code === 'ER_NO_SUCH_TABLE') {
                console.log('⚠️  tbl_laboratory_item not found, creating...');
                await query(`
                    CREATE TABLE IF NOT EXISTS tbl_laboratory_item (
                        id INT NOT NULL AUTO_INCREMENT,
                        barcode VARCHAR(50) NULL,
                        item_name VARCHAR(50) NOT NULL,
                        cat_id INT NULL,
                        buy_price DECIMAL(10, 2) DEFAULT 0.00,
                        sale_price DECIMAL(10, 2) DEFAULT 0.00,
                        remark TEXT NULL,
                        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                        PRIMARY KEY (id),
                        INDEX idx_barcode (barcode),
                        INDEX idx_item_name (item_name),
                        INDEX idx_cat_id (cat_id),
                        FOREIGN KEY (cat_id) REFERENCES tbl_service_catalog(id) ON DELETE SET NULL
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci COMMENT='Laboratory test items that can be sold in POS and Inpatient Sales'
                `);
                console.log('✅ tbl_laboratory_item created successfully');
            } else {
                throw error;
            }
        }

        // Check and create tbl_notification_read
        try {
            await query('SELECT 1 FROM tbl_notification_read LIMIT 1');
            console.log('✅ tbl_notification_read exists');
        } catch (error) {
            if (error.code === 'ER_NO_SUCH_TABLE') {
                console.log('⚠️  tbl_notification_read not found, creating...');
                await query(`
                    CREATE TABLE IF NOT EXISTS tbl_notification_read (
                        id INT NOT NULL AUTO_INCREMENT,
                        notification_id VARCHAR(255) NOT NULL,
                        user_id INT NULL,
                        read_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                        PRIMARY KEY (id),
                        UNIQUE KEY unique_notification_user (notification_id, user_id),
                        INDEX idx_user_id (user_id),
                        INDEX idx_notification_id (notification_id)
                    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
                `);
                // Add foreign key if tbl_user exists
                try {
                    await query(`SELECT 1 FROM tbl_user LIMIT 1`);
                    await query(`
                        ALTER TABLE tbl_notification_read 
                        ADD CONSTRAINT fk_notification_read_user 
                        FOREIGN KEY (user_id) REFERENCES tbl_user(id) ON DELETE CASCADE
                    `);
                } catch (fkError) {
                    // Foreign key might already exist or tbl_user doesn't exist, skip
                    console.log('   ℹ️  Foreign key will be added when tbl_user exists');
                }
                console.log('✅ tbl_notification_read created successfully');
            } else {
                throw error;
            }
        }

        console.log('\n✨ Missing tables check completed!');
        process.exit(0);
    } catch (error) {
        console.error('\n❌ Error creating missing tables:', error.message);
        process.exit(1);
    }
}

// Run if called directly
if (require.main === module) {
    createMissingTables();
}

module.exports = createMissingTables;

